/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ExternalHostProviderBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlExternalHostProviderReader;
import org.ovirt.engine.sdk4.internal.xml.XmlExternalHostProviderWriter;
import org.ovirt.engine.sdk4.services.ExternalComputeResourcesService;
import org.ovirt.engine.sdk4.services.ExternalDiscoveredHostsService;
import org.ovirt.engine.sdk4.services.ExternalHostGroupsService;
import org.ovirt.engine.sdk4.services.ExternalHostProviderService;
import org.ovirt.engine.sdk4.services.ExternalHostsService;
import org.ovirt.engine.sdk4.services.ExternalProviderCertificatesService;
import org.ovirt.engine.sdk4.types.ExternalHostProvider;

public class ExternalHostProviderServiceImpl extends ServiceImpl implements ExternalHostProviderService {
    
    public ExternalHostProviderServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlExternalHostProviderReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new GetResponseImpl(null);
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private ExternalHostProvider provider;
        
        public GetResponseImpl(ExternalHostProvider provider) {
            this.provider = provider;
        }
        
        public ExternalHostProvider provider() {
            return provider;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    private class RemoveRequestImpl implements RemoveRequest {
        private Boolean async;
        public RemoveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public RemoveResponse send() {
            HttpDelete request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                if (async != null) {
                    uriBuilder.addParameter("async", XmlWriter.renderBoolean(async));
                }
                request = new HttpDelete(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                EntityUtils.consumeQuietly(response.getEntity());
                return new RemoveResponseImpl();
            }
            else {
                checkFault(response);
                return new RemoveResponseImpl();
            }
        }
        
    }
    
    private class RemoveResponseImpl implements RemoveResponse {
    }
    
    public RemoveRequest remove() {
        return new RemoveRequestImpl();
    }
    
    private class UpdateRequestImpl implements UpdateRequest {
        private Boolean async;
        public UpdateRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private ExternalHostProvider provider;
        public UpdateRequest provider(ExternalHostProvider provider) {
            this.provider = provider;
            return this;
        }
        
        public UpdateRequest provider(ExternalHostProviderBuilder provider) {
            this.provider = provider.build();
            return this;
        }
        
        public UpdateResponse send() {
            HttpPut request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                if (async != null) {
                    uriBuilder.addParameter("async", XmlWriter.renderBoolean(async));
                }
                request = new HttpPut(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                XmlExternalHostProviderWriter.writeOne(provider, xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to parse response", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new UpdateResponseImpl(XmlExternalHostProviderReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new UpdateResponseImpl(null);
            }
        }
        
    }
    
    private class UpdateResponseImpl implements UpdateResponse {
        private ExternalHostProvider provider;
        
        public UpdateResponseImpl(ExternalHostProvider provider) {
            this.provider = provider;
        }
        
        public ExternalHostProvider provider() {
            return provider;
        }
        
    }
    
    public UpdateRequest update() {
        return new UpdateRequestImpl();
    }
    
    public ExternalProviderCertificatesService certificatesService() {
        return new ExternalProviderCertificatesServiceImpl(getConnection(), getPath() + "/certificates");
    }
    
    public ExternalComputeResourcesService computeResourcesService() {
        return new ExternalComputeResourcesServiceImpl(getConnection(), getPath() + "/computeresources");
    }
    
    public ExternalDiscoveredHostsService discoveredHostsService() {
        return new ExternalDiscoveredHostsServiceImpl(getConnection(), getPath() + "/discoveredhosts");
    }
    
    public ExternalHostGroupsService hostGroupsService() {
        return new ExternalHostGroupsServiceImpl(getConnection(), getPath() + "/hostgroups");
    }
    
    public ExternalHostsService hostsService() {
        return new ExternalHostsServiceImpl(getConnection(), getPath() + "/hosts");
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        if (path.equals("certificates")) {
            return certificatesService();
        }
        if (path.startsWith("certificates/")) {
            return certificatesService().service(path.substring(13));
        }
        if (path.equals("computeresources")) {
            return computeResourcesService();
        }
        if (path.startsWith("computeresources/")) {
            return computeResourcesService().service(path.substring(17));
        }
        if (path.equals("discoveredhosts")) {
            return discoveredHostsService();
        }
        if (path.startsWith("discoveredhosts/")) {
            return discoveredHostsService().service(path.substring(16));
        }
        if (path.equals("hostgroups")) {
            return hostGroupsService();
        }
        if (path.startsWith("hostgroups/")) {
            return hostGroupsService().service(path.substring(11));
        }
        if (path.equals("hosts")) {
            return hostsService();
        }
        if (path.startsWith("hosts/")) {
            return hostsService().service(path.substring(6));
        }
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "ExternalHostProviderService:" + getPath();
    }
    
}

