/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ImageTransferBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlImageTransferReader;
import org.ovirt.engine.sdk4.internal.xml.XmlImageTransferWriter;
import org.ovirt.engine.sdk4.services.ImageTransferService;
import org.ovirt.engine.sdk4.services.ImageTransfersService;
import org.ovirt.engine.sdk4.types.ImageTransfer;

public class ImageTransfersServiceImpl extends ServiceImpl implements ImageTransfersService {
    
    public ImageTransfersServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class AddRequestImpl implements AddRequest {
        private ImageTransfer imageTransfer;
        public AddRequest imageTransfer(ImageTransfer imageTransfer) {
            this.imageTransfer = imageTransfer;
            return this;
        }
        
        public AddRequest imageTransfer(ImageTransferBuilder imageTransfer) {
            this.imageTransfer = imageTransfer.build();
            return this;
        }
        
        public AddResponse send() {
            HttpPost request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpPost(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                XmlImageTransferWriter.writeOne(imageTransfer, xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to parse response", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
              || response.getStatusLine().getStatusCode() == 201
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new AddResponseImpl(XmlImageTransferReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new AddResponseImpl(null);
            }
        }
        
    }
    
    private class AddResponseImpl implements AddResponse {
        private ImageTransfer imageTransfer;
        
        public AddResponseImpl(ImageTransfer imageTransfer) {
            this.imageTransfer = imageTransfer;
        }
        
        public ImageTransfer imageTransfer() {
            return imageTransfer;
        }
        
    }
    
    public AddRequest add() {
        return new AddRequestImpl();
    }
    
    private class ListRequestImpl implements ListRequest {
        public ListResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new ListResponseImpl(XmlImageTransferReader.readMany(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new ListResponseImpl(null);
            }
        }
        
    }
    
    private class ListResponseImpl implements ListResponse {
        private List<ImageTransfer> imageTransfer;
        
        public ListResponseImpl(List<ImageTransfer> imageTransfer) {
            this.imageTransfer = imageTransfer;
        }
        
        public List<ImageTransfer> imageTransfer() {
            return imageTransfer;
        }
        
    }
    
    public ListRequest list() {
        return new ListRequestImpl();
    }
    
    public ImageTransferService imageTransferService(String id) {
        return new ImageTransferServiceImpl(getConnection(), getPath() + "/" + id);
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        int index = path.indexOf('/');
        if (index == -1) {
            return imageTransferService(path);
        }
        return imageTransferService(path.substring(0, index)).service(path.substring(index + 1));
    }
    
    public String toString() {
        return "ImageTransfersService:" + getPath();
    }
    
}

