/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ActionBuilder;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.builders.VmBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlActionWriter;
import org.ovirt.engine.sdk4.internal.xml.XmlVmReader;
import org.ovirt.engine.sdk4.services.StorageDomainContentDisksService;
import org.ovirt.engine.sdk4.services.StorageDomainVmService;
import org.ovirt.engine.sdk4.types.Action;
import org.ovirt.engine.sdk4.types.Cluster;
import org.ovirt.engine.sdk4.types.StorageDomain;
import org.ovirt.engine.sdk4.types.Vm;

public class StorageDomainVmServiceImpl extends ServiceImpl implements StorageDomainVmService {
    
    public StorageDomainVmServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlVmReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new GetResponseImpl(null);
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private Vm vm;
        
        public GetResponseImpl(Vm vm) {
            this.vm = vm;
        }
        
        public Vm vm() {
            return vm;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    private class ImportRequestImpl implements ImportRequest {
        private Boolean async;
        public ImportRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Boolean clone_;
        public ImportRequest clone_(Boolean clone_) {
            this.clone_ = clone_;
            return this;
        }
        
        private Cluster cluster;
        public ImportRequest cluster(Cluster cluster) {
            this.cluster = cluster;
            return this;
        }
        
        public ImportRequest cluster(ClusterBuilder cluster) {
            this.cluster = cluster.build();
            return this;
        }
        
        private Boolean collapseSnapshots;
        public ImportRequest collapseSnapshots(Boolean collapseSnapshots) {
            this.collapseSnapshots = collapseSnapshots;
            return this;
        }
        
        private StorageDomain storageDomain;
        public ImportRequest storageDomain(StorageDomain storageDomain) {
            this.storageDomain = storageDomain;
            return this;
        }
        
        public ImportRequest storageDomain(StorageDomainBuilder storageDomain) {
            this.storageDomain = storageDomain.build();
            return this;
        }
        
        private Vm vm;
        public ImportRequest vm(Vm vm) {
            this.vm = vm;
            return this;
        }
        
        public ImportRequest vm(VmBuilder vm) {
            this.vm = vm.build();
            return this;
        }
        
        public ImportResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/import");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                ActionBuilder action = new ActionBuilder();
                action.async(async);
                action.clone_(clone_);
                action.cluster(cluster);
                action.collapseSnapshots(collapseSnapshots);
                action.storageDomain(storageDomain);
                action.vm(vm);
                XmlActionWriter.writeOne(action.build(), xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new ImportResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class ImportResponseImpl implements ImportResponse {
    }
    
    public ImportRequest import_() {
        return new ImportRequestImpl();
    }
    
    private class RegisterRequestImpl implements RegisterRequest {
        private Boolean async;
        public RegisterRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Boolean clone_;
        public RegisterRequest clone_(Boolean clone_) {
            this.clone_ = clone_;
            return this;
        }
        
        private Cluster cluster;
        public RegisterRequest cluster(Cluster cluster) {
            this.cluster = cluster;
            return this;
        }
        
        public RegisterRequest cluster(ClusterBuilder cluster) {
            this.cluster = cluster.build();
            return this;
        }
        
        private Vm vm;
        public RegisterRequest vm(Vm vm) {
            this.vm = vm;
            return this;
        }
        
        public RegisterRequest vm(VmBuilder vm) {
            this.vm = vm.build();
            return this;
        }
        
        public RegisterResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/register");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                ActionBuilder action = new ActionBuilder();
                action.async(async);
                action.clone_(clone_);
                action.cluster(cluster);
                action.vm(vm);
                XmlActionWriter.writeOne(action.build(), xmlWriter);
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new RegisterResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class RegisterResponseImpl implements RegisterResponse {
    }
    
    public RegisterRequest register() {
        return new RegisterRequestImpl();
    }
    
    private class RemoveRequestImpl implements RemoveRequest {
        private Boolean async;
        public RemoveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public RemoveResponse send() {
            HttpDelete request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                if (async != null) {
                    uriBuilder.addParameter("async", XmlWriter.renderBoolean(async));
                }
                request = new HttpDelete(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                EntityUtils.consumeQuietly(response.getEntity());
                return new RemoveResponseImpl();
            }
            else {
                checkFault(response);
                return new RemoveResponseImpl();
            }
        }
        
    }
    
    private class RemoveResponseImpl implements RemoveResponse {
    }
    
    public RemoveRequest remove() {
        return new RemoveRequestImpl();
    }
    
    public StorageDomainContentDisksService disksService() {
        return new StorageDomainContentDisksServiceImpl(getConnection(), getPath() + "/disks");
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        if (path.equals("disks")) {
            return disksService();
        }
        if (path.startsWith("disks/")) {
            return disksService().service(path.substring(6));
        }
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "StorageDomainVmService:" + getPath();
    }
    
}

