/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.DiskBuilder;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.types.Disk;
import org.ovirt.engine.sdk4.types.StorageDomain;

public interface DiskService extends Service {
    /**
     * This operation copies a disk to the specified storage domain.
     * 
     * For example, copy of a disk can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/copy
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <disk>
     *     <name>mydisk</name>
     *   </disk>
     * </action>
     * ----
     */
    public interface CopyRequest extends Request<CopyRequest, CopyResponse> {
        /**
         * Indicates if the copy should be performed asynchronously.
         */
        CopyRequest async(Boolean async);
        CopyRequest disk(Disk disk);
        CopyRequest disk(DiskBuilder disk);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        CopyRequest filter(Boolean filter);
        CopyRequest storageDomain(StorageDomain storageDomain);
        CopyRequest storageDomain(StorageDomainBuilder storageDomain);
    }
    
    /**
     * This operation copies a disk to the specified storage domain.
     * 
     * For example, copy of a disk can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/copy
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <disk>
     *     <name>mydisk</name>
     *   </disk>
     * </action>
     * ----
     */
    public interface CopyResponse extends Response {
    }
    
    /**
     * This operation copies a disk to the specified storage domain.
     * 
     * For example, copy of a disk can be facilitated using the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/copy
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     *   <disk>
     *     <name>mydisk</name>
     *   </disk>
     * </action>
     * ----
     */
    CopyRequest copy();
    
    public interface ExportRequest extends Request<ExportRequest, ExportResponse> {
        /**
         * Indicates if the export should be performed asynchronously.
         */
        ExportRequest async(Boolean async);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        ExportRequest filter(Boolean filter);
        ExportRequest storageDomain(StorageDomain storageDomain);
        ExportRequest storageDomain(StorageDomainBuilder storageDomain);
    }
    
    public interface ExportResponse extends Response {
    }
    
    ExportRequest export();
    
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    public interface GetResponse extends Response {
        Disk disk();
    }
    
    GetRequest get();
    
    /**
     * Moves a disk to another storage domain.
     * 
     * For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
     * request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/move
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     * </action>
     * ----
     */
    public interface MoveRequest extends Request<MoveRequest, MoveResponse> {
        /**
         * Indicates if the move should be performed asynchronously.
         */
        MoveRequest async(Boolean async);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        MoveRequest filter(Boolean filter);
        MoveRequest storageDomain(StorageDomain storageDomain);
        MoveRequest storageDomain(StorageDomainBuilder storageDomain);
    }
    
    /**
     * Moves a disk to another storage domain.
     * 
     * For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
     * request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/move
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     * </action>
     * ----
     */
    public interface MoveResponse extends Response {
    }
    
    /**
     * Moves a disk to another storage domain.
     * 
     * For example, to move the disk with identifier `123` to a storage domain with identifier `456` send the following
     * request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/disks/123/move
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <action>
     *   <storage_domain id="456"/>
     * </action>
     * ----
     */
    MoveRequest move();
    
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    public interface RemoveResponse extends Response {
    }
    
    RemoveRequest remove();
    
    AssignedPermissionsService permissionsService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

