/* Simple test program to show ICMP ping packets; must be run as root.
 * For fun, we only answer odd-length pings (eg. ping -s 1 from
 * another machine.) */
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>

#include "fwinterface.h"

#define NOBODY_UID 65534
#define NOBODY_GID 65534

void *ping_noise = NULL;
off_t ping_noise_size;
int audio_handle;
static int packets_done = 0;

void print_ip(__u32 a)
{
    printf("%d.%d.%d.%d",
	   (ntohl(a)>>24)&0xFF,
	   (ntohl(a)>>16)&0xFF,
	   (ntohl(a)>>8)&0xFF,
	   (ntohl(a))&0xFF);
}

static const struct iphdr *
readpkt(__u32 rulenum __attribute__((unused)), 
	const char *interface,
	const ip_chainlabel name __attribute__((unused)),
	__u16 packetsize,
	struct iphdr *hdr)
{
    packets_done++;

    printf("Received packet of length %hu from `", ntohs(hdr->tot_len));
    print_ip(hdr->saddr);
    printf("' for `");
    print_ip(hdr->daddr);
    printf("' through `%s'. ", interface);

    if (ping_noise) {
	    write(audio_handle, ping_noise, ping_noise_size);
#if 1
	    /* 2.1.125: doesn't play straight away.  Force it. */
	    close(audio_handle);
	    audio_handle = open("/dev/audio", O_WRONLY);
#endif
    }

    /* Reply to packets with odd number of bytes */
    if (ntohs(hdr->tot_len) % 2) {
	printf("Replying...\n");
	return hdr;
    }
    else {
	printf("Not replying...\n");
	return NULL;
    }
}

void cleanup_and_exit(int signal __attribute((unused)))
{
    fw_close_listen(0);
    fprintf(stderr, "Packets done = %u.\n", packets_done);
    exit(0);
}

int main(int argc, const char *argv[])
{
    struct ip_fw fw 
	= { { 0 }, { 0 }, { 0 }, { 0 }, /* SRC & DST = any */
	    0,				/* Filled in by fw_reg... */
	    IPPROTO_ICMP,		/* Only ICMP packets */
	    0,				/* Nothing special */
	    IP_FW_INV_VIA,		/* Invert interface */
	    { 8, 8 },                   /* TYPE = 8 (ping) */
	    { 0, 0xFFFF },		/* CODE = any */
	    0,				/* redirpt = N/A */
	    65535,                      /* outputsize = all */
	    "tap0",			/* interface != tap0 */
	    0xFF, 0x00			/* TOS = unchanged */
	};
    int ret;

    /* Cleanup on receipt of ^C */
    signal(SIGINT, &cleanup_and_exit);

    /* Open /dev/audio and read ping noise */
    if (!argv[1] || strcmp(argv[1], "-noaudio") != 0) {
	if ((audio_handle = open("/dev/audio", O_WRONLY)) < 0) {
	    perror("Can't open `/dev/audio' - try -noaudio");
	}
	else {
	    int h = open("ping.au", O_RDONLY);
	    if (h < 0) {
		perror("Can't open `ping.au' - try -noaudio");
	    }
	    else {
		struct stat st;
		fstat(h, &st);
		/* Assume 8k sampling; one ping per sec. */
		ping_noise_size = (st.st_size > 7000 ? 7000 :
				   st.st_size);
		ping_noise = malloc(ping_noise_size);
		read(h, ping_noise, ping_noise_size);
		close(h);
	    }
	}
    }

    if (!fw_init_listen(1, 0)) exit(1);

/* Damn!  We can't setsockopt to add/delete rules unless we're root. */
#if 0
    /* Drop privs */
    setreuid(NOBODY_UID, NOBODY_UID);
    setregid(NOBODY_GID, NOBODY_GID);
#endif

    if (!fw_register_interest(&fw, "input", readpkt, "DENY", 1, 0)) exit(1);

    ret = fw_do_listen(0);

    fw_close_listen(0);
    if (ping_noise) free(ping_noise);

    exit(!ret);
}
