#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "mxSetup.h"
#include "mxBase64String.h"
#include "mxQuotedPrintableString.h"
#include "mxMailMsg.h"

////////////////////////////////////////////////////////////////////////////////
//
//	Copy Constructor
//
////////////////////////////////////////////////////////////////////////////////
mxMailMsg::mxMailMsg(const mxMailMsg& msg)
	:mxMsgSummary(msg)
{
	_bodyText      = msg._bodyText;
	_attachments   = msg._attachments;
	_mime          = msg._mime;
	_mimeSeparator = msg._mimeSeparator;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Assignment operator
//
////////////////////////////////////////////////////////////////////////////////
mxMailMsg&	mxMailMsg::operator=(const mxMailMsg& msg)
{
	if (this != &msg)
	   {
	    // Should use base class assignment operator here
	    _author        = msg._author;
	    _organisation  = msg._organisation;
	    _date          = msg._date;
	    _toAddressee   = msg._toAddressee;
	    _ccAddressee   = msg._ccAddressee;
	    _bccAddressee  = msg._bccAddressee;
	    _mailid        = msg._mailid;
	    _messageId     = msg._messageId;
	    _subject       = msg._subject;
	    _actualText    = msg._actualText;
	    _status        = msg._status;
	    _position      = msg._position;
	    _length        = msg._length;

	    _bodyText      = msg._bodyText;
	    _attachments   = msg._attachments;
	    _mime          = msg._mime;
	    _mimeSeparator = msg._mimeSeparator;
	   }
	return *this;
}
////////////////////////////////////////////////////////////////////////////////
//
//	set Message Details
//	NB The 'string' passed is the whole of the message including
//	   the headers/attachments etc - it is split up in here into
//	   header/text/attachments
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailMsg::setMessage(string body)
{
	string	msg_part;
	int	no_of_tags=0;
	int	pos_enc_start;
	int	pos_disp_start;
	int	pos_type_start;
	int	pos_part_start;
	int	pos_body_start;
	int	pos;

//	Set up message headers

	setMessageHeaders(body);

//	Find start of the message itself

	pos_body_start = bodyStart(body);

//	Search for MIME tags - find start of MIME 'part'

	// Find 'type' of MIME part
	pos_type_start = nextTagStart(body,"Content-Type");
	pos_enc_start  = nextTagStart(body,"Content-Transfer-Encoding");
	pos_disp_start = nextTagStart(body,"Content-Disposition");
	pos_part_start = pos_type_start;
	if (pos_enc_start >= 0)
	   {
	    if (pos_enc_start < pos_part_start)
	      pos_part_start = pos_enc_start;
	   }
	if (pos_disp_start >= 0)
	   {
	    if (pos_disp_start < pos_part_start)
	      pos_part_start = pos_disp_start;
	   }

	// Not a MIME format message - use whole message as text
	if (pos_part_start < 0 || pos_part_start > pos_body_start)
	   {
	    // text is everything after first blank line (after headers)
	    pos = bodyStart(_bodyText);
	    _bodyText.assign(body,pos_body_start,body.size()-pos_body_start);
	    _mime = false;
	   }

	// MIME format message - convert into attachments
	else
	   {
	    _bodyText.remove();
	    msg_part.assign(body,pos_part_start,body.size()-pos_part_start+1);
	    convertPartIntoAttachment(msg_part);
	    _mime = true;
	   }
}
////////////////////////////////////////////////////////////////////////////////
//
//	Decode message 'part' and load it as attachment/text
//
////////////////////////////////////////////////////////////////////////////////
bool	mxMailMsg::convertPartIntoAttachment(string part)
{
	int	pos_part_start;
	int	pos;
	int	tag_length=14,encoding_length=27;
	int	next_eol;
	string	type_tag;
	string	part_type;
	string	part_subtype;
	string	part_encoding;
	string	part_filename;

//	Find start of message part proper (omitting tags)

	pos_part_start = bodyStart(part);
	if (pos_part_start < 0)
	  return false;

//	Extract Content-Type tag and extract 'type','subtype' and 'filename'

	type_tag = tagValue(part,"Content-Type");

	pos = type_tag.find("/");
	if (pos >= 0)
	  {
	   part_type.assign(type_tag,0,pos);
	   int	pos_type_end;
	   pos_type_end = type_tag.find(";",pos);
	   if (pos_type_end >= 0)
	     part_subtype.assign(type_tag,pos+1,pos_type_end-pos-1);
	   else
	     part_subtype.assign(type_tag,pos+1,type_tag.size()-pos-1);
	   if (part_subtype.size() == 0)
	     part_subtype == "unknown";
	  }
	else
	  {
	   part_type    = type_tag;
	   part_subtype = "unknown";
	  }

	if ((pos = type_tag.find("name=")) >= 0)
	  {
	   int	pos2;
	   pos2 = type_tag.find(";",pos);
	   if (pos2 < 0)
	     pos2 = type_tag.size()-1;
	   part_filename.assign(type_tag,pos+5,pos2-pos-4);
	   if (part_filename[0] == '"')
	     part_filename.remove(0,1);
	   if (part_filename[part_filename.size()-1] == '"')
	     part_filename.remove(part_filename.size()-1,1);
	  }

//	Extract Content-Transfer-Encoding tag (optional specifier)

	part_encoding = tagValue(part,"Content-Transfer-Encoding");

//	Remove headers from part

	part.remove(0,pos_part_start);

//	If encoded in a supported format, then decode

	if (!part_encoding.empty())
	   {
	    if (part_encoding == "7bit")
	       {
		// Nothing to do here
	       }
	    else if (part_encoding == "base64")
	       {
		mxBase64String	base64;
		string		tempstr;

		tempstr = base64.decode(part);
		part = tempstr;
	       }
	    else if (part_encoding == "quoted-printable")
	       {
		mxQuotedPrintableString	qp;
		string			tempstr;

		tempstr = qp.decode(part);
		part = tempstr;
	       }
	    else
	       {
		// Unknown encoding so ignore it
	       }
	   }

//	Found Content-Type tag so save as relevant type

	// File Primitive type - just add as attachment
	if ((pos=type_tag.find("text/")) == 0 ||
	    (pos=type_tag.find("Text/")) == 0 ||
	    (pos=type_tag.find("TEXT/")) == 0 ||
	    (pos=type_tag.find("image/")) == 0 ||
	    (pos=type_tag.find("Image/")) == 0 ||
	    (pos=type_tag.find("IMAGE/")) == 0 ||
	    (pos=type_tag.find("audio/")) == 0 ||
	    (pos=type_tag.find("Audio/")) == 0 ||
	    (pos=type_tag.find("AUDIO/")) == 0 ||
	    (pos=type_tag.find("video/")) == 0 ||
	    (pos=type_tag.find("Video/")) == 0 ||
	    (pos=type_tag.find("VIDEO/")) == 0 ||
	    (pos=type_tag.find("application/")) == 0 ||
	    (pos=type_tag.find("Application/")) == 0 ||
	    (pos=type_tag.find("APPLICATION/")) == 0)
	   {
	    mxMailFile	file_attachment;
	    file_attachment.filename = part_filename;
	    file_attachment.contents = part;

	    mxMailAttachment	*attach;
	    attach = new mxMailAttachment(file_attachment);
	    attach->setSubtype(part_subtype);
	    if (part_type == "text" ||
		part_type == "Text" ||
		part_type == "TEXT")
	       {
		// Filename specified - add as attachment to allow filing
		if (!part_filename.empty())
		   {
		    attach->setType(MX_FILE);
		    addAttachment(*attach);
		   }
		// Just text, append to end of 'text'
		else
		   {
		    _bodyText += part;
		    return false;
		   }
	       }
	    else if (part_type == "image" ||
		     part_type == "Image" ||
		     part_type == "IMAGE")
	       {
		attach->setType(MX_IMAGE);
		addAttachment(*attach);
	       }
	    else if (part_type == "audio" ||
		     part_type == "Audio" ||
		     part_type == "AUDIO")
	       {
		attach->setType(MX_AUDIO);
		addAttachment(*attach);
	       }
	    else if (part_type == "video" ||
		     part_type == "Video" ||
		     part_type == "VIDEO")
	       {
		attach->setType(MX_VIDEO);
		addAttachment(*attach);
	       }
	    else if (part_type == "application" ||
		     part_type == "Application" ||
		     part_type == "APPLICATION")
	       {
		attach->setType(MX_APPLICATION);
		addAttachment(*attach);
	       }
	    delete attach;
	    return true;
	   }
	// Message Primitive type - just add as attachment
	else if ((pos=type_tag.find("message/")) == 0 ||
		 (pos=type_tag.find("Message/")) == 0 ||
		 (pos=type_tag.find("MESSAGE/")) == 0)
	   {
	    mxMailMsg	msg_attachment;
	    string	msg_part;

	    // Insert end-of-line to give message the correct form
	    msg_part = '\n' + part;
	    msg_attachment.setMessage(msg_part);

	    mxMailAttachment	*attach;
	    attach = new mxMailAttachment(msg_attachment);
	    attach->setType(MX_MESSAGE);
	    attach->setSubtype(part_subtype);
	    addAttachment(*attach);
	    delete attach;
	    return true;
	   }
	// Multipart type - split down to next level
	else if ((pos=type_tag.find("multipart/")) == 0 ||
		 (pos=type_tag.find("Multipart/")) == 0 ||
		 (pos=type_tag.find("MULTIPART/")) == 0)
	   {
	    int		pos1,pos2;
	    string	boundary;
	    string	boundary_start;
	    string	boundary_end;
	    string	sub_part;
	    int		boundary_start_pos,boundary_end_pos;
	    bool	end_reached=false;

	    // Extract boundary tag - used to split into subparts
	    boundary = "";
	    if ((pos1=type_tag.find("boundary=",pos)) >= 0)
	       {
		boundary.assign(type_tag,pos1+9,type_tag.size()-pos1-9);

		// If more tags after, chop them off
		pos = boundary.find(';');
		if (pos >= 0)
		  boundary.remove(pos,boundary.size()-pos);

		// Remove any leading/trailing quote characters
		if (boundary[0] == '"')
		   {
		    boundary.remove(0,1);
		    pos = boundary.find('"');
		    if (pos >= 0)
		       {
			boundary.remove(pos,boundary.size()-pos);
		       }
		   }
		_mimeSeparator = boundary;

		// Set up start/end boundary tag strings
		boundary_start = boundary;
		boundary_end   = boundary_start;
		boundary_end.insert(boundary_end.size(),"--");
	       }
	    // No boundary tag - append the part to the 'text' area
	    else
	       {
		_bodyText += part;
		_mime = false;
		return false;
	       }

	    // Check if any boundary tags exist
	    if ((boundary_start_pos = part.find(boundary_start)) < 0)
	       {
		// No boundary tag found - just append to text area
		_bodyText += part;
		_mime = FALSE;
		return false;
	       }

	    // Find start of next sub-part
	    while (!end_reached)
	       {
		boundary_end_pos = part.find(boundary_start,boundary_start_pos+1);
		if (boundary_end_pos < 0)
		  end_reached = true;
		else
		   {
		    //Set sub_part to region between boundary tags and decode
		    sub_part.assign(part,boundary_start_pos+1,boundary_end_pos-boundary_start_pos-3);
		    convertPartIntoAttachment(sub_part);
		    boundary_start_pos = boundary_end_pos;
		    _mime = true;
		   }
	       }
	    return true;
	   }
	// Unsupported MIME type - add to message text
	else
	   {
	    _bodyText += part;
	    return true;
	   }
}
////////////////////////////////////////////////////////////////////////////////
//
//	add an attachment
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailMsg::addAttachment(mxMailAttachment& msg_att)
{
	_attachments.push_back(msg_att);
}
////////////////////////////////////////////////////////////////////////////////
//
//	remove attachments
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailMsg::removeAttachments()
{
	for (int i=_attachments.size();i>0;i--)
	  _attachments.pop_back();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Write to file
//
////////////////////////////////////////////////////////////////////////////////
bool	mxMailMsg::writeToFile(string filename,bool append,bool sendheaders) const
{
	bool		success=true;

//	Append to file

	if (append)
	  {
	   ofstream	outfile_append((char *)filename.c_str(),ios::app);
	   if (outfile_append)
	     writeMsgToStream(outfile_append,sendheaders);
	   else
	     success = false;
	   outfile_append.close();
	  }

//	Overwrite file

	else
	  {
	   ofstream	outfile_overwrite((char *)filename.c_str());
	   if (outfile_overwrite)
	     writeMsgToStream(outfile_overwrite,sendheaders);
	   else
	     success = false;
	   outfile_overwrite.close();
	  }

	return success;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Write message to file stream
//
////////////////////////////////////////////////////////////////////////////////
void	mxMailMsg::writeMsgToStream(ofstream& outfile,bool sendheader) const
{
	string	mime_separator;

//	Set up MIME separator if not already set

	mime_separator = _mimeSeparator;
	if (mime_separator.size() == 0)
	  {
	   time_t	the_time;
	   char		tempstr[20];

	   time(&the_time);
	   sprintf(tempstr,"%d",the_time);
	   mime_separator = "mxMail-MIME-Separator-";
	   mime_separator += tempstr;
	  }

//	Add send-header if required (the ones that 'sendmail' adds)

	if (sendheader)
	   {
	    outfile << "From " << theMxSetup->mailSenderName() << " " << _date << endl;
	   }

//	Write message header

	outfile << "From: " << _author << " <" << _mailid << ">" << endl;
	outfile << "Date: " << _date << endl;
	if (!_toAddressee.empty())
	  outfile << splitAddressees("To: ",_toAddressee) << endl;
	if (!_ccAddressee.empty())
	  outfile << splitAddressees("Cc: ",_ccAddressee) << endl;
	if (!_bccAddressee.empty())
	  outfile << splitAddressees("Bcc: ",_bccAddressee) << endl;
	if (!(_organisation.empty()))
	  outfile << "Organization: " << _organisation << endl;
	outfile << "Subject: " << _subject << endl;
	outfile << "Message-Id: <" << _messageId << ">" << endl;

//	MIME format message

	if (theMxSetup->mailMimeFormat())
	  {
	   outfile << "MIME-Version: 1.0" << endl;

	   if (_attachments.size() > 0)
	     {
	      outfile << "Content-Type: multipart/mixed; boundary=\"";
	      outfile << mime_separator << "\"" << endl << endl;
	      outfile << "This is a multi-part message in MIME format." << endl << endl;
	      outfile << "--" << mime_separator << endl;
	     }

	   // Write message text

	   outfile << "Content-Type: text/plain; charset=US-ASCII" << endl;
	   outfile << "Content-Transfer-Encoding: 7bit" << endl << endl;
	   outfile << _bodyText << endl;

	   // Write message attachments

	   for (int i=0;i<_attachments.size();i++)
	     {
	      outfile << "--" << mime_separator << endl;

	      if (_attachments[i].type() == MX_MESSAGE)
	        {
	         outfile << "Content-Type: message/" << _attachments[i].subtype() << endl;
	         outfile << "Content-Transfer-Encoding: 7bit" << endl;
	         outfile << "Content-Disposition: inline" << endl << endl;

	         _attachments[i].message().writeMsgToStream(outfile,FALSE);
	        }
	      else if (_attachments[i].type() == MX_FILE ||
		       _attachments[i].type() == MX_IMAGE ||
		       _attachments[i].type() == MX_AUDIO ||
		       _attachments[i].type() == MX_VIDEO ||
		       _attachments[i].type() == MX_APPLICATION)
	        {	

	         // Write the message_part to file

	         if (_attachments[i].type() == MX_FILE)
		   outfile << "Content-Type: text/" << _attachments[i].subtype() << "; name=\"" << _attachments[i].file().filename << "\"" << endl;
	         else if (_attachments[i].type() == MX_IMAGE)
		   outfile << "Content-Type: image/" << _attachments[i].subtype() << "; name=\"" << _attachments[i].file().filename << "\"" << endl;
	         else if (_attachments[i].type() == MX_AUDIO)
		   outfile << "Content-Type: audio/" << _attachments[i].subtype() << "; name=\"" << _attachments[i].file().filename << "\"" << endl;
	         else if (_attachments[i].type() == MX_VIDEO)
		   outfile << "Content-Type: video/" << _attachments[i].subtype() << "; name=\"" << _attachments[i].file().filename << "\"" << endl;
	         else if (_attachments[i].type() == MX_APPLICATION)
		   outfile << "Content-Type: application/" << _attachments[i].subtype() << "; name=\"" << _attachments[i].file().filename << "\"" << endl;

	         outfile << "Content-Transfer-Encoding: base64" << endl;
	         outfile << "Content-Disposition: inline; filename=\"" << _attachments[i].file().filename << "\"" << endl << endl;

	         // Encode the file (in base64)

		 string		encoded_part;
		 mxBase64String	base64;

		 // Encode all text files as TEXT, and others as BINARY
		 // NB Really need a file-type detection mechanism here
		 //    to do this properly
		 if (_attachments[i].type() == MX_FILE)
		   encoded_part = base64.encode(_attachments[i].file().contents);
		 else
		   encoded_part = base64.encode(_attachments[i].file().contents);

	         outfile << encoded_part << endl;
	        }
	      else
	        {
	        }
	     }
	   if (_attachments.size() > 0)
	     outfile << "--" << mime_separator << "--" << endl;
	  }

//	Text only format message

	else
	  {
	   outfile << endl;

	   // Write message text

	   outfile << _bodyText << endl;

	   // Write message attachments

	   for (int i=0;i<_attachments.size();i++)
	     {
	      if (_attachments[i].type() == MX_MESSAGE)
		{
		 outfile << "---->Attached Message<----" << endl;
	         _attachments[i].message().writeMsgToStream(outfile,FALSE);
		 outfile << "---->End Attached Message<----" << endl;
		}
	      else if (_attachments[i].type() == MX_FILE)
		{
		 outfile << "---->Attached Text File '" << _attachments[i].file().filename << "'<----" << endl;
	         outfile << _attachments[i].file().contents << endl;
		 outfile << "---->End Attached Text File<----" << endl;
		}
	      else if (_attachments[i].type() == MX_IMAGE)
		{
		 outfile << "---->Attached Image File '" << _attachments[i].file().filename << "'<----" << endl;
	         outfile << _attachments[i].file().contents << endl;
		 outfile << "---->End Attached Image File<----" << endl;
		}
	      else if (_attachments[i].type() == MX_AUDIO)
		{
		 outfile << "---->Attached Audio File '" << _attachments[i].file().filename << "'<----" << endl;
	         outfile << _attachments[i].file().contents << endl;
		 outfile << "---->End Attached Audio File<----" << endl;
		}
	      else if (_attachments[i].type() == MX_VIDEO)
		{
		 outfile << "---->Attached Video File '" << _attachments[i].file().filename << "'<----" << endl;
	         outfile << _attachments[i].file().contents << endl;
		 outfile << "---->End Attached Video File<----" << endl;
		}
	      else if (_attachments[i].type() == MX_APPLICATION)
		{
		 outfile << "---->Attached Application '" << _attachments[i].file().filename << "'<----" << endl;
	         outfile << _attachments[i].file().contents << endl;
		 outfile << "---->End Attached Application<----" << endl;
		}
	      else
		{
		}
	     }
	  }
}
////////////////////////////////////////////////////////////////////////////////
//
//	Split addressees onto separate lines
//	NB Should really put addresses as
//		"address_name <address>"
//	   but need address book for that
//	I.E
//	address_name = _addressbook->book().nameForAddress(address);
//	if (!address_name.empty())
//	  outbox << "To: " << address_name << " <" << address << ">," << endl;
//	else
//	  outbox << "To: " << address << "," << endl;
//
////////////////////////////////////////////////////////////////////////////////
string	mxMailMsg::splitAddressees(string prefix_label,string addresslist) const
{
	string	list;
	string	address;
	int	pos1,pos2;

	list = prefix_label;
	pos1 = 0;
	pos2 = addresslist.find(',',pos1);

	// Only one address in list
	if (pos2 < 0)
	  {
	   address.assign(addresslist,pos1,addresslist.size()-pos1);
	   list += address;
	  }
	// Multiple elements in list
	else
	  {
	   address.assign(addresslist,pos1,pos2-pos1);
	   list += address;
	   pos1 = pos2 + 1;

	   while ((pos2=addresslist.find(',',pos1)) >= 0)
	     {
	      address.assign(addresslist,pos1,pos2-pos1);
	      list += ",\n    ";
	      list += address;
	      pos1 = pos2 + 1;
	     }
	   address.assign(addresslist,pos1,addresslist.size()-pos1);
	   list += ",\n    ";
	   list += address;
	  }

	return list;
}
