/* xdp.c -- routines for reading and writing xdp files			*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"
#include "rfiles.h"
#include <sys/stat.h>

#define	XDP_VERSION		"0.1"
#define	XDP_MAGIC_FORMAT	"! xdp-%s\n"
#define	XDP_MAGIC		"! xdp-"

#ifdef	__STDC__
static	String	get_application_resource(String);
static	String	get_widget_resource(Widget, String);
static	Bool	read_xdp_stream(FILE *);
static	String	read_xdp_line(FILE *, int *);
static	int	stream_escape(FILE *, int *);
#else	/* __STDC__ */
static	String	get_application_resource();
static	String	get_widget_resource();
static	Bool	read_xdp_stream();
static	String	read_xdp_line()
static	int	stream_escape();
#endif	/* __STDC__ */

/* write_xdp -- write resources to an xdp file
 *
 * String	file_name;
 * String	save_list;
 * if (write_xdp(file_name, save_list)) error...
 *
 * save_list is the name of a whitespace-separated list of resources.
 * Their values are fetched and written to the indicated file in X
 * resources format.
 */
Bool
write_xdp(fname, list)
String		fname;
String		list;
{
    FILE		*xdp;
    Bool		ret;

    if (NULL == (xdp = fopen(fname, "w"))) return(TRUE);
    ret = write_xdp_stream(xdp, list);
    fclose(xdp);
    return(ret);
}

Bool
write_xdp_stream(xdp, list)
FILE		*xdp;
String		list;
{
    String		warn = XtNewString("");
    int			ncwarn = 0;
    String		s;
    int			len;
    String		t = XtNewString("");
    int			tlen = 0;
    String		val;
    String		msg;
    Bool		ret = FALSE;

    fprintf(xdp, XDP_MAGIC_FORMAT, XDP_VERSION);
    for(
	s = list;
	'\0' != s[strspn(s, WHITE)];
	s += len
    ) {
	s += strspn(s, WHITE);
	len = strcspn(s, WHITE);
	if (len > tlen) {
	    t = (String) XtRealloc(t, len+1);
	    tlen = len;
	}
	strncpy(t, s, len);
	t[len] = '\0';			/* fully qualified resource	*/
	if (NULL != (msg = get_resource_val(t, &val))) {
	    warn = append_string(warn, msg, &ncwarn);
	    ret = TRUE;
	    continue;
	}
	put_resource_line(xdp, t, val);
    }
    Free(t);
    if (0 < ncwarn) PU_error(warn, NULL);
    Free(warn);
    return(ret);
}

String
get_resource_val(t, val)
String	t;
String	*val;
{
    char		lbuf[LLEN];
    String		u;
    String		v;
    String		res = XtNewString(t);
    String		wid = XtNewString(t);
    static String	msg = NULL;
    Widget		widget;

    u = STRRCHR(t, '.');
    v = STRRCHR(t, '*');
    if ((NULL != u) || (NULL != v)) {
	if ((NULL == u) || ((NULL != v) && (v > u))) u = v;
	wid[u-t] = '\0';		/* widget name			*/
	strcpy(res, u+1);		/* resource name		*/
    }
    else {
	strcpy(wid, "*");
	strcpy(res, u);
    }
    if (strlen(wid) == strspn(wid, "*.")) {
	widget = toplevel;
    }
    else {
	if (NO_WIDGET == (widget = XtNameToWidget(toplevel, wid))) {
	    sprintf(lbuf, "unknown widget %s\n", wid);
	    Nfree(msg);
	    msg = XtNewString(lbuf);
	    Free(res);
	    Free(wid);
	    return(msg);
	}
    }
    if (widget == toplevel) *val = get_application_resource(res);
    else *val = get_widget_resource(widget, res);
    if (NULL == *val) {
	sprintf(lbuf, "unknown resource %s.%s\n", wid, res);
	Nfree(msg);
	msg = XtNewString(lbuf);
	Free(res);
	Free(wid);
	return(msg);
    }
    Free(res);
    Free(wid);
    return(NULL);
}

/* get_application_resource -- return app resource value as String	*/
static String
get_application_resource(res)
String		res;
{
    XtResourceList	list = resources;
    int			nlist = num_app_resources;
    int			i;

    for(i=0; i<nlist; i++) {
	if (
	    (0 == strcmp(res, list[i].resource_name)) ||
	    (0 == strcmp(res, list[i].resource_class))
	) break;
    }
    if (i >= nlist) return(NULL);
    return(resource_to_string(toplevel, &app_data, &list[i]));
}

static String
get_widget_resource(widget, res)
Widget	widget;
String	res;
{
    String		s = NULL;
    XtResourceList	reslist;
    Cardinal		nres;
    register int	i;
    String		type;
    Cardinal		size;
    XrmValue		from;
    XrmValue		to;
    Bool		ret;

    /*
     * First, find out what type and how big it is (there has to be an
     * easier way...)
     */
    XtGetResourceList(XtClass(widget), &reslist, &nres);
    for(i=0; i<nres; i++) {
	if (
	    (0 == strcmp(res, reslist[i].resource_name)) ||
	    (0 == strcmp(res, reslist[i].resource_class))
	) break;
    }
    if (i >= nres) {
	Free(reslist);
	return(NULL);
    }
    type = XtNewString(reslist[i].resource_type);
    size = reslist[i].resource_size;
    Free(reslist);
    /*
     * If it's a string, simple
     */
    if (0 == strcmp(type, XtRString)) {
	XtVaGetValues(widget, res, &s, NULL);
	Free(type);
	return(s);
    }
    /*
     * Not a string: get value, then convert
     */
    from.addr = XtMalloc(size);
    from.size = size;
    XtVaGetValues(widget, res, from.addr, NULL);
    to.addr = NULL;
    ret = XtConvertAndStore(widget, type, &from, XtRString, &to);
    Free(type);
    return(ret ? (String) to.addr : NULL);
}

/* set_application_resource -- set app resource value from String	*/
Bool
set_application_resource(res, val)
String		res;
String		val;
{
    XtResource		*appres = resources;
    int			i;
    XrmValue		from;
    XrmValue		to;
    String		type;
    Bool		ret;

    for(i=0; i<num_app_resources; i++) {
	if (
	    (0 == strcmp(res, appres[i].resource_name)) ||
	    (0 == strcmp(res, appres[i].resource_class))
	) break;
    }
    if (i >= num_app_resources) return(TRUE);
    ret = string_to_resource(toplevel, &app_data, &appres[i], val);
    set_debug(app_data.debug);
    return(ret);
}

/* xdp_get_datafile -- get datafile name from an xdp file
 * xdp_get_values -- get widget values from an xdp file
 *
 * String	xdp_file_name;
 * String	datafile_name;
 * datafile_name = xdp_get_datafile(xdp_file_name);
 * if (NULL == datafile_name) error...
 *
 *  ...open datafile...
 *
 * if (xdp_get_values(xdp_file_name) error...
 *
 * There are two steps to reading an xdp file.  First, call
 * xdp_get_datafile to get the name of the datafile refered to by the
 * xdp file.  It returns a pointer to the datafile name, which should
 * eventually be freed with XtFree.  If the file isn't an xdp file or
 * if the datafile can't be found, xdp_get_datafile returns NULL.
 *
 * The second step comes after the datafile has been loaded.  Call
 * xdp_get_vales with the same xdp file name to set widget and
 * application resource values.  xdp_get_values should only be called
 * after a successful call to xdp_get_datafile, since it doesn't check
 * that the file is an xdp file.  If it encounters an error it pops up
 * a warning message and returns TRUE; otherwise it returns FALSE.
 */
String
xdp_get_datafile(fname)
String		fname;
{
    FILE		*xdp;
    Bool		ret;
    char		lbuf[LLEN];
    XrmDatabase		db;
    String		class;
    String		name;
    String		type;
    XrmValue		where;
    struct stat		stat_buf;

    if (NULL == (xdp = fopen(fname, "r"))) return(NULL);
    ret = (1 != fread(lbuf, strlen(XDP_MAGIC), 1, xdp)) ||
	  (0 != strncmp(lbuf, XDP_MAGIC, strlen(XDP_MAGIC)));
    fclose(xdp);
    if (ret) return(NULL);
    if (NULL == (db = XrmGetFileDatabase(fname))) return(NULL);
    sprintf(lbuf, "%s.%s", CLASS, "DataFile");
    class = XtNewString(lbuf);
    sprintf(lbuf, "%s.%s", XtName(toplevel), "dataFile");
    name = XtNewString(lbuf);
    ret = (!XrmGetResource(db, name, class, &type, &where)) ||
	  (0 != strcmp(XtRString, type));
    Free(name);
    Free(class);
    if (!ret) strncpy(lbuf, (String) where.addr, LLEN);
    XrmDestroyDatabase(db);
    if (ret) return(NULL);
    for(
	name = lbuf;
	((String) NULL + 1) != name;
	name = 1 + STRCHR(name, DIRSEP)
    ) {
	if ((EOF != stat(name, &stat_buf)) && S_ISREG(stat_buf.st_mode))
	    break;
    }
    if (((String) NULL + 1) == name) return(XtNewString(lbuf));
    return(XtNewString(name));
}

Bool
xdp_get_values(fname)
String		fname;
{
    FILE		*xdp;
    int			ret;

    if (NULL == (xdp = fopen(fname, "r"))) return(TRUE);
    ret = read_xdp_stream(xdp);
    fclose(xdp);
    return(ret);
}

static Bool
read_xdp_stream(xdp)
FILE		*xdp;
{
    char		lbuf[LLEN];
    String		warn = XtNewString("");
    int			ncwarn = 0;
    int			lineno = 0;
    String		line;
    String		wid;
    String		res;
    String		val;
    Widget		widget;
    Bool		ret = FALSE;

    while(NULL != (line = read_xdp_line(xdp, &lineno))) {
	if (parse_resource_line(line, &wid, &res, &val)) {
	    sprintf(lbuf, "parse error in xdp file near line %d\n", lineno);
	    warn = append_string(warn, lbuf, &ncwarn);
	    ret = TRUE;
	    continue;
	}
	Free(line);
	if (NULL == wid) continue;
	if (strlen(wid) == strspn(wid, "*.")) {
	    widget = toplevel;
	}
	else {
	    if (NO_WIDGET == (widget = XtNameToWidget(toplevel, wid))) {
		sprintf(lbuf, "unknown widget %s in xdp file near line %d\n",
			wid, lineno);
		warn = append_string(warn, lbuf, &ncwarn);
		ret = TRUE;
		continue;
	    }
	}
	if (
	    (widget != toplevel) ||
	    (set_application_resource(res, val))
	) {
	    XtVaSetValues(widget,
		XtVaTypedArg, res, XtRString,
		val, strlen(val)+1,
	    NULL);
	}
	Free(wid);
	Free(res);
	Free(val);
    }
    if (0 < ncwarn) PU_error(warn, NULL);
    Free(warn);
    return(ret);
}

static String
read_xdp_line(xdp, lineno)
FILE		*xdp;
int		*lineno;
{
    String		line = XtMalloc(LLEN);
    String		s;
    int			len = 0;
    int			nc = LLEN;
    int			c;
    Bool		comment = FALSE;

    s = line;
    while((EOF != (c = getc(xdp))) && ('\n' != c)) {
	if ('!' == c) comment = TRUE;
	if ('\\' == c) c = stream_escape(xdp, lineno);
	if ((EOF != c) && !comment) addc(line, s, len, nc, c);
    }
    addc(line, s, len, nc, '\0');
    if ((EOF == c) && ('\0' == line[0])) {
	Free(line);
	line = NULL;
    }
    else {
	line = XtRealloc(line, len);
	if ('\n' == c) (*lineno)++;
    }
    return(line);
}

static int
stream_escape(xdp, lineno)
FILE		*xdp;
int		*lineno;
{
    int			c;
    int			d;

    switch(c = getc(xdp)) {
    case 'x':				/* hex escape			*/
	c = 0;
	while(isxdigit(d = getc(xdp))) {
	    c *= 0x10;
	    c += isdigit(d) ? (d - '0') : (tolower(d) - 'a' + 0xa);
	}
	ungetc(d, xdp);
	break;

    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
	ungetc(c, xdp);
	c = 0;
	while(('0' <= (d = getc(xdp))) && ('7' >= d)) {
	    c *= 010;
	    c += d - '0';
	}
	ungetc(d, xdp);
	break;

    case 'a':
	c = '\a';
	break;

    case 'b':
	c = '\b';
	break;

    case 'f':
	c = '\f';
	break;

    case 'n':
	c = '\n';
	break;

    case 'r':
	c = '\r';
	break;

    case 't':
	c = '\t';
	break;

    case 'v':
	c = '\v';
	break;

    case '\n':
	c = EOF;
	(*lineno)++;
	break;

    case EOF:
    default:
	break;
    }
    return(c);
}
