/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter and Red Hat Software
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Dick Porter <dick@cymru.net>
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "orbit.h"

#include "orbit_poa.h"
#include "orbit_object.h"
#include "orbit_object_type.h"

static void ORBit_ORB_release(CORBA_ORB orb, CORBA_Environment *ev);

ORBit_RootObject_Interface CORBA_ORB_epv =
{
	(gpointer)ORBit_ORB_release,
};


static int ORBit_ORBid_setup(CORBA_ORB orb, CORBA_ORBid id, CORBA_boolean use_boa)
{
	g_assert(orb!=NULL);
	g_assert(id!=NULL);

	if(!strcmp(id, "orbit-local-orb")) {
		/* Default to POA */
		orb->use_poa=use_boa?CORBA_FALSE:CORBA_TRUE;
	} else if(!strcmp(id, "mico-local-orb")) {
		/* Fall back to BOA for MICO compatibility */
		orb->use_poa=CORBA_FALSE;
	} else {
		fprintf(stderr, "ORBit_ORBid_setup: Unknown ORB id: %s\n", id);
		return(0);
	}

	orb->orb_identifier=g_strdup(id);

	return(1);
}

static void ORBit_ORB_release(CORBA_ORB orb, CORBA_Environment *ev) 
{
	g_assert(orb!=NULL);

	if(--(ORBIT_ROOT_OBJECT(orb)->refs))
		return;

	if(orb->orb_identifier!=NULL) {
		g_free(orb->orb_identifier);
	}
	if(!CORBA_Object_is_nil(orb->imr, ev)) {
		CORBA_Object_release(orb->imr, ev); 
	}
	if(!CORBA_Object_is_nil(orb->ir, ev)) {
		CORBA_Object_release(orb->ir, ev); 
	}
	if(!CORBA_Object_is_nil(orb->naming, ev)) {
		CORBA_Object_release(orb->naming, ev);
	}
	if(!CORBA_Object_is_nil(orb->root_poa, ev)) {
		CORBA_Object_release(orb->root_poa, ev);
	}
	giop_connection_unref(orb->connection);

	g_free(orb);
}


/* Section 4.4
 *
 * Adjusts argc and argv appropriately
 */
CORBA_ORB CORBA_ORB_init(int *argc, char **argv, CORBA_ORBid orb_identifier, CORBA_Environment *ev)
{
	CORBA_ORB orb=NULL;
	int no_iiop_server=0;
	int no_iiop_proxy=0;
	int use_boa=0;
	int debug_level=0;
	int debug_modules=0;
	char *imr_ior=NULL, *imr_addr=NULL;
	char *ir_ior=NULL, *ir_addr=NULL;
	char *naming_ior=NULL, *naming_addr=NULL;
	char *root_poa_ior=NULL, *root_poa_addr=NULL;
	char *orb_id_opt=NULL;

	/* The variable addresses in this struct need to be assigned at
	 * run-time if you want to compile with -pedantic
	 *
	 * (You will also get scads of warnings about "long long" too)
	 */

	/* These options are compatible with MICO */
	orb_options options[]={
		{"ORBNoIIOPServer", no_arg, &no_iiop_server},
		{"ORBNoIIOPProxy", no_arg, &no_iiop_proxy},
		{"ORBid", string_arg, &orb_id_opt},
		{"ORBImplRepoIOR", string_arg, &imr_ior},
		{"ORBImplRepoAddr", string_arg, &imr_addr},
		{"ORBIfaceRepoIOR", string_arg, &ir_ior},
		{"ORBIfaceRepoAddr", string_arg, &ir_addr},
		{"ORBNamingIOR", string_arg, &naming_ior},
		{"ORBNamingAddr", string_arg, &naming_addr},
		{"ORBDebugLevel", int_arg, &debug_level},
		{"ORBBindAddr", string_arg, NULL}, /* XXX need to make
						      libIIOP support this */
		{"ORBIIOPAddr", string_arg, NULL},

	/* These options aren't */
		{"ORBDebugModules", int_arg, &debug_modules},
		{"ORBUseBOA", no_arg, &use_boa},
		{"ORBRootPOAIOR", string_arg, &root_poa_ior},
		{"ORBRootPOAADDR", string_arg, &root_poa_addr},
		{NULL,0,NULL},
	};

	g_return_val_if_fail(ev != NULL, NULL);
	g_return_val_if_fail(argc != NULL, NULL);
	g_return_val_if_fail(argv != NULL, NULL);
	g_return_val_if_fail(orb_identifier != NULL, NULL);

	ORBit_option_parse(argc, argv, options);

	ORBit_Trace_setLevel(debug_level);
	ORBit_Trace_setModules(debug_modules);

	CORBA_exception_init(ev);


	ORBit_chunks_init();

	giop_init();

	orb=g_new0(struct CORBA_ORB_type, 1);

	if(orb==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		goto error;
	}

	ORBit_pseudo_object_init(ORBIT_PSEUDO_OBJECT(orb), ORBIT_PSEUDO_ORB);
	
	ORBit_RootObject_set_interface(ORBIT_ROOT_OBJECT(orb),&CORBA_ORB_epv,ev);

	ORBIT_ROOT_OBJECT(orb)->refs = 1;

	if(orb_id_opt!=NULL) {
		if(!ORBit_ORBid_setup(orb, orb_id_opt, use_boa))
			goto error;
		g_free(orb_id_opt);
	} else if(orb_identifier!=NULL) {
		if(!ORBit_ORBid_setup(orb, orb_identifier, use_boa))
			goto error;
	} else {
		orb->orb_identifier=g_strdup("orbit-local-orb");
		orb->use_poa=use_boa?CORBA_FALSE:CORBA_TRUE;
	}

	if(orb->orb_identifier==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		goto error;
	}

	orb->connection = GIOP_CONNECTION(iiop_connection_server());
	giop_connection_ref(orb->connection);
	GIOP_CONNECTION(orb->connection)->orb_data = orb;

	/* when I figure out what MICO is doing with the iiop_proxy and
	 * iiop_server stuff, it'll get handled here.
	 */

	/*
	 * Connect to / create implementation repository
	 */

	{
		CORBA_Object imr=NULL;

		if(imr_ior!=NULL) {
			imr=CORBA_ORB_string_to_object(orb, imr_ior, ev);
			g_free(imr_ior);
		} else if(imr_addr!=NULL) {
			/*imr=CORBA_ORB_bind(orb, "IDL:omg.org/CORBA/ImplRepository:1.0", imr_addr, ev);*/
			g_free(imr_addr);
		}

		if(!CORBA_Object_is_nil(imr, ev)) {
			CORBA_ORB_set_initial_reference(orb, "ImplementationRepository", imr, ev);
		}
	}

	/*
	 * Connect to / create interface repository
	 */

	{
		CORBA_Object ir=NULL;

		if(ir_ior!=NULL) {
			ir=CORBA_ORB_string_to_object(orb, ir_ior, ev);
			g_free(ir_ior);
		} else if(ir_addr!=NULL) {
			/*ir=CORBA_ORB_bind(orb, "IDL:omg.org/CORBA/Repository:1.0", ir_addr, ev);*/
			g_free(ir_addr);
		}

		if(!CORBA_Object_is_nil(ir, ev)) {
			CORBA_ORB_set_initial_reference(orb, "InterfaceRepository", ir, ev);
		}
	}

	/*
	 * Connect to naming service
	 */

	{
		CORBA_Object naming=NULL;

		if(naming_ior!=NULL) {
			naming=CORBA_ORB_string_to_object(orb, naming_ior, ev);
			g_free(naming_ior);
		} else if(naming_addr!=NULL) {
			/*CORBA_ORB_ObjectTag tag=CORBA_ORB_string_to_tag(orb, "root", ev);*/

			/*naming=CORBA_ORB_bind_tag(orb, "IDL:omg.org/CosNaming/NamingContext:1.0", tag, naming_addr, ev);*/
			g_free(naming_addr);
		}

		if(!CORBA_Object_is_nil(naming, ev)) {
			CORBA_ORB_set_initial_reference(orb, "NameService", naming, ev);
		}
	}

	/*
	 * Connect to / create RootPOA
	 */

	{
		PortableServer_POA root_poa=NULL;
		PortableServer_POAManager poa_mgr=NULL;
		CORBA_PolicyList policies;
	  
		if(root_poa_ior!=NULL) {
			root_poa=(PortableServer_POA )CORBA_ORB_string_to_object(orb, root_poa_ior, ev);
			g_free(root_poa_ior);
		} else if(root_poa_addr!=NULL) {
			g_free(root_poa_addr);
		} else {
			/* Create a poa manager */
			poa_mgr = ORBit_POAManager_new();

			/* Initialise the policy list */
			policies._maximum=0;
			policies._length=0;
			policies._buffer=NULL; /* default policies */
			CORBA_sequence_set_release(&policies,CORBA_TRUE); 

			/* Create the root poa */
			root_poa = ORBit_POA_new("RootPOA",poa_mgr,&policies,ev);
		}
		       
		/* And attatch it to the orb */

		if(!CORBA_Object_is_nil((CORBA_Object)root_poa, ev)) {
			CORBA_ORB_set_initial_reference((CORBA_ORB)orb, "RootPOA", (CORBA_Object)root_poa, ev);
		}
	}


	return(orb);

error:
	if(orb!=NULL) {
		ORBit_ORB_release(orb, ev);
	}
	if(imr_ior!=NULL) {
		g_free(imr_ior);
	}
	if(imr_addr!=NULL) {
		g_free(imr_addr);
	}
	if(ir_ior!=NULL) {
		g_free(ir_ior);
	}
	if(ir_addr!=NULL) {
		g_free(ir_addr);
	}
	if(naming_ior!=NULL) {
		g_free(naming_ior);
	}
	if(naming_addr!=NULL) {
		g_free(naming_addr);
	}
	if(root_poa_ior!=NULL) {
		g_free(root_poa_ior);
	}
	if(root_poa_addr!=NULL) {
		g_free(root_poa_addr);
	}
	if(orb_id_opt!=NULL) {
		g_free(orb_id_opt);
	}
	return(NULL);
}

CORBA_char *CORBA_ORB_object_to_string(CORBA_ORB orb, CORBA_Object obj, CORBA_Environment *ev)
{
  int i;
  CDR_Codec *codec, *encaps;
  GString *retval;
  CORBA_char *rc;

  codec = CDR_codec_init();

  codec->buffer = g_new(CORBA_octet, 2048);
  codec->buf_len = 2048;
  codec->readonly = CORBA_FALSE;
  codec->host_endian = codec->data_endian = FLAG_ENDIANNESS;

  encaps = CDR_codec_init();
  encaps->buffer = g_new(CORBA_octet, 2048);
  encaps->buf_len = 2048;
  encaps->readonly = CORBA_FALSE;
  encaps->host_endian = encaps->data_endian = FLAG_ENDIANNESS;

  CDR_put_octet(codec, FLAG_ENDIANNESS);

  CDR_put_string(codec, obj->obj_id);
  CDR_put_ulong(codec, 1); /* Just a TAG_INTERNET_IOP */
  CDR_put_ulong(codec, IOP_TAG_INTERNET_IOP);

  CDR_put_octet(encaps, FLAG_ENDIANNESS);

  CDR_put_octet(encaps, 1); /* iiop_major */
  CDR_put_octet(encaps, 0); /* iiop_minor */
  CDR_put_string(encaps, IIOP_CONNECTION(obj->connection)->hostname);
  CDR_put_ushort(encaps, ntohs(IIOP_CONNECTION(obj->connection)->location.sin_port));
  CDR_put_ulong(encaps, obj->object_key._length);
  CDR_put_octets(encaps, obj->object_key._buffer, obj->object_key._length);

  CDR_put_ulong(codec, encaps->wptr);
  CDR_put_octets(codec, encaps->buffer, encaps->wptr);

  CDR_codec_free(encaps);

  retval = g_string_new("IOR:");
  for(i = 0; i < codec->wptr; i++)
    {
      g_string_sprintfa(retval, "%02x", codec->buffer[i]);
    }

  rc = CORBA_string_alloc(retval->len);
  strcpy(rc, retval->str);
  g_string_free(retval, TRUE);

  CDR_codec_free(codec);

  return rc;
}

static void ORBit_ORB_skip_encaps(CDR_Codec *codec)
{
	CORBA_octet *dump_buffer;
	CORBA_unsigned_long buf_len;

	if(CDR_get_ulong(codec, &buf_len)==CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "ORBit_ORB_skip_encaps: error reading encapsulation length\n");
		return;
	}

	dump_buffer=g_new(CORBA_octet, buf_len);
	if(dump_buffer==NULL) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "ORBit_ORB_skip_encaps: malloc error creating encapsulation dump_buffer\n");
		return;
	}

	/* Copying the buffer this way positions the read
	 * pointer in "codec" correctly, and gets the
	 * encapsulation out of our way in case of error.
	 */
	if(CDR_buffer_gets(codec, dump_buffer, buf_len)==CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "ORBit_ORB_skip_encaps: error reading encapsulation dump_buffer\n");
	}

	/* just dump it */
	g_free(dump_buffer);
}

CORBA_Object CORBA_ORB_string_to_object(CORBA_ORB orb, CORBA_char *str, CORBA_Environment *ev)
{
	CORBA_Object new;
	CDR_Codec *codec;
	CORBA_octet *buffer, byteorder;
	CORBA_char *oid;
	CORBA_unsigned_long seq_len;
	CORBA_unsigned_long buf_len;
	unsigned long len;
	int i,j;

	g_return_val_if_fail(ev != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(orb != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(str != NULL, CORBA_OBJECT_NIL);

	if(strncmp(str, "IOR:", 4)) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: unknown IOR reference\n");
		return(CORBA_OBJECT_NIL);
	}

	codec=CDR_codec_init();
	if(codec==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(CORBA_OBJECT_NIL);
	}

	len=strlen(str);
	if(len % 2) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: IOR reference has odd number of digits\n");
		CDR_codec_free(codec);
		return(CORBA_OBJECT_NIL);
	}
	if(len<4) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: IOR reference is too short\n");
		CDR_codec_free(codec);
		return(CORBA_OBJECT_NIL);
	}

	buf_len=(len-4)/2;
	buffer=g_new(CORBA_octet, buf_len);
	if(buffer==NULL) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: malloc error creating buffer\n");
		CDR_codec_free(codec);
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(CORBA_OBJECT_NIL);
	}

	codec->buffer=buffer;
	codec->buf_len=buf_len;
	codec->readonly=CORBA_TRUE;

	j=0;
	for(i=4; i<len; i+=2) {
		buffer[j++]=HEXOCTET(str[i], str[i+1]);
	}

	if(CDR_get_octet(codec, &byteorder)==CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading byteorder octet\n");
		CDR_codec_free(codec);
		return(CORBA_OBJECT_NIL);
	}

	if(byteorder) {
		codec->data_endian=LittleEndian;
	} else {
		codec->data_endian=BigEndian;
	}

	/* CORBA_char *type_id */

	if(CDR_get_string(codec, &oid)==CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading oid string\n");
		CDR_codec_free(codec);
		return(CORBA_OBJECT_NIL);
	}
	ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: OID: %s\n", oid);

	/* CORBA_sequence_TaggedProfile profiles */

	if(CDR_get_seq_begin(codec, &seq_len)==CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading profile list length\n");
		g_free(oid);
		CDR_codec_free(codec);
		return(CORBA_OBJECT_NIL);
	}
	ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: %d profiles\n", seq_len);

	/* Use the first TAG_INTERNET_IOP profile */
	for(i=0; i<seq_len; i++) {
		IOP_ProfileId tag;

		if(CDR_get_ulong(codec, &tag)==CORBA_FALSE) {
			ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading profile tag no.%d\n", i);
			g_free(oid);
			CDR_codec_free(codec);
			return(CORBA_OBJECT_NIL);
		}

		if(tag==IOP_TAG_INTERNET_IOP) {
			CDR_Codec *encaps_codec;
			CORBA_octet iiop_major, iiop_minor;
			CORBA_char *host;
			CORBA_unsigned_short port;
			CORBA_unsigned_long key_len;
			CORBA_octet *object_key;

			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: profile no.%d type TAG_INTERNET_IOP\n", i);

			if(CDR_get_ulong(codec, &buf_len)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading encapsulation length\n");
				continue;
			}

			encaps_codec=CDR_codec_init();
			if(encaps_codec==NULL) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error creating encapsulation codec\n");
				CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
				continue;
			}

			encaps_codec->buffer=g_new(CORBA_octet, buf_len);
			if(encaps_codec->buffer==NULL) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: malloc error creating encapsulation buffer\n");
				CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
				goto iiop_error1;
			}

			/* Copying the buffer this way positions the read
			 * pointer in "codec" correctly, and gets the
			 * encapsulation out of our way in case of error.
			 */
			if(CDR_buffer_gets(codec, encaps_codec->buffer, buf_len)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading encapsulation buffer\n");
				goto iiop_error2;
			}

			encaps_codec->buf_len=buf_len;
			encaps_codec->readonly=CORBA_TRUE;

			if(CDR_get_octet(encaps_codec, &byteorder)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading encapsulation byteorder\n");
				goto iiop_error2;
			}
			
			if(byteorder) {
				encaps_codec->data_endian=LittleEndian;
			} else {
				encaps_codec->data_endian=BigEndian;
			}

			if(CDR_get_octet(encaps_codec, &iiop_major)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading encapsulation iiop_major\n");
				goto iiop_error2;
			}

			if(iiop_major!=1) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: encapsulation iiop_major is incompatible version (%d)\n", iiop_major);
				goto iiop_error2;
			}

			if(CDR_get_octet(encaps_codec, &iiop_minor)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading encapsulation iiop_minor\n");
				goto iiop_error2;
			}

			if(iiop_minor>1) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: encapsulation iiop_minor is incompatible version (%d)\n", iiop_minor);
				goto iiop_error2;
			}

			if(CDR_get_string(encaps_codec, &host)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading iiop host\n");
				goto iiop_error2;
			}
			if(CDR_get_ushort(encaps_codec, &port)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading iiop port\n");
				goto iiop_error3;
			}

			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: host: [%s] port: %d (IIOP v%d.%d)\n", host, port, iiop_major, iiop_minor);

			if(CDR_get_seq_begin(encaps_codec, &key_len)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading iiop key length\n");
				goto iiop_error3;
			}
			object_key=g_new(CORBA_octet, key_len);
			if(object_key==NULL) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: malloc error creating iiop object key\n");
				CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
				goto iiop_error3;
			}

			if(CDR_buffer_gets(encaps_codec, object_key, key_len)==CORBA_FALSE) {
				ORBit_Trace(TraceMod_ORB, TraceLevel_Notice, "CORBA_ORB_string_to_object: error reading iiop object key\n");
				goto iiop_error4;
			}
			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: object_key: [%s]\n", object_key);

			if(iiop_minor==1) {
				/* read tagged component sequence */
				g_assert(!"IIOP v1.1 not yet implemented");
			}

			/* Now turn this data into an object */

			new = ORBit_CORBA_Object_new(ev);
			new->orb=orb;
			if(ORBit_parse_inet(new, host, port)) {
				CORBA_exception_set_system(ev, ex_CORBA_COMM_FAILURE, CORBA_COMPLETED_NO);
				goto iiop_error4;
			}

			CDR_codec_free(encaps_codec);
			CDR_codec_free(codec);
			new->obj_id = oid;
			ORBit_set_object_key(new, object_key, key_len);
			new = CORBA_Object_duplicate(new, ev);
			g_free(host);
			return new;

iiop_error4:
			g_free(object_key);
iiop_error3:
			g_free(host);
iiop_error2:
			g_free(encaps_codec->buffer);
iiop_error1:
			g_free(encaps_codec);
		} else if(tag==IOP_TAG_MULTIPLE_COMPONENTS) {
			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: profile no.%d type TAG_MULTIPLE_COMPONENTS\n", i);
			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: Skipping profile\n");

			ORBit_ORB_skip_encaps(codec);
		} else {
			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: unknown profile no.%d type %d\n", i, tag);
			ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: Skipping profile\n");

			ORBit_ORB_skip_encaps(codec);
		}
	}

	while(CDR_get_octet(codec, &byteorder)!=CORBA_FALSE) {
		ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: trailing data: 0x%x\n", byteorder);
	}

	/* Didn't find a TAG_INTERNET_IOP profile */
	ORBit_Trace(TraceMod_ORB, TraceLevel_Debug, "CORBA_ORB_string_to_object: Didn't find a TAG_INTERNET_IOP profile\n");

	g_free(oid);
	CDR_codec_free(codec);

	return(CORBA_OBJECT_NIL);
}

/* Section 5.6.1 */
CORBA_Status CORBA_ORB_get_default_context(CORBA_ORB orb, CORBA_Context *ctx, CORBA_Environment *ev)
{
	g_return_val_if_fail(ev != NULL, -1);

	if(!ctx) {
		CORBA_exception_set_system(ev,
					   ex_CORBA_BAD_PARAM,
					   CORBA_COMPLETED_NO);
		return -1;
	}

	/* I have no clue what this is supposed to be...
	   v2.2 spec page 5-13 says a CORBA_Context is a list of
	   name-value pairs, the orb.idl file says it's a string,
	   and I have no idea which is correct
	*/

	*ctx = g_strdup("orbit-local-orb");

	return 0;
}

/* Section 4.1.2 */
CORBA_boolean CORBA_ORB_get_service_information(CORBA_ORB orb, CORBA_ServiceType service_type, CORBA_ServiceInformation *service_information, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(CORBA_FALSE);
}

CORBA_Current *CORBA_ORB_get_current(CORBA_ORB orb, CORBA_Environment *ev)
{
	/* XXX check this over */
	return (CORBA_Current *)GET_THREAD_DATA();
}

/* Section 4.5 */
CORBA_ORB_ObjectIdList *CORBA_ORB_list_initial_services(CORBA_ORB orb, CORBA_Environment *ev)
{
	CORBA_ORB_ObjectIdList *list;

	g_assert(orb!=NULL);

	list= g_new(CORBA_ORB_ObjectIdList, 1);
	if(list==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(NULL);
	}

	list->_maximum=2;
	list->_length=2;
	list->_flags=0;
	list->_buffer=g_new(CORBA_ORB_ObjectId, 2);
	if(list->_buffer==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		g_free(list);
		return(NULL);
	}
	CORBA_sequence_set_release((void *)list, CORBA_TRUE);

	/* defined reserved references are:
	 *	RootPOA
	 *	POACurrent
	 *	InterfaceRepository
	 *	NameService
	 *	TradingService
	 *	SecurityCurrent
	 *	TransactionCurrent
	 */
	list->_buffer[0]="ImplementationRepository";
	list->_buffer[1]="RootPOA";

	return(list);
}

/* Section 4.5
 *
 * raises InvalidName
 */
CORBA_Object CORBA_ORB_resolve_initial_references(CORBA_ORB orb, CORBA_ORB_ObjectId identifier, CORBA_Environment *ev)
{
	g_assert(orb!=NULL);
	g_assert(identifier!=NULL);

	if(!strcmp(identifier, "ImplementationRepository")) {
		if(!CORBA_Object_is_nil(orb->imr, ev)) {
			return(CORBA_Object_duplicate(orb->imr, ev));
		}
	} else if(!strcmp(identifier, "InterfaceRepository")) {
		if(!CORBA_Object_is_nil(orb->ir, ev)) {
			return(CORBA_Object_duplicate(orb->ir, ev));
		}
	} else if(!strcmp(identifier, "NameService")) {
		if(!CORBA_Object_is_nil(orb->naming, ev)) {
			return(CORBA_Object_duplicate(orb->naming, ev));
		}
	} else if(!strcmp(identifier, "RootPOA")) {
		if(!CORBA_Object_is_nil(orb->root_poa, ev)) {
			return(CORBA_Object_duplicate(orb->root_poa, ev));
		}
	} else {
		/* throw user exception: InvalidName */
		CORBA_exception_set(ev,CORBA_USER_EXCEPTION,ex_CORBA_ORB_InvalidName,NULL);	
		goto error;
	}
error:
	return(NULL);
}

/* This is a MICO extension
 *
 * raises InvalidName
 */
void CORBA_ORB_set_initial_reference(CORBA_ORB orb, CORBA_ORB_ObjectId identifier, CORBA_Object obj, CORBA_Environment *ev)
{
	g_assert(orb!=NULL);
	g_assert(identifier!=NULL);
	g_assert(obj!=NULL);

	if(!strcmp(identifier, "ImplementationRepository")) {
		if(!CORBA_Object_is_nil(orb->imr, ev)) {
			CORBA_Object_release(orb->imr, ev);
		}
		orb->imr=CORBA_Object_duplicate(obj, ev);
	} else if(!strcmp(identifier, "InterfaceRepository")) {
		if(!CORBA_Object_is_nil(orb->ir, ev)) {
			CORBA_Object_release(orb->ir, ev);
		}
		orb->ir=CORBA_Object_duplicate(obj, ev);
	} else if(!strcmp(identifier, "NameService")) {
		if(!CORBA_Object_is_nil(orb->naming, ev)) {
			CORBA_Object_release(orb->naming, ev);
		}
		orb->naming=CORBA_Object_duplicate(obj, ev);
	} else if(!strcmp(identifier, "RootPOA")) {
		if(!CORBA_Object_is_nil(orb->root_poa, ev)) {
			CORBA_Object_release(orb->root_poa, ev);
		}
		orb->root_poa=CORBA_Object_duplicate(obj, ev);
	} else {
		/* throw user exception: InvalidName */
		CORBA_exception_set(ev,CORBA_USER_EXCEPTION,ex_CORBA_ORB_InvalidName,NULL);	
		goto error;
	}

	return;
error:
	return;
}

/* Section 4.9.1 */
CORBA_boolean CORBA_ORB_work_pending(CORBA_ORB orb, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(CORBA_FALSE);
}

/* Section 4.9.2 */
void CORBA_ORB_perform_work(CORBA_ORB orb, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return;
}

/* Section 4.9.4 */
void CORBA_ORB_shutdown(CORBA_ORB orb, CORBA_boolean wait_for_completion, CORBA_Environment *ev)
{
	/* XXX implement on a per-ORB basis, and also
	   handle whatever wait_for_completion means */

	giop_main_quit();
}

/* Section 4.9.3 */
/* CORBA_ORB_run is in server.c */

/* Section 4.7 */
CORBA_PolicyType CORBA_Policy__get_policy_type(CORBA_Policy obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(0);
}

/* Section 4.7 */
CORBA_Policy *CORBA_Policy_copy(CORBA_Policy obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 4.7
 *
 * raises CORBA_NO_PERMISSION
 */
void CORBA_Policy_destroy(CORBA_Policy obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return;
}

/* Section 4.8.2 */
CORBA_Policy *CORBA_DomainManager_get_domain_policy(CORBA_DomainManager obj, CORBA_PolicyType policy_type, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 4.8.2 */
void CORBA_ConstructionPolicy_make_domain_manager(CORBA_ConstructionPolicy obj, CORBA_InterfaceDef object_type, CORBA_boolean constr_policy, CORBA_Environment *
ev)
{
	g_assert(!"Not yet implemented");
	return;
}

/* Section 4.2.8 */
CORBA_DomainManagerList *CORBA_Object_get_domain_managers(CORBA_Object obj, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2 */
CORBA_DynAny *CORBA_ORB_create_dyn_any(CORBA_ORB obj, CORBA_any value, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynAny *CORBA_ORB_create_basic_dyn_any(CORBA_ORB obj, CORBA_TypeCode type, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynStruct *CORBA_ORB_create_dyn_struct(CORBA_ORB obj, CORBA_TypeCode type, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynSequence *CORBA_ORB_create_dyn_sequence(CORBA_ORB obj, CORBA_TypeCode type, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynArray *CORBA_ORB_create_dyn_array(CORBA_ORB obj, CORBA_TypeCode type, CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynUnion *CORBA_ORB_create_dyn_union(CORBA_ORB obj,
					   CORBA_TypeCode type,
					   CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynEnum *CORBA_ORB_create_dyn_enum(CORBA_ORB obj,
					 CORBA_TypeCode type,
					 CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

/* Section 7.2.2
 *
 * raises: InconsistentTypeCode
 */
CORBA_DynFixed *CORBA_ORB_create_dyn_fixed(CORBA_ORB obj,
					   CORBA_TypeCode type,
					   CORBA_Environment *ev)
{
	g_assert(!"Not yet implemented");
	return(NULL);
}

CORBA_TypeCode CORBA_ORB_create_struct_tc(CORBA_ORB obj, CORBA_RepositoryId id, CORBA_Identifier name, CORBA_StructMemberSeq members, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;
	int i;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc == NULL)
	  goto tc_alloc_failed;

	tc->subtypes=g_new0(CORBA_TypeCode, members._length);
	if(tc->subtypes == NULL)
	  goto subtypes_alloc_failed;

	tc->subnames=g_new0(char *, members._length);
	if(tc->subnames == NULL)
	  goto subnames_alloc_failed;

	tc->kind=CORBA_tk_struct;
	tc->name=g_strdup(name);
	tc->repo_id=g_strdup(id);
	tc->sub_parts=members._length;
	tc->length=members._length;

	for(i=0;i<members._length;i++) {
		CORBA_StructMember *mem=(CORBA_StructMember *)&(members._buffer[i]);

		g_assert(&(mem->type)!=NULL);

		tc->subtypes[i] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
		memcpy(tc->subtypes[i], mem->type, (size_t)sizeof(struct CORBA_TypeCode_struct));
		tc->subnames[i]=g_strdup(mem->name);
	}

	return(tc);

 subnames_alloc_failed:
	g_free(tc->subtypes);
 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return NULL;
}

CORBA_TypeCode 
CORBA_ORB_create_union_tc(CORBA_ORB obj, CORBA_RepositoryId id,
			  CORBA_Identifier name,
			  CORBA_TypeCode discriminator_type,
			  CORBA_UnionMemberSeq members,
			  CORBA_Environment *ev)
{
	CORBA_TypeCode tc;
	int i;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);

	if(tc == NULL)
	  goto tc_alloc_failed;

	tc->discriminator = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);

	if(tc->discriminator == NULL)
	  goto discriminator_alloc_failed;

	memcpy(tc->discriminator, discriminator_type, (size_t)sizeof(CORBA_TypeCode));

	tc->subtypes=g_new0(CORBA_TypeCode, members._length);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->subnames=g_new0(char *, members._length);
	if(tc->subnames==NULL)
	  goto subnames_alloc_failed;

	tc->sublabels=g_new0(CORBA_any, members._length);
	if(tc->sublabels == NULL)
	  goto sublabels_alloc_failed;

	tc->kind=CORBA_tk_union;
	tc->name=g_strdup(name);
	tc->repo_id=g_strdup(id);
	tc->sub_parts=members._length;
	tc->length=members._length;
	tc->default_index=-1;

	for(i=0;i<members._length;i++) {
		CORBA_UnionMember *mem=(CORBA_UnionMember *)&(members._buffer[i]);

		g_assert(&(mem->label)!=NULL);
		memcpy(&(tc->sublabels[i]), &(mem->label), (size_t)sizeof(CORBA_any));
		g_assert(&(mem->type)!=NULL);
		tc->subtypes[i] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
		memcpy(tc->subtypes[i], mem->type, (size_t)sizeof(struct CORBA_TypeCode_struct));
		tc->subnames[i]=g_strdup(mem->name);

		if(mem->label._type->kind==CORBA_tk_octet) {
			tc->default_index=i;
		}
	}

	return(tc);

sublabels_alloc_failed:
	g_free(tc->sublabels);
subnames_alloc_failed:
	g_free(tc->subtypes);
subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc->discriminator);
discriminator_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return NULL;
}

CORBA_TypeCode CORBA_ORB_create_enum_tc(CORBA_ORB obj, CORBA_RepositoryId id, CORBA_Identifier name, CORBA_EnumMemberSeq members, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;
	int i;

	tc = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc == NULL)
	  goto tc_alloc_failed;

	tc->subnames=g_new0(char *, members._length);
	if(tc->subnames==NULL)
	  goto subnames_alloc_failed;

	tc->kind = CORBA_tk_enum;
	tc->name = g_strdup(name);
	tc->repo_id = g_strdup(id);
	tc->sub_parts = members._length;
	tc->length = members._length;

	for(i=0;i<members._length;i++) {
		tc->subnames[i]=g_strdup(members._buffer[i]);
	}

	return(tc);

 subnames_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return(NULL);
}

CORBA_TypeCode CORBA_ORB_create_alias_tc(CORBA_ORB obj, CORBA_RepositoryId id, CORBA_Identifier name, CORBA_TypeCode original_type, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL)
	  goto tc_alloc_failed;
	
	/* Can't use chunks here, because it's sometimes an array. Doh! */
	tc->subtypes=g_new0(CORBA_TypeCode, 1);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->kind=CORBA_tk_alias;
	tc->name=g_strdup(name);
	tc->repo_id=g_strdup(id);
	tc->sub_parts=1;
	tc->length=1;

	tc->subtypes[0] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	memcpy(tc->subtypes[0], original_type, (size_t)sizeof(struct CORBA_TypeCode_struct));

	return(tc);
 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return NULL;
}

CORBA_TypeCode CORBA_ORB_create_exception_tc(CORBA_ORB obj, CORBA_RepositoryId id, CORBA_Identifier name, CORBA_StructMemberSeq members, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;
	int i;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL)
	  goto tc_alloc_failed;

	tc->subtypes=g_new0(CORBA_TypeCode, members._length);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->subnames=g_new0(char *, members._length);
	if(tc->subnames==NULL)
	  goto subnames_alloc_failed;

	tc->kind=CORBA_tk_except;
	tc->name=g_strdup(name);
	tc->repo_id=g_strdup(id);
	tc->sub_parts=members._length;
	tc->length=members._length;

	for(i=0;i<members._length;i++) {
		CORBA_StructMember *mem=(CORBA_StructMember *)&(members._buffer[i]);

		g_assert(mem->type != NULL);
		tc->subtypes[i] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
		memcpy(tc->subtypes[i], mem->type, (size_t)sizeof(struct CORBA_TypeCode_struct));
		tc->subnames[i]=g_strdup(mem->name);
	}

	return(tc);

 subnames_alloc_failed:
	g_free(tc->subtypes);
 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return(NULL);
}

CORBA_TypeCode CORBA_ORB_create_interface_tc(CORBA_ORB obj, CORBA_RepositoryId id, CORBA_Identifier name, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY,
					   CORBA_COMPLETED_NO);
		return(NULL);
	}

	tc->kind=CORBA_tk_objref;
	tc->name=g_strdup(name);
	tc->repo_id=g_strdup(id);

	return(tc);
}

CORBA_TypeCode CORBA_ORB_create_string_tc(CORBA_ORB obj, CORBA_unsigned_long bound, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(NULL);
	}

	tc->kind=CORBA_tk_string;
	tc->length=bound;

	return(tc);
}

CORBA_TypeCode CORBA_ORB_create_wstring_tc(CORBA_ORB obj, CORBA_unsigned_long bound, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(NULL);
	}

	tc->kind=CORBA_tk_wstring;
	tc->length=bound;

	return(tc);
}

CORBA_TypeCode CORBA_ORB_create_fixed_tc(CORBA_ORB obj, CORBA_unsigned_short digits, CORBA_short scale, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL) {
		CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
		return(NULL);
	}

	tc->kind=CORBA_tk_fixed;
	tc->digits=digits;
	tc->scale=scale;

	return(tc);
}

CORBA_TypeCode CORBA_ORB_create_sequence_tc(CORBA_ORB obj, CORBA_unsigned_long bound, CORBA_TypeCode element_type, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL)
	  goto tc_alloc_failed;

	/* Can't use chunks here because we can only be sure of getting
	   one consecutive chunk from glib */
	tc->subtypes=g_new0(CORBA_TypeCode, 1);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->kind=CORBA_tk_sequence;
	tc->sub_parts=1;
	tc->length=bound;

	tc->subtypes[0] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	memcpy(tc->subtypes[0], element_type,
	       (size_t)sizeof(struct CORBA_TypeCode_struct));

	return(tc);

 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return(NULL);
}

CORBA_TypeCode CORBA_ORB_create_recursive_sequence_tc(CORBA_ORB obj, CORBA_unsigned_long bound, CORBA_unsigned_long offset, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL)
	  goto tc_alloc_failed;

	tc->subtypes=g_new0(CORBA_TypeCode, 1);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->kind=CORBA_tk_sequence;
	tc->sub_parts=1;
	tc->length=bound;

	tc->subtypes[0] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	tc->subtypes[0]->kind=CORBA_tk_recursive;
	tc->subtypes[0]->recurse_depth=offset;

	return(tc);

 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return(NULL);
}

CORBA_TypeCode CORBA_ORB_create_array_tc(CORBA_ORB obj, CORBA_unsigned_long length, CORBA_TypeCode element_type, CORBA_Environment *ev)
{
	CORBA_TypeCode tc;

	tc=ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	if(tc==NULL)
	  goto tc_alloc_failed;

	tc->subtypes=g_new0(CORBA_TypeCode, 1);
	if(tc->subtypes==NULL)
	  goto subtypes_alloc_failed;

	tc->kind=CORBA_tk_array;
	tc->sub_parts=1;
	tc->length=length;

	tc->subtypes[0] = ORBIT_CHUNK_ALLOC(CORBA_TypeCode);
	memcpy(tc->subtypes[0], element_type, (size_t)sizeof(CORBA_TypeCode));

	return(tc);

 subtypes_alloc_failed:
	ORBIT_CHUNK_FREE(CORBA_TypeCode, tc);
 tc_alloc_failed:
	CORBA_exception_set_system(ev, ex_CORBA_NO_MEMORY, CORBA_COMPLETED_NO);
	return(NULL);
}
