/* vim: set sw=8: */
/*
 * GUI-file.c:
 *
 * Authors:
 *    Jon K Hellan (hellan@acm.org)
 *    Zbigniew Chyla (cyba@gnome.pl)
 *    Andreas J. Guelzow (aguelzow@taliesin.ca)
 */
#include <gnumeric-config.h>
#include <gnumeric-i18n.h>
#include "gnumeric.h"
#include "gui-file.h"

#include "gui-util.h"
#include "dialogs.h"
#include "sheet.h"
#include "application.h"
#include "io-context.h"
#include "command-context.h"
#include "workbook-control-gui-priv.h"
#include "workbook-view.h"
#include "workbook-priv.h"
#include "gnumeric-gconf.h"
#include "widgets/widget-charmap-selector.h"

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>

typedef struct 
{
	CharmapSelector *charmap_selector;
	GtkWidget	*charmap_label;
	GList *openers;
} file_format_changed_cb_data;

 

static gint
file_opener_description_cmp (gconstpointer a, gconstpointer b)
{
	GnmFileOpener const *fo_a = a, *fo_b = b;

	return g_utf8_collate (gnm_file_opener_get_description (fo_a),
			       gnm_file_opener_get_description (fo_b));
}

static gint
file_saver_description_cmp (gconstpointer a, gconstpointer b)
{
	GnmFileSaver const *fs_a = a, *fs_b = b;

	return g_utf8_collate (gnm_file_saver_get_description (fs_a),
			       gnm_file_saver_get_description (fs_b));
}

static GtkWidget *
make_format_chooser (GList *list, GtkOptionMenu *omenu)
{
	GList *l;
	GtkMenu *menu;

	/* Make format chooser */
	menu = GTK_MENU (gtk_menu_new ());
	for (l = list; l != NULL; l = l->next) {
		GtkWidget *item;
		gchar const *descr;

		if (!l->data)
			descr = _("Automatically detected");
		else if (IS_GNM_FILE_OPENER (l->data))
			descr = gnm_file_opener_get_description (
						GNM_FILE_OPENER (l->data));
		else
			descr = gnm_file_saver_get_description (
				                GNM_FILE_SAVER (l->data));

		item = gtk_menu_item_new_with_label (descr);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu),  item);
	}
	gtk_option_menu_set_menu (omenu, GTK_WIDGET (menu));

	return (GTK_WIDGET (omenu));
}

gboolean
gui_file_read (WorkbookControlGUI *wbcg, char const *file_name,
	       GnmFileOpener const *optional_format, gchar const *optional_encoding)
{
	IOContext *io_context;
	WorkbookView *wbv;

	gnm_cmd_context_set_sensitive (GNM_CMD_CONTEXT (wbcg), FALSE);
	io_context = gnumeric_io_context_new (GNM_CMD_CONTEXT (wbcg));
	wbv = wb_view_new_from_file  (file_name, optional_format, io_context, 
				      optional_encoding);

	if (gnumeric_io_error_occurred (io_context) ||
	    gnumeric_io_warning_occurred (io_context))
		gnumeric_io_error_display (io_context);

	g_object_unref (G_OBJECT (io_context));
	gnm_cmd_context_set_sensitive (GNM_CMD_CONTEXT (wbcg), TRUE);

	if (wbv != NULL) {
		Workbook *tmp_wb = wb_control_workbook (WORKBOOK_CONTROL (wbcg));
		if (workbook_is_pristine (tmp_wb)) {
			g_object_ref (G_OBJECT (wbcg));
			workbook_unref (tmp_wb);
			workbook_control_set_view (WORKBOOK_CONTROL (wbcg), wbv, NULL);
			workbook_control_init_state (WORKBOOK_CONTROL (wbcg));
		} else
			(void) wb_control_wrapper_new (WORKBOOK_CONTROL (wbcg), wbv, NULL, NULL);

		sheet_update (wb_view_cur_sheet	(wbv));
		return TRUE;
	}
	return FALSE;
}

static void
file_format_changed_cb (GtkOptionMenu *omenu_format,
			file_format_changed_cb_data *data)
{
	GnmFileOpener *fo = g_list_nth_data (data->openers,
		gtk_option_menu_get_history (omenu_format));
	gboolean is_sensitive = fo != NULL && gnm_file_opener_is_encoding_dependent (fo);

	charmap_selector_set_sensitive (data->charmap_selector,  is_sensitive);
	gtk_widget_set_sensitive (data->charmap_label,  is_sensitive);
}


static gint
file_opener_find_by_id (GList *openers, char const *id)
{
	GList *l;
	gint i = 0;

	if (id == NULL)
		return 0;
	
	for (l = openers; l != NULL; l = l->next, i++) {
		if (IS_GNM_FILE_OPENER (l->data) &&
		    strcmp (id, gnm_file_opener_get_id(l->data)) == 0)
			return i;
	}

	return 0;
}

/*
 * Suggests automatic file type recognition, but lets the user choose an
 * import filter for selected file.
 */
void
gui_file_open (WorkbookControlGUI *wbcg, char const *default_format)
{
	GList *openers;
	GtkFileSelection *fsel;
	GtkOptionMenu *omenu;
	GtkWidget *format_chooser;
	GtkWidget *charmap_selector;
	GtkWidget *box, *label;
	GnmFileOpener *fo = NULL;
	gchar const *file_name;
	gchar const *encoding;
	file_format_changed_cb_data data;
	gint opener_default;
	char const *title;

	openers = get_file_openers ();

	openers = g_list_copy (openers);
	openers = g_list_sort (openers, file_opener_description_cmp);
	/* NULL represents automatic file type recognition */
	openers = g_list_prepend (openers, NULL);
	opener_default = file_opener_find_by_id (openers, default_format);
	title = (opener_default == 0) ? _("Load file") 
		: gnm_file_opener_get_description 
		(g_list_nth_data(openers, opener_default));

	/* Make format chooser */
	omenu = GTK_OPTION_MENU (gtk_option_menu_new ());
	format_chooser = make_format_chooser (openers, omenu);

	/* Make charmap chooser */
	charmap_selector = charmap_selector_new (CHARMAP_SELECTOR_TO_UTF8);
	data.charmap_selector = CHARMAP_SELECTOR(charmap_selector);
	data.charmap_label = gtk_label_new_with_mnemonic (_("Character _encoding:")),
	
	/* Pack it into file selector */
	fsel = GTK_FILE_SELECTION (gtk_file_selection_new (title));
	gtk_file_selection_hide_fileop_buttons (fsel);
	
	box = gtk_table_new (2, 2, FALSE);
	gtk_table_attach (GTK_TABLE (box),
			  format_chooser,
			  1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_SHRINK, 5, 2);
	label = gtk_label_new_with_mnemonic (_("File _type:")),
	gtk_table_attach (GTK_TABLE (box), label,
			  0, 1, 0, 1, GTK_SHRINK | GTK_FILL, GTK_SHRINK, 5, 2);
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), format_chooser);

        gtk_table_attach (GTK_TABLE (box),
                          charmap_selector,
			  1, 2, 1, 2, GTK_EXPAND | GTK_FILL, GTK_SHRINK, 5, 2);
	gtk_table_attach (GTK_TABLE (box), data.charmap_label,
			  0, 1, 1, 2, GTK_SHRINK | GTK_FILL, GTK_SHRINK, 5, 2);
	gtk_box_pack_start (GTK_BOX (fsel->action_area), box, FALSE, TRUE, 0);
	gtk_label_set_mnemonic_widget (GTK_LABEL (data.charmap_label),
				       charmap_selector);

	data.openers = openers;
	g_signal_connect (G_OBJECT (omenu), "changed",
                          G_CALLBACK (file_format_changed_cb), &data);
	gtk_option_menu_set_history 
		(omenu, opener_default);
	gtk_widget_set_sensitive (GTK_WIDGET (omenu), opener_default == 0);
	file_format_changed_cb (omenu, &data);

	/* Show file selector */
	if (!gnumeric_dialog_file_selection (wbcg, fsel)) {
		g_list_free (openers);
		gtk_object_destroy (GTK_OBJECT (fsel));
		return;
	}

	fo = g_list_nth_data (openers,
			      gtk_option_menu_get_history (omenu));

	file_name = gtk_file_selection_get_filename (fsel);
	encoding = charmap_selector_get_encoding (CHARMAP_SELECTOR(charmap_selector));
	gui_file_read (wbcg, file_name, fo, encoding);

	gtk_object_destroy (GTK_OBJECT (fsel));
	g_list_free (openers);
}

/*
 * Check if it makes sense to try saving.
 * If it's an existing file and writable for us, ask if we want to overwrite.
 * We check for other problems, but if we miss any, the saver will report.
 * So it doesn't have to be bulletproof.
 *
 * FIXME: The message boxes should really be children of the file selector,
 * not the workbook.
 *
 * Note: filename is filesys, not UTF-8 encoded.
 */
static gboolean
can_try_save_to (WorkbookControlGUI *wbcg, char const *name)
{
	gboolean result = TRUE;
	gchar *msg;
	char *filename_utf8 = name
		? g_filename_to_utf8 (name, -1, NULL, NULL, NULL)
		: NULL;

	if (name == NULL || name[0] == '\0') {
		result = FALSE;
	} else if (filename_utf8 == NULL) {
		result = FALSE;
	} else if (name [strlen (name) - 1] == '/' ||
	    g_file_test (name, G_FILE_TEST_IS_DIR)) {
		msg = g_strdup_printf (_("%s\nis a directory name"), filename_utf8);
		gnumeric_notice (wbcg, GTK_MESSAGE_ERROR, msg);
		g_free (msg);
		result = FALSE;
	} else if (access (name, W_OK) != 0 && errno != ENOENT) {
		msg = g_strdup_printf (
		      _("You do not have permission to save to\n%s"),
		      filename_utf8);
		gnumeric_notice (wbcg, GTK_MESSAGE_ERROR, msg);
		g_free (msg);
		result = FALSE;
	} else if (g_file_test (name, G_FILE_TEST_EXISTS)) {
		msg = g_strdup_printf (
		      _("Workbook %s already exists.\n"
		      "Do you want to save over it?"), filename_utf8);
		result = gnumeric_dialog_question_yes_no (
			wbcg, msg, gnm_app_prefs->file_overwrite_default_answer);
		g_free (msg);
	}

	g_free (filename_utf8);
	return result;
}

static gboolean
check_multiple_sheet_support_if_needed (GnmFileSaver *fs,
					WorkbookControlGUI *wbcg,
					WorkbookView *wb_view)
{
	gboolean ret_val = TRUE;

	if (gnm_file_saver_get_save_scope (fs) != FILE_SAVE_WORKBOOK &&
	    gnm_app_prefs->file_ask_single_sheet_save) {
		GList *sheets;
		gchar *msg = _("Selected file format doesn't support "
			       "saving multiple sheets in one file.\n"
			       "If you want to save all sheets, save them "
			       "in separate files or select different file format.\n"
			       "Do you want to save only current sheet?");

		sheets = workbook_sheets (wb_view_workbook (wb_view));
		if (g_list_length (sheets) > 1) {
			ret_val = gnumeric_dialog_question_yes_no (wbcg, msg, TRUE);
		}
		g_list_free (sheets);
	}
	return (ret_val);
}

/*
 * Note: filename is filesys, not UTF-8 encoded.
 */
static gboolean
do_save_as (WorkbookControlGUI *wbcg, WorkbookView *wb_view,
            GnmFileSaver *fs, char const *name)
{
	char *filename = NULL;
	gboolean success = FALSE;

	if (*name == 0 || name[strlen (name) - 1] == '/') {
		gnumeric_notice (wbcg, GTK_MESSAGE_ERROR,
				 _("Please enter a file name,\nnot a directory"));
		return FALSE;
	}

	if (!gnm_file_saver_fix_file_name (fs, name, &filename) &&
		!gnumeric_dialog_question_yes_no (wbcg,
                      _("The given file extension does not match the"
			" chosen file type. Do you want to use this name"
			" anyways?"),
                       TRUE)) {
		g_free (filename);
                return FALSE;
	}
	if (!can_try_save_to (wbcg, filename)) {
		g_free (filename);
		return FALSE;
	}

	wb_view_preferred_size (wb_view, GTK_WIDGET (wbcg->notebook)->allocation.width,
				GTK_WIDGET (wbcg->notebook)->allocation.height);

	success = check_multiple_sheet_support_if_needed (fs, wbcg, wb_view);
	if (!success) {
		g_free (filename);
		return FALSE;
	}

	success = wb_view_save_as (wb_view, fs, filename, GNM_CMD_CONTEXT (wbcg));
	g_free (filename);
	return success;
}

gboolean
gui_file_save_as (WorkbookControlGUI *wbcg, WorkbookView *wb_view)
{
	GList *savers = NULL, *l;
	GtkFileSelection *fsel;
	GtkOptionMenu *omenu;
	GtkWidget *format_chooser;
	GnmFileSaver *fs;
	gboolean success  = FALSE;
	gchar const *wb_file_name;

	g_return_val_if_fail (wbcg != NULL, FALSE);

	for (l = get_file_savers (); l; l = l->next) {
		if ((l->data == NULL) || 
		    (gnm_file_saver_get_save_scope (GNM_FILE_SAVER (l->data)) 
		     != FILE_SAVE_RANGE))
			savers = g_list_prepend (savers, l->data);
	}
	savers = g_list_sort (savers, file_saver_description_cmp);

	/* Make format chooser */
	omenu = GTK_OPTION_MENU (gtk_option_menu_new ());
	format_chooser = make_format_chooser (savers, omenu);

	/* Pack it into file selector */
	fsel = GTK_FILE_SELECTION (gtk_file_selection_new (_("Save workbook as")));
	gtk_box_pack_start (GTK_BOX (fsel->action_area), format_chooser,
	                    FALSE, TRUE, 0);

	/* Set default file saver */
	fs = wbcg->current_saver;
	if (fs == NULL)
		fs = workbook_get_file_saver (wb_view_workbook (wb_view));
	if (fs == NULL || g_list_find (savers, fs) == NULL)
		fs = gnm_file_saver_get_default ();

	gtk_option_menu_set_history (omenu, g_list_index (savers, fs));

	/* Set default file name */
	wb_file_name = workbook_get_filename (wb_view_workbook (wb_view));
	if (wb_file_name != NULL) {
		gchar *tmp_name, *p;

		tmp_name = g_strdup (wb_file_name);
		p = strrchr (tmp_name, '.');
		if (p != NULL) {
			*p = '\0';
		}
		gtk_file_selection_set_filename (fsel, tmp_name);
		g_free (tmp_name);
	}

	/* Show file selector */
	if (gnumeric_dialog_file_selection (wbcg, fsel)) {
		fs = g_list_nth_data (savers, gtk_option_menu_get_history (omenu));
		if (fs != NULL) {
			success = do_save_as (wbcg, wb_view, fs,
			                      gtk_file_selection_get_filename (fsel));
			if (success) {
				wbcg->current_saver = fs;
			}
		} else {
			success = FALSE;
		}
	}

	gtk_widget_destroy (GTK_WIDGET (fsel));
	g_list_free (savers);

	return success;
}

gboolean
gui_file_save (WorkbookControlGUI *wbcg, WorkbookView *wb_view)
{
	Workbook *wb;

	wb_view_preferred_size (wb_view,
	                        GTK_WIDGET (wbcg->notebook)->allocation.width,
	                        GTK_WIDGET (wbcg->notebook)->allocation.height);

	wb = wb_view_workbook (wb_view);
	if (wb->file_format_level < FILE_FL_AUTO)
		return gui_file_save_as (wbcg, wb_view);
	else
		return wb_view_save (wb_view, GNM_CMD_CONTEXT (wbcg));
}
