/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gnome;
import org.gnu.gdk.Color;
import org.gnu.gdk.RgbDither;
import org.gnu.glib.GObject;
import org.gnu.glib.Type;
import org.gnu.gtk.Layout;

/**
 * You can use the GnomeCanvas widget to draw figures, display
 * graphics, position widgets, and more.
 */
public class Canvas extends Layout {
	/**
	 * Constructs a new Canvas.
	 */
	public Canvas() {
		super(gnome_canvas_new());
	}

	/**
	 * Constructs a new Canvas object form a native resource.
	 * @param handle The handle to the native resource.
	 */
	public Canvas(int handle) {
	    super(handle);
	}
	
	/**
	 * Takes a string specification for a color and allocates it into the
	 * specified GdkColor.
	 * @param spec The color to allocate.
	 */
	public Color getColor(String spec) {
		int[] ch = new int[1];
		if (gnome_canvas_get_color(getHandle(), spec, ch)) {
			return new Color(ch[0]);
		} else {
			return null;
		}
	}

	/**
	 * Allocates a color from the RGB value passed into this function.
	 * @param rgba The RGB value.
	 */
	public long getColorPixel(int rgba) {
		return gnome_canvas_get_color_pixel(getHandle(), rgba);
	}

	/**
	 * The item containing the mouse pointer, or NULL if none.
	 */
	public CanvasItem getCurrentCanvasItem() {
		int cih = getCurrentItem(getHandle());
		if (cih > 0) {
			return buildCanvasItem(cih);
		} else {
			return null;
		}
	}

	/**
	 * Returns the dither mode of an antialiased canvas.  Only applicable to
	 * antialiased canvases - ignored by non-antialiased convases.
	 */
	public RgbDither getDither() {
		int dh = gnome_canvas_get_dither(getHandle());
		return RgbDither.intern(dh);
	}

	/**
	 * The currently focused item, or NULL if none.
	 */
	public CanvasItem getFocusedCanvasItem() {
		int cih = getFocusedItem(getHandle());
		if (cih > 0) {
			return buildCanvasItem(cih);
		} else {
			return null;
		}
	}

	/**
	 * The item that holds a pointer grab, or NULL if none.
	 */
	public CanvasItem getGrabbedCanvasItem() {
		int cih = getGrabbedItem(getHandle());
		if (cih > 0) {
			return buildCanvasItem(cih);
		} else {
			return null;
		}
	}

	/**
	 * Returns the item that is at the specified position in world coordinates,
	 * or NULL if no item is there.
	 * @param x The horizontal coordinate.
	 * @param y The vertical coordinate.
	 */
	public CanvasItem getItemAt(double x, double y) {
		int cih = gnome_canvas_get_item_at(getHandle(), x, y);
		if (cih > 0) {
			return buildCanvasItem(cih);
		} else {
			return null;
		}
	}

	/**
	 * Scaling factor to be used for display.
	 */
	public double getPixelsPerUnit() {
		return (getPixelsPerUnit(getHandle()));
	}

	/**
	 * Returns the root canvas item group of the canvas
	 */
	public CanvasGroup getRoot() {
		int hndl = gnome_canvas_root(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (CanvasGroup)obj;
		return new CanvasGroup(hndl);
	}

	/**
	 * Returns the horizontal scroll offset of the canvas in canvas pixel
	 * coordinates.
	 * @return The scroll offset x coordinate.
	 */
	public double getScrollOffsetX() {
		int[] x = new int[1];
		int[] y = new int[1];
		gnome_canvas_get_scroll_offsets(getHandle(), x, y);
		return x[0];
	}
	/**
	 * Returns the vertical scroll offset of the canvas in canvas pixel
	 * coordinates.
	 * @return The scroll offset y coordinate.
	 */
	public double getScrollOffsetY() {
		int[] x = new int[1];
		int[] y = new int[1];
		gnome_canvas_get_scroll_offsets(getHandle(), x, y);
		return y[0];
	}

	/**
	 * The x1 coordinate of the scrolling region.
	 */
	public double getScrollX1() {
		return (getScrollX1(getHandle()));
	}

	/**
	 * The x2 coordinate of the scrolling region.
	 */
	public double getScrollX2() {
		return (getScrollX2(getHandle()));
	}

	/**
	 * The y1 coordinate of the scrolling region.
	 */
	public double getScrollY1() {
		return (getScrollY1(getHandle()));
	}

	/**
	 * The y2 coordinate of the scrolling region.
	 */
	public double getScrollY2() {
		return (getScrollY2(getHandle()));
	}

	/**
	 * Scrolls the canvas to the specified offsets, given in canvas pixel
	 * coordinates.
	 * @param cx Horizontal coordinate.
	 * @param cy Vertical coordinate.
	 */
	public void scrollTo(int cx, int cy) {
		gnome_canvas_scroll_to(getHandle(), cx, cy);
	}

	/**
	 * Controls the dithering used when the canvas renders.  Only applicable to
	 * antialiased canvases - ignored by non-antialiased canvases.
	 */
	public void setDither(RgbDither dither) {
		gnome_canvas_set_dither(getHandle(), dither.getValue());
	}

	/**
	 * Sets the number of pixels that correspond to one unit in world
	 * coordinates.
	 * @param n The number of pixels
	 */
	public void setPixelsPerUnit(double n) {
		gnome_canvas_set_pixels_per_unit(getHandle(), n);
	}

	/**
	 * Convenience method to set all 4 coordinates of the scroll region at
	 * once.
	 * @param x1 First horizontal coordinate.
	 * @param y1 First vertical coordinate.
	 * @param x2 Second horizontal coordinate.
	 * @param y2 Second vertical coordinate.
	 */
	public void setScrollRegion(double x1, double y1, double x2, double y2) {
		gnome_canvas_set_scroll_region(getHandle(), x1, y1, x2, y2);
	}

	/**
	 * Set the scroll region's X1 coordinate.
	 * @param x1
	 */
	public void setScrollX1(double x1) {
		gnome_canvas_set_scroll_region(getHandle(), x1, getScrollY1(), getScrollX2(), getScrollY2());
	}

	/**
	 * Set the scroll region's Y1 coordinate.
	 * @param y1
	 */
	public void setScrollY1(double y1) {
		gnome_canvas_set_scroll_region(getHandle(), getScrollX1(), y1, getScrollX2(), getScrollY2());
	}

	/**
	 * Set the scroll region's X2 coordinate.
	 * @param x2
	 */
	public void setScrollX2(double x2) {
		gnome_canvas_set_scroll_region(getHandle(), getScrollX1(), getScrollY1(), x2, getScrollY2());
	}

	/**
	 * Set the scroll region's Y2 coordinate.
	 * @param y2
	 */
	public void setScrollY2(double y2) {
		gnome_canvas_set_scroll_region(getHandle(), getScrollX1(), getScrollY1(), getScrollX2(), y2);
	}

	/**
	 * Requests that the canvas be repainted immediately instead of during the
	 * idle loop.
	 */
	public void update() {
		gnome_canvas_update_now(getHandle());
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gnome_canvas_get_type());
	}
	
	private CanvasItem buildCanvasItem(int hndl) {
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (CanvasItem)obj;
		return new CanvasItem(hndl);
	}

	/****************************************
	 * BEGINNING OF GENERATED CODE
	 ****************************************/
	native static final protected double getScrollX1(int cptr);
	native static final protected double getScrollY1(int cptr);
	native static final protected double getScrollX2(int cptr);
	native static final protected double getScrollY2(int cptr);
	native static final protected double getPixelsPerUnit(int cptr);
	native static final protected int getCurrentItem(int cptr);
	native static final protected int getFocusedItem(int cptr);
	native static final protected int getGrabbedItem(int cptr);
	native static final protected int gnome_canvas_get_type();
	native static final protected int gnome_canvas_new();
	native static final protected int gnome_canvas_root(int canvas);
	native static final protected void gnome_canvas_set_scroll_region(int canvas, double x1, double y1, double x2, double y2);
	native static final protected void gnome_canvas_get_scroll_region(int canvas, double[] x1, double[] y1, double[] x2, double[] y2);
	native static final protected void gnome_canvas_set_pixels_per_unit(int canvas, double n);
	native static final protected void gnome_canvas_scroll_to(int canvas, int cx, int cy);
	native static final protected void gnome_canvas_get_scroll_offsets(int canvas, int[] cx, int[] cy);
	native static final protected void gnome_canvas_update_now(int canvas);
	native static final protected int gnome_canvas_get_item_at(int canvas, double x, double y);
	native static final protected boolean gnome_canvas_get_color(int canvas, String spec, int[] color);
	native static final protected long gnome_canvas_get_color_pixel(int canvas, int rgba);
	native static final protected void gnome_canvas_set_stipple_origin(int canvas, int gc);
	native static final protected void gnome_canvas_set_dither(int canvas, int dither);
	native static final protected int gnome_canvas_get_dither(int canvas);
	/****************************************
	 * END OF GENERATED CODE
	 ****************************************/
}
