#ifndef _FW_INTERFACE_H
#define _FW_INTERFACE_H
/*
 * Author: Paul.Russell@rustcorp.com.au
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This some useful interface code to the NETLINK interface for
 * ipchains 1.3.  Using this library you can do more comprehensive
 * logging, dynamic firewalling, etc.
 *
 * Your program must be running as root. */
#include <ipfwc_kernel_headers.h>

struct iphdr;

/* Must be called before any other routines. */
extern int fw_init_listen(int need_injection, int silent);
/* Returns 0 for failure. 
 * If need_injection, callback functions may return packet to be reinjected
 * through ethertap(requires ethertap device).
 * If !silent, prints any error messages to stderr. */

/* Does a cleanup; call before terminating. */
extern void fw_close_listen(int silent);

/* This is the type of function which gets called when a matching
 * packet occurs. */
typedef const struct iphdr *fw_readpktfn(__u32 rulenum, 
                                         const char *interface,
                                         const ip_chainlabel chainname,
                                         __u16 packetsize,
                                         struct iphdr *hdr);
/* rulenum: the rule number returned by register_interest.
   interface: the NUL-terminated name of the interface (incoming or outgoing).
   chain: the NUL-terminated name of the chain supplied in register_interest.
   packetsize: the number of valid bytes pointed to by hdr.
   hdr: the pointer to a static region containing the packet. 
   If non-NULL is returned, the packet is reinjected (completely RAW).
   Beware creating firewall loops. */

/* Register interest in a certain type of packet; returns 0 for failure,
   or a unique rule number. */
extern __u32 fw_register_interest(const struct ip_fw *packetspecs, 
				  const ip_chainlabel chainname,
				  fw_readpktfn *fn,
				  const char *policy,
				  int is_input,
				  int silent);
/* packetspecs: the filter the rule must pass to call the function.
   (the fw_mark field is overridden bu this function).
   chainname: the name of the chain to attach this filter to.
   fn: the function to call upon match.
   policy: what to do with the packet (NULL => don't effect it, otherwise
           try "ACCEPT", "DENY" or "REJECT".).
   is_input: reinject packet (if any) through ethertap device.
   silent: don't print errors to stderr. */

/* Unregister interest; returns 0 for failure. */
extern int fw_deregister_interest(__u32 rulenum, int silent);

/* Use one of the two interfaces below.  Either simply call
 * "fw_do_listen();" from main(), or for more complex programs,
 * use fw_get_handle(), then fw_do_event(). */

/* Simple interface: runs until something goes wrong. */
extern int fw_do_listen(int silent);

/* More complex interface. */
/* Returns -1 or a filehandle */
extern int fw_get_handle();
/* Returns 0 if there's a problem, otherwise handles a read */
extern int fw_do_event(int silent);
#endif /*_FW_INTERFACE_H*/
