/* Options.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.awt.Button;
import java.awt.Color;
import java.awt.Event;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Label;
import java.awt.Panel;
import java.awt.TextField;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * @author Mark Boyns
 */
public class Options extends Frame
{
    Properties props;
    Vector hostsAllow;
    Vector hostsDeny;
    
    TextField httpProxyHost;
    TextField httpProxyPort;
    TextField hostsAllowList;
    TextField hostsDenyList;
    
    public Options ()
    {
	super ("Options");

	setBackground (Color.lightGray);
	setFont (new Font ("Helvetica", Font.BOLD, 12));

	/* Load the properties with some reasonable defaults. */
	props = new Properties ();
	putString ("muffin.port", "51966");
	putString ("muffin.httpProxyHost", "proxy");
	putString ("muffin.httpProxyPort", "3128");

	/* By default only allow the localhost access */
	try
	{
	    InetAddress localhost = InetAddress.getLocalHost ();
	    putString ("muffin.hostsAllow", localhost.getHostAddress ());
	}
	/* If this fails allow access to everyone, oh well */
	catch (Exception e)
	{
	    putString ("muffin.hostsAllow", "ALL");
	}
	
	putString ("muffin.hostsDeny", "ALL");

	load ();

	updateHostsAllow ();
	updateHostsDeny ();
	
	Panel panel = new Panel ();
	GridBagLayout layout = new GridBagLayout ();
	panel.setLayout (layout);

	Label l;
	TextField t;
	Button b;
	GridBagConstraints c;

	l = new Label ("HTTP Proxy:", Label.RIGHT);
	c = new GridBagConstraints ();
	layout.setConstraints (l, c);
	panel.add (l);

	httpProxyHost = new TextField (20);
	httpProxyHost.setText (getString ("muffin.httpProxyHost"));
	c = new GridBagConstraints ();
	layout.setConstraints (httpProxyHost, c);
	panel.add (httpProxyHost);

	l = new Label ("Port:", Label.RIGHT);
	c = new GridBagConstraints ();
	layout.setConstraints (l, c);
	panel.add (l);
	
	httpProxyPort = new TextField (10);
	httpProxyPort.setText (getString ("muffin.httpProxyPort"));
	c = new GridBagConstraints ();
	c.gridwidth = GridBagConstraints.REMAINDER;
	layout.setConstraints (httpProxyPort, c);
	panel.add (httpProxyPort);

	l = new Label ("HostsAllow:", Label.RIGHT);
	c = new GridBagConstraints ();
	layout.setConstraints (l, c);
	panel.add (l);
	
	hostsAllowList = new TextField (50);
	hostsAllowList.setText (getString ("muffin.hostsAllow"));
	c = new GridBagConstraints ();
	c.gridwidth = GridBagConstraints.REMAINDER;
	layout.setConstraints (hostsAllowList, c);
	panel.add (hostsAllowList);

	l = new Label ("HostsDeny:", Label.RIGHT);
	c = new GridBagConstraints ();
	layout.setConstraints (l, c);
	panel.add (l);
	
	hostsDenyList = new TextField (50);
	hostsDenyList.setText (getString ("muffin.hostsDeny"));
	c = new GridBagConstraints ();
	c.gridwidth = GridBagConstraints.REMAINDER;
	layout.setConstraints (hostsDenyList, c);
	panel.add (hostsDenyList);
	
	add ("Center", panel);

	Panel buttonPanel = new Panel ();
	buttonPanel.setLayout (new GridLayout (1, 2));
	buttonPanel.add (new Button ("Apply"));
	buttonPanel.add (new Button ("Save"));
	buttonPanel.add (new Button ("Close"));
	add ("South", buttonPanel);

	pack ();
	resize (preferredSize ());

	/*
	System.out.println (System.getProperty ("java.version"));
	System.out.println (System.getProperty ("java.vendor"));
	System.out.println (System.getProperty ("java.version.url"));
	System.out.println (System.getProperty ("os.name"));
	System.out.println (System.getProperty ("os.arch"));
	System.out.println (System.getProperty ("os.version"));
	System.out.println (System.getProperty ("user.home"));
	System.out.println (System.getProperty ("user.name"));
	System.out.println (System.getProperty ("user.dir"));
	*/
    }
    
    void updateHostsAllow ()
    {
	hostsAllow = new Vector (100);
	StringTokenizer st = new StringTokenizer (props.getProperty ("muffin.hostsAllow"), ", \t");
	while (st.hasMoreTokens ())
	{
	    hostsAllow.addElement (st.nextToken ());
	}
    }

    void updateHostsDeny ()
    {
	hostsDeny = new Vector (100);
	StringTokenizer st = new StringTokenizer (props.getProperty ("muffin.hostsDeny"), ", \t");
	while (st.hasMoreTokens ())
	{
	    hostsDeny.addElement (st.nextToken ());
	}
    }
    
    boolean hostAccess (InetAddress addr)
    {
	String ip = addr.getHostAddress ();
	    
	/* First check if the addr in on the allow list */
	Enumeration e = hostsAllow.elements ();
 	while (e.hasMoreElements ())
 	{
	    String str = (String) e.nextElement ();
	    if (str.equals ("ALL") || ip.startsWith (str))
	    {
		return true;
	    }
	}

	/* Next check if the addr in on the deny list */
	e = hostsDeny.elements ();
 	while (e.hasMoreElements ())
 	{
	    String str = (String) e.nextElement ();
	    if (str.equals ("ALL") || ip.startsWith (str))
	    {
		return false;
	    }
	}

	/* Otherwise allow access */
	return true;
    }

    public void hideshow ()
    {
	if (isShowing ())
	{
	    hide ();
	}
	else
	{
	    show ();
	}
    }

    public void load ()
    {
	File file = new File (System.getProperty ("user.dir"), "properties");
	if (!file.exists ())
	{
	    return;
	}
	try
	{
	    FileInputStream in = new FileInputStream (file);
	    props.load (in);
	}
	catch (Exception e)
	{
	    System.out.println (e);
	}
    }

    public void save ()
    {
	File file = new File (System.getProperty ("user.dir"), "properties");
	try
	{
	    FileOutputStream out = new FileOutputStream (file);
	    props.save (out, null);
	}
	catch (Exception e)
	{
	    System.out.println (e);
	}
    }

    void putString (String key, String value)
    {
	props.put (key, value);
    }

    void putBoolean (String key, boolean value)
    {
	props.put (key, value ? "true" : "false");
    }

    void putInteger (String key, int value)
    {
	props.put (key, Integer.toString (value));
    }

    String getString (String key)
    {
	return props.getProperty (key);
    }

    boolean getBoolean (String key)
    {
	String val = props.getProperty (key);
	return "true".equalsIgnoreCase (val) ? true : false;
    }

    int getInteger (String key)
    {
	int val;
	try
	{
	    val = Integer.parseInt (props.getProperty (key));
	}
	catch (Exception e)
	{
	    val = -1;
	}
	return val;
    }

    boolean useProxy ()
    {
	return ((getString ("muffin.httpProxyHost")).length () > 0
		&& getInteger ("muffin.httpProxyPort") > 0);
    }

    void sync ()
    {
	putString ("muffin.httpProxyHost", httpProxyHost.getText ());
	putString ("muffin.httpProxyPort", httpProxyPort.getText ());
	putString ("muffin.hostsAllow", hostsAllowList.getText ());
	updateHostsAllow ();
	putString ("muffin.hostsDeny", hostsDenyList.getText ());
	updateHostsDeny ();
    }

    public boolean action (Event event, Object arg)
    {
	if (event.target instanceof Button)
	{
	    if ("Apply".equals (arg))
	    {
		sync ();
	    }
	    else if ("Save".equals (arg))
	    {
		sync ();
		save ();
	    }
	    else if ("Close".equals (arg))
	    {
		hide ();
	    }
	    return true;
	}
	return false;
    }
}
