/* Handler.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.net.Socket;
import java.net.URL;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

/**
 * @author Mark Boyns
 */
class Handler extends Thread
{
    Monitor monitor = null;
    Filters filters = null;
    Options options = null;
    Client client = null;
    Socket socket = null;
    Request request = null;
    Reply reply = null;
    HttpRelay http = null;
    int currentLength = 0;
    int contentLength = 0;

    Handler (ThreadGroup t, Runnable r, Monitor m, Filters filters, Options options)
    {
	super (t, r);
	this.monitor = m;
	this.filters = filters;
	this.options = options;
    }

    void doit (Socket s)
    {
	socket = s;
	start ();
    }

    synchronized void close ()
    {
	if (client != null)
	{
	    client.close ();
	}
	if (http != null)
	{
	    http.close ();
	}
    }

    public void run ()
    {
	try
	{
	    client = new Client (socket);
	    request = client.read ();
	    filters.filter (request);
	    monitor.register (this);

	    if (options.useProxy ())
	    {
		http = new Proxy (options.getString ("muffin.httpProxyHost"),
				  options.getInteger ("muffin.httpProxyPort"));
	    }
	    else
	    {
		http = new Http (new URL (request.getURL ()));
	    }
				   
	    http.write (request);
	    monitor.update (this);

	    reply = http.read ();
	    filters.filter (reply);
	    monitor.update (this);

	    /* Send the reply headers */
	    client.write (reply);

	    try 
	    {
		contentLength = Integer.parseInt (reply.getHeaderField ("Content-length"));
	    }
	    catch (Exception e)
	    {
		contentLength = 0;
	    }
	    currentLength = 0;

	    /* Test content filtering */
// 	    if ((reply.getHeaderField ("Content-type")).equals ("text/html"))
// 	    {
//  		PipedInputStream in = new PipedInputStream ();
// 		PipedOutputStream out = new PipedOutputStream ();
// 		Test test = new Test (http.getInputStream (), out);
// 		System.out.println ("Starting filter...");
// 		new Thread (test).start ();
// 		System.out.println ("Starting copy...");
// 		copy (new DataInputStream (in), client.getOutputStream ());
// 	    }
// 	    else
// 	    {
// 		copy (new DataInputStream (http.getInputStream ()), client.getOutputStream ());
// 	    }

	    /* Send the content */
	    copy (new DataInputStream (http.getInputStream ()), client.getOutputStream ());
	}
	catch (Exception e)
	{
	    if (request != null)
	    {
		System.out.println (e + ": " + request.getRequest ());
	    }
	    else
	    {
		System.out.println (e);
	    }
	    e.printStackTrace ();
	}
	
	close ();
	
	monitor.unregister (this);
    }

    public int getTotalBytes ()
    {
	return contentLength > 0 ? contentLength : 0;
    }

    public int getCurrentBytes ()
    {
	return currentLength;
    }

    public String toString ()
    {
	StringBuffer str = new StringBuffer ();
	if (contentLength > 0)
	{
	    str.append ((int)(((float)currentLength/contentLength)*100));
	    str.append ("% of ");
	    str.append (contentLength/1024);
	    str.append ("k");
	}
	else
	{
	    str.append (currentLength/1024);
	    str.append ("k");
	}
	str.append (" ");
	str.append (request.getURL ());
	
	return str.toString ();
    }

    void copy (DataInputStream in,
	       DataOutputStream out) throws IOException
    {
	int n;
	byte buffer[] = new byte[8*1024];

	while ((n = in.read (buffer)) > 0)
	{
	    currentLength += n;
	    out.write (buffer, 0, n);
	    monitor.update (this);
	}
	out.flush ();
    }
}


