/* NoThanksFilter.java */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import muffin.html.*;
import JP.ac.osaka_u.ender.util.regex.*;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Hashtable;

public class NoThanksFilter implements ContentFilter, RequestFilter, ReplyFilter
{
    NoThanks factory;
    Prefs prefs;
    InputStream in = null;
    OutputStream out = null;
    Request request = null;

    public NoThanksFilter (NoThanks factory)
    {
	this.factory = factory;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
    }

    public void filter (Request request) throws FilterException
    {
	String url = request.getURL ();

	/* Check for redirect */
	String location = factory.redirect (url);
	if (location != null)
	{
	    request.setURL (location);
	    return;
	}

	/* Check for killed URL */
	if (factory.isKilled (url))
	{
	    factory.process ("Killed: " + url + "\n");
	    throw new FilterException ("NoThanks URL " + url + " rejected");
	}
    }

    public void filter (Reply reply) throws FilterException
    {
	String content = reply.getHeaderField ("Content-type");
	if (content != null && factory.killContent (content))
	{
	    factory.process ("Killed: " + content + "\n");
	    throw new FilterException ("NoThanks content-type " + content + " rejected");
	}
    }
    
    public boolean needsFiltration (Request request, Reply reply)
    {
	this.request = request;
	String s = reply.getHeaderField ("Content-type");
	return s != null && s.startsWith ("text/html");
    }
    
    public void setInputStream (InputStream in)
    {
	this.in = in;
    }

    public void setOutputStream (OutputStream out)
    {
	this.out = out;
    }

    public void run ()
    {
	try
	{
	    HtmlTokenizer html = new HtmlTokenizer (in);
	    int token;
	    String endTag = null;
	    boolean killingComment = false;

            while ((token = html.getToken ()) != HtmlTokenizer.TT_EOF)
            {
		if (killingComment && token != HtmlTokenizer.TT_COMMENT)
		{
		    continue;
		}

		switch (token)
		{
		case HtmlTokenizer.TT_COMMENT:
		    String s = html.getTokenValue ();
		    killingComment = factory.killComment (s);
		    if (killingComment)
		    {
			factory.process ("Comment: " + s + "\n");
		    }
		    else
		    {
			html.writeToken (out);
		    }
		    break;
		    
		case HtmlTokenizer.TT_TAG:
		    boolean output = true;
		    Tag tag = html.getTag ();

		    if (endTag != null && tag.is (endTag))
		    {
			endTag = null;
			continue;
		    }

		    if (factory.stripTag (tag.name ()))
		    {
			factory.process ("Stripped: " + tag.name () + "\n");
			endTag = factory.stripUntil (tag.name ());
			output = false;
		    }
		    else if (factory.checkTag (tag.name ()))
		    {
			for (int i = 0; i < factory.hyperlist.length; i++)
			{
			    if (tag.has (factory.hyperlist[i]))
			    {
				String link = tag.get (factory.hyperlist[i]);
				if (factory.isKilled (link))
				{
				    factory.process ("Killed: " + link + "\n");
				    output = false;
				    if (tag.is ("a"))
				    {
					endTag = "/a";
				    }
				    break;
				}
			    }
			}
		    }
		    if (output)
		    {
			html.writeTag (tag, out);
		    }
		    break;
		    
		default:
		    if (endTag == null)
		    {
			html.writeToken (out);
		    }
		    break;
		}
	    }
	    
	    out.flush ();
	    out.close ();
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }
}

