/* Message.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.util.Hashtable;
import java.util.Enumeration;
import java.io.PushbackInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * Abstract class to represent message headers.
 *
 * @author Mark Boyns
 */
public abstract class Message
{
    /**
     * Hashtable used to store message headers.
     */
    protected Hashtable headers = new Hashtable (33);

    /**
     *
     */
    protected String statusLine = null;

    protected String readLine (InputStream in) throws IOException
    {
	char buf[] = new char[128];
	int offset = 0;
	int ch;

	for (;;)
	{
	    ch = in.read ();
	    if (ch == -1 || ch == '\n')
	    {
		break;
	    }
	    else if (ch == '\r')
	    {
		int tmpch = in.read ();
		if (tmpch != '\n')
		{
		    if (! (in instanceof PushbackInputStream))
		    {
			in = new PushbackInputStream (in);
		    }
		    ((PushbackInputStream) in).unread (tmpch);
		}
		break;
	    }
	    else
	    {
		if (offset == buf.length)
		{
		    char tmpbuf[] = buf;
		    buf = new char[tmpbuf.length * 2];
		    System.arraycopy (tmpbuf, 0, buf, 0, offset);
		}
		buf[offset++] = (char) ch;
	    }
	}
	return String.copyValueOf (buf, 0, offset);
    }

    /**
     * Read headers and store them in the hashtable.
     */
    protected void readHeaders (InputStream in) throws IOException
    {
	int i;
	String name = null;
	
	for (;;)
	{
	    String s = readLine (in);
	    if (s == null)
	    {
		break;
	    }
	    i = s.indexOf (':');
	    if (i == -1)
	    {
		/* end of header */
		if (s.length () == 0)
		{
		    break;
		}
		/* multi-line headers */
		else if (name != null && (s.startsWith (" ") || s.startsWith ("\t")))
		{
		    String value = (String) headers.get (new Key (name));
		    headers.put (new Key (name), value + "\n" + s);
		}
	    }
	    else
	    {
		name = s.substring (0, i);
		headers.put (new Key (name), s.substring (i+2));
	    }
	}	
    }
    
    /**
     * Return the headers hash.
     */
    public Hashtable getHeaders ()
    {
	return headers;
    }

    /**
     * Return a header value with a case-insensitive lookup.
     */
    public String getHeaderField (String name)
    {
	return (String) headers.get (new Key (name));
    }

    /**
     * Set a header value.
     *
     * @param name header name
     * @param value header value
     */
    public void setHeaderField (String name, String value)
    {
	headers.put (new Key (name), value);
    }

    public void setHeaderField (String name, int value)
    {
	headers.put (new Key (name), new Integer (value).toString ());
    }
    
    /**
     * Remove a header.
     *
     * @param name header to remove
     */
    public void removeHeaderField (String name)
    {
	headers.remove (new Key (name));
    }

    /**
     * Return whether or not a header exists.
     *
     * @param name header name
     */
    public boolean containsHeaderField (String name)
    {
	return headers.containsKey (new Key (name));
    }
    
    /**
     * Return a string represenation of the headers.
     */
    public String toString ()
    {
	StringBuffer buf = new StringBuffer ();
	Key key;
	String value;

	buf.append (statusLine);
	buf.append ("\r\n");
	
	for (Enumeration e = headers.keys (); e.hasMoreElements (); )
	{
	    key = (Key) e.nextElement ();
	    value = (String) headers.get (key);
	    buf.append (key + ": " + value + "\r\n");
	}
	buf.append ("\r\n");

	return buf.toString ();
    }
}
