/* Connection.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.io.IOException;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;

/**
 * Create a TCP connection from a Socket or hostname/port
 * with buffered IO.
 *
 * @see java.net.Socket
 * @author Mark Boyns
 */
class Connection
{
    private final int bufferSize = 8192;
    Socket socket = null;
    InputStream in = null;
    OutputStream out = null;

    /**
     * Create a Connection from a Socket.
     *
     * @param socket a socket
     */
    Connection (Socket socket) throws IOException
    {
	this.socket = socket;
	in = new BufferedInputStream (socket.getInputStream (), bufferSize);
	out = new BufferedOutputStream (socket.getOutputStream (), bufferSize);
    }

    /**
     * Create a Connection from a hostname and port.
     *
     * @param host remote hostname
     * @param port remote port
     */
    Connection (String host, int port) throws IOException
    {
	socket = SocketManager.open (host, port);
	in = new BufferedInputStream (socket.getInputStream (), bufferSize);
	out = new BufferedOutputStream (socket.getOutputStream (), bufferSize);
    }

    Connection ()
    {
    }

    /**
     * Return the input stream.
     */
    InputStream getInputStream ()
    {
	return in;
    }

    /**
     * Return the output stream.
     */
    OutputStream getOutputStream ()
    {
	return out;
    }

    /**
     * Close the connection.
     */
    void close ()
    {
	if (socket != null)
	{
	    try
	    {
		SocketManager.close (socket);
	    }
	    catch (Exception e)
	    {
		System.out.println ("Connection: " + e);
	    }
	}
    }

    public Socket getSocket ()
    {
	return socket;
    }

    public InetAddress getInetAddress ()
    {
	return socket.getInetAddress ();
    }
    
    public int getPort ()
    {
	return socket.getPort ();
    }
}
