
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 *
 * policy - parse policy configuration and report argus data that
 *          does not conform to this policy.   no policy will report
 *          all data.
 *
 *          this is the principle argus client that uses the client
 *          libraries which provide code to read argus archive files,
 *          attach to remote argus servers, parse policy configuration
 *          files, parse argus data streams and calls data specific
 *          subroutines that are provided in this file.
 *
 *          the principle funciton of this client is to print out
 *          argus data, and its intended use is to review argus data.
 *
 * Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */

#define ARGUS_CLIENT

#include <stdlib.h>
#include <unistd.h>

#include <compat.h>

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#ifndef SOLARIS
#include <strings.h>
#endif
#include <string.h>
#include <time.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>

#include <arpa/inet.h>
#include <netdb.h>

#include <policy.h>

#include <cons_out.h>
#include <argus_parse.h>

#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>
#include <nametoaddr.h>
#include <cons_icmp.h>


char *appOptstring = NULL;
extern int major_version, minor_version;
extern struct tm tm_startime, tm_lasttime;

init () {}
void argus_parse_complete () {}
void clientTimeout () {}
parse_arg (argc, argv)
int argc;
char**argv;
{}

void
usage (ptr)
char *ptr;
{
   fprintf (stderr, "usage: %s [-bchmnEIMNORTWX] ", ptr);
   fprintf (stderr, "[-C access-file] ");
   fprintf (stderr, "[-d debug-level]\n");
   fprintf (stderr, "              [-r input_file] ");
   fprintf (stderr, "[-w output_file] ");
   fprintf (stderr, "[-F file] [-P port]\n");
   fprintf (stderr, "              [-S argus_server] expression.\n");
   fprintf (stderr, "options: b - dump packet-matching code.\n");
   fprintf (stderr, "         c - print packet and byte counts.\n");
   fprintf (stderr, "         C - specify Cisco access-file.\n");
   fprintf (stderr, "         h - print help.\n");
   fprintf (stderr, "         m - print MAC addresses.\n");
   fprintf (stderr, "         n - don't convert numbers to names.\n");
   fprintf (stderr, "         r - read Argus input file.\n");
   fprintf (stderr, "         w - write Argus output file.\n");
   fprintf (stderr, "         P - specify remote argus port.\n");
   fprintf (stderr, "         S - specify remote argus host.\n");
   fprintf (stderr, "         E - print ESTABLISHED transactions.\n");
   fprintf (stderr, "         I - print extended ICMP status.\n");
   fprintf (stderr, "         F - use file to define filter expression.\n");
   fprintf (stderr, "         M - print MULTIROUTE transactions.\n");
   fprintf (stderr, "         N - print NORMAL_CLOSED transactions.\n");
   fprintf (stderr, "         O - print transactions with ip OPTIONS.\n");
   fprintf (stderr, "         R - print RESET transactions.\n");
   fprintf (stderr, "         T - print TIMEDOUT transactions.\n");
   fprintf (stderr, "         W - print WINDOW_SHUT transactions.\n");
   fprintf (stderr, "         X - print PKTS_RETRANS transactions.\n");
   exit(1);
}



process_tcp (ptr)
struct writeStruct *ptr;
{
   int i, src_bytes, dst_bytes, src_count, dst_count, ind, vc = 0;
   unsigned int state;
   struct in_addr *srcAddr, *dstAddr;
   struct ether_addr *esrcAddr, *edstAddr;
   char date [128], buf [256], fmtstr[256];
   char *dstString, *srcString, *protoStr;
   char *edstString, *esrcString;
   char *processStr = NULL;
   unsigned short srcPort, dstPort;
   u_long port;

   bzero (fmtstr, 256);

   port = ptr->addr.port;
   if (ptr->status & DETAIL) {
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_lasttime);
   } else {
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_startime);
   }
   state = ptr->status & 
         (SAW_SYN | NORMAL_CLOSE | SAW_SYN_SENT | CON_ESTABLISHED);

   protoStr = (nflag > 1) ? "6  " : "tcp";
   if (mflag) {
      vc = 16;
      strcpy (fmtstr, 
         "%s%17.17s %17.17s     %3s  %15.15s.%-5.5s  -  %15.15s.%-5.5s ");
   } else
      strcpy (fmtstr, "%s    %3s  %15.15s.%-5.5s  -  %15.15s.%-5.5s ");

   if (cflag)
      strcat (fmtstr, "%-5d %-5d  %-7d %-7d");

   if (state) {
      if (state & NORMAL_CLOSE) {
         fmtstr[28 + vc] = '>';
         processStr = process_state_strings[3];
      } else {
         if (ptr->status & DETAIL) {
            if (state & CLOSE_WAITING) {
               fmtstr[26 + vc] = '<'; fmtstr[28 + vc] = '>';
               processStr = process_state_strings[6];
            } else
            if (state & CON_ESTABLISHED) {
               fmtstr[26 + vc] = '<'; fmtstr[28 + vc] = '>';
               processStr = process_state_strings[2];
            } else
            if (state & SAW_SYN_SENT) {
               fmtstr[26 + vc] = '<';
               processStr = process_state_strings[1];
            } else
            if (state & SAW_SYN) {
               fmtstr[28 + vc] = '>';
               processStr = process_state_strings[0];
            }
         } else {
            fmtstr[28 + vc] = '>';
            if (!(state & (SAW_SYN | SAW_SYN_SENT)))
               fmtstr[26 + vc] = '<';
            processStr = process_state_strings[3];
         }
      }
   }
   
   if (ptr->status & RESET) {
      fmtstr[27 + vc] = '|';
      processStr = process_state_strings[5];
   } 

   if (ptr->status & TIMED_OUT) {
      fmtstr[27 + vc] = 'o';
      processStr = process_state_strings[4];
   }

   strcat (fmtstr, processStr);

   if (ptr->status & PKTS_RETRANS) fmtstr[vc + 2] = '*';
   if (ptr->status & BLOCKED) fmtstr[vc + 2] = 'B';
   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 3] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 3] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 3] = 'T'; break;
         case SECURITY:     fmtstr[vc + 3] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 3] = 'R'; break;
         case SATNETID:     fmtstr[vc + 3] = 'N'; break;
         default:           fmtstr[vc + 3] = '?'; break;
      }
   }
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';

   edstAddr = &ptr->etherdst;
   esrcAddr = &ptr->ethersrc;
   srcAddr = &ptr->addr.src; dstAddr = &ptr->addr.dst;
   srcPort = ((unsigned short *) &port)[0];
   dstPort = ((unsigned short *) &port)[1];
   src_count = ptr->src_count;
   src_bytes = ptr->src_bytes;
   dst_count = ptr->dst_count;
   dst_bytes = ptr->dst_bytes;

   if (src_count < 0) src_count = 0;
   if (dst_count < 0) dst_count = 0;

   if (src_bytes < 0) src_bytes = 0;
   if (dst_bytes < 0) dst_bytes = 0;
  
   srcString = ipaddr_string (srcAddr);
   dstString = ipaddr_string (dstAddr);

   esrcString = etheraddr_string ((char *) esrcAddr);
   edstString = etheraddr_string ((char *) edstAddr);

   if ((srcPort == 20) || ((srcPort >= 6000) && (srcPort < 6010))) {
      int swap26 = 0, swap28 = 0;
      if (fmtstr[26 + vc] == '<') swap26 = 1;
      if (fmtstr[28 + vc] == '>') swap28 = 1;
      if (!(swap26 && swap28)) {
         if (swap26) fmtstr[28 + vc] = '>', fmtstr[26 + vc] = ' ';
         if (swap28) fmtstr[26 + vc] = '<', fmtstr[28 + vc] = ' ';
      }
 
      if (mflag)
         sprintf (buf, fmtstr, date,
            edstString, esrcString, protoStr,
            dstString, tcpport_string(dstPort),
            srcString, tcpport_string(srcPort),
            dst_count, src_count, dst_bytes, src_bytes);
      else
         sprintf (buf, fmtstr, date, protoStr,
            dstString, tcpport_string(dstPort),
            srcString, tcpport_string(srcPort),
            dst_count, src_count, dst_bytes, src_bytes);
   } else
      if (mflag)
         sprintf (buf, fmtstr, date,
            esrcString, edstString, protoStr,
            srcString, tcpport_string(srcPort),
            dstString, tcpport_string(dstPort),
            src_count, dst_count, src_bytes, dst_bytes);
      else
         sprintf (buf, fmtstr, date, protoStr,
            srcString, tcpport_string(srcPort),
            dstString, tcpport_string(dstPort),
            src_count, dst_count, src_bytes, dst_bytes);
 
   (void) printf ("%s\n", buf);
}


char *icmptypestr[ICMP_MAXTYPE + 1] = {
   "ECR", "   ", "   ", "UR", "SRC", "RED",
   "   ", "   ", "ECO", "   ", "   ", "TIM",
   "PAR", "TST", "TSR", "IRQ", "IRR", "MAS",
   "MSR",
};    
 
process_icmp (ptr)
struct writeStruct *ptr;
{
   int i, vc = 0, rev = 0;
   struct ip *oip;
   struct udphdr *ouh;
   int hlen;
   struct icmpStruct *icmp;
   char fmtstr[1024], *blankstring = "     ", icmptype[128];
   char *dstString, *srcString, *protoStr;
   char *edstString, *esrcString;
   char buf [256], str[1024], startdate[128];
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;
   extern u_long getnetnumber (), ipaddrtonetmask ();
      
   if (ptr) {
      icmp = (struct icmpStruct *) &ptr->src_count;
      bzero (icmptype, sizeof (icmptype));
      strcpy (icmptype, icmptypestr[icmp->type]);
      esrcString = etheraddr_string ((char *) &ptr->ethersrc);
      edstString = etheraddr_string ((char *) &ptr->etherdst);
      srcString = ipaddr_string (&ptr->addr.src);
      dstString = ipaddr_string (&ptr->addr.dst);
      strftime ((char *) &startdate, 128, "%a %m/%d %T ", &tm_lasttime);
      switch (icmp->type) {
         case ICMP_UNREACH:
            switch (icmp->code) {
               case ICMP_UNREACH_NET:
                  strcat (icmptype, "N");
                  if (Iflag)
                     if (icmp->dstaddr.s_addr) {
                        u_long addr = icmp->dstaddr.s_addr;
                        sprintf (&icmptype[strlen(icmptype)], " %s", 
                                 getnetname (getnetnumber (addr &
                                           ipaddrtonetmask (addr))));
                  }
                  break;
               case ICMP_UNREACH_HOST:
                  strcat (icmptype, "H");
                  if (Iflag)
                     if (icmp->dstaddr.s_addr)
                        sprintf (&icmptype[strlen(icmptype)], " %s", 
                              ipaddr_string (&icmp->dstaddr.s_addr));
                  break;
               case ICMP_UNREACH_PROTOCOL:
                  strcat (icmptype, "O");
                  if (Iflag)
                     if (icmp->data)
                        sprintf (&icmptype[strlen(icmptype)]," %d", icmp->data);
                  break;
               case ICMP_UNREACH_PORT: {
                  strcat (icmptype, "P"); 
                  if (Iflag)
                     if (icmp->data & icmp->gwaddr.s_addr)
                        sprintf (&icmptype[strlen(icmptype)],
                             " proto %d port %d",
                             icmp->data, *((unsigned short *)&icmp->gwaddr));
                     else if (icmp->data)
                        sprintf (&icmptype[strlen(icmptype)]," %d", icmp->data);
                  break;
               }
               case ICMP_UNREACH_NEEDFRAG:
                  strcat (icmptype, "F"); break;
               case ICMP_UNREACH_SRCFAIL:
                  strcat (icmptype, "S"); break;
            }
            break;

         case ICMP_MASKREPLY:
            if (Iflag)
               if (icmp->srcaddr.s_addr)
                  sprintf (&icmptype[strlen(icmptype)], " 0x%08x", 
                                                   icmp->srcaddr.s_addr);
            break;

         case ICMP_REDIRECT:
            switch (icmp->code) {
            case ICMP_REDIRECT_NET:
               (void)sprintf (buf, " %s to net %s",
                         ipaddr_string (&icmp->dstaddr.s_addr),
                         ipaddr_string (&icmp->gwaddr.s_addr));
               break;
            case ICMP_REDIRECT_HOST:
               (void)sprintf (buf, " %s to host %s",
                         ipaddr_string (&icmp->dstaddr.s_addr),
                         ipaddr_string (&icmp->gwaddr.s_addr));
               break;
            case ICMP_REDIRECT_TOSNET:
               (void)sprintf (buf, " tos %s to net %s",
                         ipaddr_string (&icmp->dstaddr.s_addr),
                         ipaddr_string (&icmp->gwaddr.s_addr));
               break;
            case ICMP_REDIRECT_TOSHOST:
               (void)sprintf (buf, " tos %s to host %s",
                         ipaddr_string (&icmp->dstaddr.s_addr),
                         ipaddr_string (&icmp->gwaddr.s_addr));
               break;
            }
            strcat (icmptype, buf);
            break;

         case ICMP_ECHOREPLY:
            rev = 1;
            break;

         case ICMP_PARAMPROB:
         case ICMP_SOURCEQUENCH:
         case ICMP_ECHO:
         case ICMP_TIMXCEED:
         case ICMP_TSTAMP:
         case ICMP_TSTAMPREPLY:
         case ICMP_IREQ:
         case ICMP_IREQREPLY:
         case ICMP_MASKREQ:
            break;
      }

      protoStr = (nflag > 1) ? "1   " : "icmp";
   
      if (mflag) {
         sprintf (fmtstr, "%s",
              "%s%17.17s %17.17s     icmp %15.15s %-5.5s  -> %15.15s %-5.5s");
         if (rev) {fmtstr[42] = '<'; fmtstr[44] = ' ';}
         if (Iflag) strcat (fmtstr, " %s");
         else strcat (fmtstr, " %3.3s");
         if (rev)
            sprintf (str, fmtstr, startdate, esrcString, edstString,
               dstString, blankstring, srcString, blankstring, icmptype);
         else
            sprintf (str, fmtstr, startdate, esrcString, edstString,
               srcString, blankstring, dstString, blankstring, icmptype);
      } else {
         sprintf (fmtstr, "%s",
              "%s    %-4.4s %15.15s %-5.5s  -> %15.15s %-5.5s");
         if (rev) {fmtstr[28] = '<'; fmtstr[30] = ' ';}
         if (Iflag) strcat (fmtstr, " %s");
         else strcat (fmtstr, " %3.3s");

         if (rev)
            sprintf (str, fmtstr, startdate, protoStr,
               dstString, blankstring, srcString, blankstring, icmptype);
         else
            sprintf (str, fmtstr, startdate, protoStr,
               srcString, blankstring, dstString, blankstring, icmptype);
      }

      if (ptr->status & BLOCKED) fmtstr[vc + 2] = 'B';
      if (ptr->status & IPOPTIONMASK) {
         switch (ptr->status & IPOPTIONMASK) {
            case SSRCROUTE:    fmtstr[vc + 3] = 'S'; break;
            case LSRCROUTE:    fmtstr[vc + 3] = 'L'; break;
            case TIMESTAMP:    fmtstr[vc + 3] = 'T'; break;
            case SECURITY:     fmtstr[vc + 3] = '+'; break;
            case RECORDROUTE:  fmtstr[vc + 3] = 'R'; break;
            case SATNETID:     fmtstr[vc + 3] = 'N'; break;
            default:           fmtstr[vc + 3] = '?'; break;
         }
      }
      if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
      if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';

      printf ("%s\n", str);
   }
}


process_udp (ptr)
struct writeStruct *ptr;
{
   int i, vc = 0;
   char *blankstring = "*    ";
   char *dstString, *srcString, *udpString;
   char *edstString, *esrcString, *protoStr;
   char buf[1024], date[128], fmtstr[256];
   u_short srcport, dstport;
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;

   esrcString = etheraddr_string ((char *) &ptr->ethersrc);
   edstString = etheraddr_string ((char *) &ptr->etherdst);
   srcString = ipaddr_string (&ptr->addr.src.s_addr);
   dstString = ipaddr_string (&ptr->addr.dst.s_addr);
   srcport = ((u_short *)&ptr->addr.port)[0];
   dstport = ((u_short *)&ptr->addr.port)[1];

   if (ptr->status & DETAIL)
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_lasttime);
   else
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_startime);

   protoStr = (nflag > 1) ? "17 " : "udp";

   if (mflag) {
      vc = 16;
      strcpy (fmtstr, 
         "%s%17.17s %17.17s     %3s  %15.15s.%-5.5s  -  %15.15s.%-5.5s ");
   } else
      strcpy (fmtstr, "%s    %3s  %15.15s.%-5.5s  -  %15.15s.%-5.5s ");

   if (cflag)
      strcat (fmtstr, "%-5d %-5d  %-7d %-7d");

   if (ptr->status & BLOCKED) fmtstr[vc + 2] = 'B';
   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 3] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 3] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 3] = 'T'; break;
         case SECURITY:     fmtstr[vc + 3] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 3] = 'R'; break;
         case SATNETID:     fmtstr[vc + 3] = 'N'; break;
         default:           fmtstr[vc + 3] = '?'; break;
      }
   }
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';

   fmtstr[26 + vc] = (ptr->dst_count) ? '<' : ' '; 
   fmtstr[28 + vc] = (ptr->src_count) ? '>' : ' '; 
   if ((ptr->status & TIMED_OUT)) strcat (fmtstr, "TIM");
   else if (ptr->status & CON_ESTABLISHED) {
      if ((ptr->src_count == 1) && (ptr->dst_count == 1))
         strcat (fmtstr, "ACC");
      else strcat (fmtstr, "CON");
   } else if (ptr->status & UDP_INIT) strcat (fmtstr, "INT");

   if (mflag)
      sprintf (buf, fmtstr, date,
         esrcString, edstString,
         protoStr,
         srcString, udpport_string (srcport), 
         dstString, udpport_string (dstport), 
         ptr->src_count, ptr->dst_count,
         ptr->src_bytes, ptr->dst_bytes);
   else
      sprintf (buf, fmtstr, date, protoStr,
         srcString, udpport_string (srcport),
         dstString, udpport_string (dstport), 
         ptr->src_count, ptr->dst_count,
         ptr->src_bytes, ptr->dst_bytes);

   printf ("%s\n", buf);
}

#define IPPROTOSTR 99
char *ip_proto_string [IPPROTOSTR] = {"ip", "icmp", "igmp", "ggp",
   "ipnip", "st", "tcp", "ucl", "egp", "igp", "bbn-rcc-mon", "nvp-ii",
   "pup", "argus", "emcon", "xnet", "chaos", "udp", "mux", "dcn-meas",
   "hmp", "prm", "xns-idp", "trunk-1", "trunk-2", "leaf-1", "leaf-2",
   "rdp", "irtp", "iso-tp4", "netblt", "mfe-nsp", "merit-inp", "sep",
   "3pc", "idpr", "xtp", "ddp", "idpr-cmtp", "tp++", "il", "unas",
   "unas", "unas", "unas", "unas", "unas", "unas", "unas", "unas",
   "unas", "unas", "unas", "unas", "unas", "unas", "unas", "unas",
   "unas", "unas", "unas", "any", "cftp", "any", "sat-expak", "kryptolan",
   "rvd", "ippc", "any", "sat-mon", "visa", "ipcv", "cpnx", "cphb", "wsn",
   "pvp", "br-sat-mon", "sun-nd", "wb-mon", "wb-expak", "iso-ip", "vmtp",
   "secure-vmtp", "vines", "ttp", "nsfnet-igp", "dgp", "tcf", "igrp",
   "ospfigp", "sprite-rpc", "larp", "mtp", "ax.25", "ipip", "micp",
   "aes-sp3-d", "etherip", "encap",
};



process_ip (ptr)
struct writeStruct *ptr;
{
   int i, vc = 0;
   char *blankstring = "*    ";
   char *dstString, *srcString, *protoStr ;
   char *edstString, *esrcString;
   char buf[1024], date[128], fmtstr[256], protoStrbuf[16];
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;
   int src_count, dst_count, src_bytes, dst_bytes;
   u_char proto;
   
   srcString = ipaddr_string (&ptr->addr.src.s_addr);
   dstString = ipaddr_string (&ptr->addr.dst.s_addr);
   esrcString = etheraddr_string ((char *) &ptr->ethersrc);
   edstString = etheraddr_string ((char *) &ptr->etherdst);
   src_count = ptr->src_count; dst_count = ptr->dst_count;
   src_bytes = ptr->src_bytes; dst_bytes = ptr->dst_bytes;

   if (ptr->status & DETAIL)
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_lasttime);
   else
      strftime ((char *) &date, 128, "%a %m/%d %T ", &tm_startime);

   if (mflag) {
      vc = 16;
      strcpy (fmtstr, 
            "%s%17.17s %17.17s     %-5.5s%20.20s   -  %20.20s  ");
   } else
      strcpy (fmtstr, "%s    %-5.5s%20.20s   -  %20.20s  ");

   fmtstr[21 + vc] = (dst_count) ? '<' : ' '; 
   fmtstr[23 + vc] = (src_count) ? '>' : ' '; 

   if (cflag)
      strcat (fmtstr, "%-5d %-5d  %-7d %-7d");

   if ((ptr->status & TIMED_OUT)) strcat (fmtstr, "TIM");
   else if (ptr->status & CON_ESTABLISHED) {
      if ((ptr->src_count == 1) && (ptr->dst_count == 1))
         strcat (fmtstr, "ACC");
      else strcat (fmtstr, "CON");
   } else if (ptr->status & UDP_INIT) strcat (fmtstr, "INT");

   if (ptr->status & BLOCKED) fmtstr[vc + 2] = 'B';
   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 3] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 3] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 3] = 'T'; break;
         case SECURITY:     fmtstr[vc + 3] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 3] = 'R'; break;
         case SATNETID:     fmtstr[vc + 3] = 'N'; break;
         default:           fmtstr[vc + 3] = '?'; break;
      }
   }
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';

   proto = ((unsigned char *)&ptr->addr.port)[3];
   sprintf (protoStrbuf, "%u", proto);
   protoStr = (nflag > 1) ? protoStrbuf :
             proto >= IPPROTOSTR ? "unas" : ip_proto_string[proto];

   if (mflag && cflag)
      sprintf (buf, fmtstr, date, 
            esrcString, edstString, protoStr,
            srcString, dstString, 
            src_count, dst_count,
            src_bytes, dst_bytes);
   else
   if (cflag)
      sprintf (buf, fmtstr, date, protoStr,
            srcString, dstString, 
            src_count, dst_count,
            src_bytes, dst_bytes);
   else
   if (mflag)
      sprintf (buf, fmtstr, date, 
            esrcString, edstString,  protoStr,
            srcString, dstString);
   else
      sprintf (buf, fmtstr, date, protoStr,
            srcString, dstString);

   printf ("%s\n", buf);
}

