
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * argus_parse - parse argus output.
 *       this module performs all the argus(1) related connection parsing,
 *       selects datum from a set of criteria, and then calls specific
 *       protocol dependant routines, depending on the selected datum.
 *       at the end of processing, argus_parse calls an application
 *       specific finish routine, argus_parse_complete(), and when
 *       connected to a remote data source, it supplies a periodic
 *       timeout routine;
 *
 *       this module defines all things, except:
 *
 *   (void) usage ((char *) argv[0]);
 *                    this routine should print the standard usage message
 *                    for the specific application.
 *
 *          init ();  this is the application specific init
 *                    routine, which is called after all parsing
 *                    initialization is done, prior to reading the
 *                    first monitor(1) datum.
 *
 *   (void) clientTimeout ();
 *                    this routine is called every second, when
 *                    argus_parse is connected to a remote data source.
 *
 *          process_tcp ((struct writeStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_udp ((struct writeStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_icmp ((struct writeStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_ip ((struct writeStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *   (void) argus_parse_complete ();
 *                    this routine will be called after all the
 *                    monitor data has been read.
 *
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */

#define MONITOR_PARSE


#include <stdlib.h>
#include <stdio.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#ifndef SOLARIS
#include <strings.h>
#endif
#include <string.h>
#include <time.h>
#include <errno.h>
 
#include <netinet/in.h>
#include <arpa/inet.h>

#include <compat.h>
#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>

#include <argus_util.h>
#include <policy.h>
#include <argus_parse.h>
#include <argus_client.h>

#include <netinet/ip_icmp.h>

#include <cons_def.h>
#include <cons_icmp.h>
#include <cons_ip.h>
#include <cons_udp.h>


extern init ();
extern void usage ();
extern parse_arg ();
extern process_tcp ();
extern process_icmp ();
extern process_udp ();
extern process_ip ();
extern char *copy_argv ();
extern void argus_parse_complete ();
extern char *appOptstring;


#include <nametoaddr.h>
#include <sys/socket.h>


struct policyStruct *policy = NULL;
char *policyfile =  NULL;
char *wfile =  NULL;

struct FILE_ENTRY *input_file_list = NULL;
struct IP_ENTRY *remote_host_list = NULL;

#include <time.h>

#define MAXSTR  256
#define MAXTIME 100000

u_long localnet, netmask;

argus_parse_init (fd, read_mode)
int fd;
int read_mode;
{
   struct timeval now;
   struct timezone tz;
   char errbuf[256];
   char *device = NULL;

   if (gettimeofday(&now, &tz) < 0)
      error("gettimeofday");
   thiszone = tz.tz_minuteswest * -60;
   if (localtime ((time_t *)&now.tv_sec)->tm_isdst)
      thiszone += 3600;

   if (device = pcap_lookupdev (errbuf))
      pcap_lookupnet(device, &localnet, &netmask, errbuf);

   init_addrtoname (0, localnet, netmask);
}


#include <pcap-int.h>

char *progname = NULL;
char cmdline[256]; 

int portnum = 0;

main (argc, argv)
int argc;
char **argv;
{
   register int i, op;
   register char *file = NULL;
   struct bpf_program bpfcode;
   struct FILE_ENTRY *ptr;
   char *cmdbuf = NULL, *infile = NULL, *read_infile ();
   pcap_t p;
   extern char *optarg;
   extern int optind, opterr;

   opterr = 0;
   for (i = 0, *cmdline = '\0'; i < argc; i++) {
      strcat (cmdline, argv[i]);
      strcat (cmdline, " ");
   }
   if (strchr (argv[0], '/'))
      argv[0] = strrchr(argv[0], '/') + 1;
   progname = argv[0];
   while ((op = getopt (argc, argv, "bcC:d:EF:Ir:w:P:hnmMNOPRS:TWX")) != EOF) {
      switch (op) {
         case 'C': policyfile = optarg; break;
         case 'd':  debugflag = atoi (optarg); break;
         case 'P':  portnum = atoi (optarg); break;
         case 'b': ++bflag; break;
         case 'c': ++cflag; break;
         case 't': ++tflag; break;
         case 'E': ++Eflag; break;
         case 'I': ++Iflag; break;
         case 'R': ++Rflag; break;
         case 'T': ++Tflag; break;
         case 'u': ++uflag; break;
         case 'm': ++mflag; break;
         case 'M': ++Mflag; break;
         case 'n': ++nflag; break;
         case 'N': ++Normflag; break;
         case 'W': ++Wflag; break;
         case 'O': ++Oflag; break;
         case 'X': ++Xflag; break;
         case 'F': infile = optarg; break;
         case 'w':  
            if ((wfile = optarg) == NULL)
               if (!strcmp (argv[optind], "-")) {
                  wfile = "-";
               }
            break;
         case 'r': ++rflag; 
            if (optarg == NULL) optarg = "-";
            if (!(new_file_arg (optarg))) {
               fprintf (stderr, "%s: error: file arg %s \n", *argv, optarg);
               exit (1);
            }
            break;
         case 'S': ++Sflag;
            if (!(remote_host_arg (optarg))) {
               fprintf (stderr, "%s: host %s unknown\n", *argv, optarg);
               exit (1);
            }
            break;

         case 'h':
            default:  
               usage (argv[0]);
            /* NOTREACHED */
      }
   }
 
   if (infile)
      cmdbuf = read_infile (infile);
   else
      cmdbuf = copy_argv (&argv[optind]);

   bzero (&bpfcode, sizeof (bpfcode));
   if (policy_compile (&p, &bpfcode, cmdbuf, 1, netmask) < 0) {
      fprintf (stderr, "%s: expression: %s\n", progname, p.errbuf);
      exit (-1);
   }

   if (bflag) {
      bpf_dump(&bpfcode, bflag);
      exit (0);
   }

   if (readPolicy (&policy, policyfile)) {
      if (Sflag) {
         if (read_remote_connection (get_server_socket (remote_host_list),
                        &bpfcode))
            fprintf (stderr, "%s: connection failed\n", *argv);
      } else
         if (ptr = input_file_list) {
            while (ptr) {
               if (strcmp ((char *) ptr->str, "-")) {
                  if (read_connection (fopen (ptr->str, "r"), &bpfcode))
                     perror ("fopen");
               } else
                  read_connection (stdin, &bpfcode);
               ptr = ptr->nxt;
            }
         } else
            read_connection (stdin, &bpfcode);

      argus_parse_complete ();
   }
   exit (0);
}

readPolicy (policy, file)
struct policyStruct **policy;
char *file;
{
   register int retn = 1, linenum = 0, parse;
   register struct policyStruct *pol, *policyLast = NULL;
   register FILE *fd;
   char buffer [1024], *str;

   if (file) {
      if (fd = fopen (file, "r")) {
         while (fgets (buffer, 1024, fd)) {
            linenum++;
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!')) {
               if (pol = (struct policyStruct *) calloc (1, 
                                sizeof (struct policyStruct))) {
                  pol->str = strdup (buffer);
                  if (!(parsePolicy (pol, buffer))) {
                     free (pol->str);
                     free (pol);
                     str = parseerrorstr[parseerror];
                     fprintf (stderr, "%s: parsing error: %s\n",
                                           progname, file);
                     fprintf (stderr, "\tline number %d: %s\n", linenum, str);
                     retn = 0;
                  }
                  if (policyLast)  {
                     policyLast->nxt = pol;
                     pol->prv = policyLast;
                     policyLast = pol;
                  } else
                     *policy = policyLast = pol;
               }
            }
         }
         fclose (fd);
      } else {
         retn = 0;
         fprintf (stderr, "%s: error couldn't open %s\n", progname, file);
      }
   }

   return (retn);
}


#include <string.h>

parsePolicy (policy, buf)
struct policyStruct *policy;
char *buf;
{
   register int retn = 1, error = 0, i;
   register char *ptr;

   if (strncmp (buf, "access", 6)) {
      if (strstr (buf, "ip"))
         if (strstr (buf, "source-route"))
            policy->type = SSRCROUTE;
   } else {
      policy->type = ACCESSLIST;
      for (i = 0; ((i < POLICYFIELDNUM) && !(error)); i++) {
         if ((ptr = strtok (buf, " ")) && (*ptr != '\n')) {
            switch (i) {
               case POLICYSTRING: 
                  if ((strcmp (ptr, POLICY_STRING))) error++; 
                  break;
   
               case POLICYID: 
                  if (!(policy->policyID = atoi (ptr))) error++;
                  break;
   
               case POLICYACTION: 
                  if (!(strcmp (ptr, "permit")))
                     policy->flags |= PERMIT;
                  else if (!(strcmp (ptr, "deny")))
                     policy->flags |= DENY;
                  else error++;
                  break;
   
               case POLICYPROTO: 
                  if (!(strcmp (ptr, "icmp")))
                     policy->flags |= ICMP;
                  else if (!(strcmp (ptr, "tcp")))
                     policy->flags |= TCP;
                  else if (!(strcmp (ptr, "udp")))
                     policy->flags |= UDP;
                  else if (!(strcmp (ptr, "ip")))
                     policy->flags |= IP;
                  else error++;
                  break;
   
               case POLICYDSTADDR: 
                  policy->dst.addr = inet_addr (ptr); break;
               case POLICYDSTMASK: 
                  policy->dst.mask = inet_addr (ptr); break;
               case POLICYSRCADDR: 
                  policy->src.addr = inet_addr (ptr); break;
               case POLICYSRCMASK: 
                  policy->src.mask = inet_addr (ptr); break;
   
               case POLICYPORTACT: 
                  if (!(strcmp (ptr, "eq")))
                     policy->action |= EQ;
                  else if (!(strcmp (ptr, "lt")))
                     policy->action |= LT;
                  else if (!(strcmp (ptr, "gt")))
                     policy->action |= GT;
                  else if (!(strcmp (ptr, "neq")))
                     policy->action |= NEQ;
                  else if (!(strncmp (ptr, "established",11)))
                     policy->action |= EST;
                  else error++;
                  break;
   
               case POLICYPORT: 
                  if (!(policy->port = (u_short) atoi (ptr))) error++;
                  break;
   
               case POLICYNOTIFICATION: 
                  break;
            }
   
            buf = NULL;
         } else
            break;
      }
   }

   if (error) {
      parseerror = i - 1; retn = 0;
   }

   return (retn);
}

check_flags (ptr)
struct writeStruct *ptr;
{
   int retn = 0;

   if (Eflag|Mflag|Normflag|Oflag|Xflag|Rflag|Tflag|Wflag) {
      if (Eflag && !retn) retn = ptr->status & CON_ESTABLISHED;
      if (Mflag && !retn) retn = ptr->status & MULTIADDR;
      if (Normflag && !retn) retn = ptr->status & NORMAL_CLOSE;
      if (Oflag && !retn) retn = ptr->status & IPOPTIONMASK;
      if (Xflag && !retn) retn = ptr->status & PKTS_RETRANS;
      if (Rflag && !retn) retn = ptr->status & RESET;
      if (Tflag && !retn) retn = ptr->status & TIMED_OUT;
      if (Wflag && !retn) retn = ptr->status &
                             (SRC_WINDOW_SHUT|DST_WINDOW_SHUT);
   } else
      retn = 1;

   return (retn);
}


check_policy (ptr, policy)
struct writeStruct *ptr;
struct policyStruct *policy;
{
   int retn = 1, i, policymatch = 0;

   if (policy) {
      while (policy) {
         if (retn = meets_policy_criteria (ptr, policy)) {
            retn = do_notification (ptr, policy);
            policymatch = 1;
            break;
         }
         policy = policy->nxt;
      }
   }

   return (retn);
}


meets_policy_criteria (ptr, policy)
struct writeStruct *ptr;
struct policyStruct *policy;
{
   register int retn = 0, i = 0;
   u_long saddr = 0, daddr = 0, addr = 0;
   u_short sport = 0, dport = 0, port = 0;

   switch (policy->type) {
   case SSRCROUTE:
      if ((ptr->status & SSRCROUTE) || (ptr->status & LSRCROUTE))
         retn++;
      break;

   case ACCESSLIST:
      if ((ptr->status & ICMPPROTO) || (ptr->status & UDPPROTO) ||
         ((ptr->status & TCPPROTO) && !(ptr->status & REVERSE))) {
         saddr = ptr->addr.src.s_addr;
         daddr = ptr->addr.dst.s_addr;
         sport = ((u_short *)&ptr->addr.port)[0];
         dport = ((u_short *)&ptr->addr.port)[1];
         port = dport;
      } else {
         saddr = ptr->addr.dst.s_addr;
         daddr = ptr->addr.src.s_addr;
         sport = ((u_short *)&ptr->addr.port)[1];
         dport = ((u_short *)&ptr->addr.port)[0];
         port = dport;
      }
   
      for (i = 0, retn = 1; ((i < POLICYTESTCRITERIA) && retn); i++) {
         retn = 0;
         switch (i) {
            case POLICYTESTPROTO:
               switch (policy->flags & (UDP | TCP | ICMP | IP)) {
                  case  UDP: if (ptr->status &  UDPPROTO) retn++; break;
                  case  TCP: if (ptr->status &  TCPPROTO) retn++; break;
                  case ICMP: if (ptr->status & ICMPPROTO) retn++; break;
                  case   IP: retn++; break;
               }
               break;
   
            case POLICYTESTSRC:
               if ((saddr & ~policy->src.mask) == policy->src.addr)
                  retn++;
               break;
            case POLICYTESTDST:
               if ((daddr & ~policy->dst.mask) == policy->dst.addr)
                  retn++;
               break;
            case POLICYTESTPORT:
               switch (policy->action) {
                  case  EQ:
                     if (port == policy->port)
                        retn++; break;
                  case  LT:
                     if (port < policy->port)
                        retn++; break;
                  case  GT:
                     if (port > policy->port)
                        retn++; break;
                  case NEQ:
                     if (port != policy->port)
                        retn++; break;
                  case  EST:
                     if (!(ptr->status & SAW_SYN))
                        retn++; break;
                  default:
                     retn++; break;
               }
               break;
         }
      }
   }

   return (retn);
}


do_notification (ptr, policy)
struct writeStruct *ptr;
struct policyStruct *policy;
{
   register int retn = 1;

   if (policy)
      if (policy->flags & PERMIT) {
         if (debugflag > 1)
            printf ("%s %s", "policy: permitted", policy->str);
         else
            retn = 0;
      } else
         if (debugflag)
            printf ("%s %s", "policy: denyed", policy->str);

   return (retn);
}

 

struct naddrmem *naddrtable [HASHNAMESIZE];

struct naddrmem {
   struct naddrmem *nxt;
   u_long addr;
   u_short port;
};


char *tag_string = "Argus Version ";
extern u_long localaddr;
int major_version = 0, minor_version = 0;


handle_argus_control (ptr)
struct writeStruct *ptr;
{
}

handle_datum (ptr, read_mode, bpfcode)
struct writeStruct *ptr;
int read_mode;
struct bpf_program *bpfcode;
{
   int retn;
   struct writeStruct buf;
   struct bpf_insn *fcode = bpfcode->bf_insns;

   if (ptr->status & ARGUSCONTROL)
      handle_argus_control (&ptr);

   else {
      reformat_datum (ptr, &buf);
      if (retn = check_flags (&buf)) {
         if (retn = argus_filter (fcode, &buf)) {
            if (retn = check_policy (&buf, policy)) {
               if (wfile) {
                  writeNewLogfile (ptr);
   
                  if (strcmp (wfile, "-"))
                     version_process (&buf, read_mode);
               } else
                  version_process (&buf, read_mode);
            }
         }
      }
   }
}

#define IPPROTOMASK  (TCPPROTO | ICMPPROTO | UDPPROTO)
#define PROTOMASK     (IPPROTO | ARPPROTO)

reformat_datum (ptr, buf)
struct writeStruct *ptr, *buf;
{
   u_long saddr, daddr;
   u_short sport, dport;
   struct writeStruct tmpbuf, *tmp = &tmpbuf;

   if (ptr && buf) {
      bcopy (ptr, tmp, sizeof (struct writeStruct));
      if (tmp->status & REVERSE) {
         bcopy (&ptr->etherdst, &tmp->ethersrc, 6);
         bcopy (&ptr->ethersrc, &tmp->etherdst, 6);
         bcopy (&ptr->addr.dst, &tmp->addr.src, 4);
         bcopy (&ptr->addr.src, &tmp->addr.dst, 4);
         sport = ((unsigned short *) &ptr->addr.port)[0];
         dport = ((unsigned short *) &ptr->addr.port)[1];
         bcopy ((char *) &sport, &((unsigned short *) &tmp->addr.port)[1], 2);
         bcopy ((char *) &dport, &((unsigned short *) &tmp->addr.port)[0], 2);
         tmp->status &= ~REVERSE;
      }

      bcopy (tmp, buf, sizeof (struct writeStruct));
      sport = ((unsigned short *) &tmp->addr.port)[0];
      dport = ((unsigned short *) &tmp->addr.port)[1];

      if (tmp->status & IPPROTOMASK) {
         tmp->status |= IPPROTO;
         if ((dport > sport) && (sport != 0)) {
            tmp->status |= REVERSE;
            sport = ((unsigned short *) &tmp->addr.port)[1];
            dport = ((unsigned short *) &tmp->addr.port)[0];
         }

         switch (tmp->status & IPPROTOMASK) {
            case TCPPROTO:
               if (dport == 20)
                  tmp->status ^= REVERSE;
               break;

            case UDPPROTO:
               if (((sport == 53) && (dport != 53)) ||
                   ((sport == 2049) && (dport != 2049)) ||
                   ((sport >= 7000) && (sport < 7010)) ||
                   ((sport >= 6000) && (sport < 6010)))
                     tmp->status ^= REVERSE;
               break;
         }
      }
      
      if (tmp->status & REVERSE) {
         bcopy (&tmp->etherdst, &buf->ethersrc, 6);
         bcopy (&tmp->ethersrc, &buf->etherdst, 6);
         bcopy (&tmp->addr.dst, &buf->addr.src, 4);
         bcopy (&tmp->addr.src, &buf->addr.dst, 4);
         sport = ((unsigned short *) &tmp->addr.port)[0];
         dport = ((unsigned short *) &tmp->addr.port)[1];
         bcopy ((char *) &sport, &((unsigned short *) &buf->addr.port)[1], 2);
         bcopy ((char *) &dport, &((unsigned short *) &buf->addr.port)[0], 2);
         buf->src_count = tmp->dst_count; buf->dst_count = tmp->src_count;
         buf->src_bytes = tmp->dst_bytes; buf->dst_bytes = tmp->src_bytes;
         buf->status = tmp->status & ~REVERSE;
      }
   }
}



read_remote_connection (fd, bpfcode)
int fd;
struct bpf_program *bpfcode;
{
   int retn = 0, cnt = 0;
   int read_mode = 0;
   char buf[256], *str = buf, c;

   bzero (buf, 256);
   if (fd) {
      while (read (fd, str, 1)) {
         cnt++;
         if (*str++ == '\n') break;
      }

      if (cnt) {
         if (strstr (buf, tag_string)) {
            str = &buf[strlen (tag_string)];
            if (sscanf (str, "%d.%d", &major_version, &minor_version)) {
               fprintf (stderr, "%s", buf);
               read_mode = major_version;
               argus_parse_init (fd, read_mode);
               init ();
               retn = read_remote (fd, read_mode, bpfcode);
            }
         } else {
            fprintf (stderr, "not argus server: disconnecting.\n");
            retn = 1;
         }
      } else {
         fprintf (stderr, "refused.\n");
         retn = 1;
      }

      close (fd);
   }
   return (retn);
}


read_connection (fd, bpfcode)
FILE *fd;
struct bpf_program *bpfcode;
{
   int retn = 1;
   int read_mode = 0;
   char buf[256], *str, c;

   if (fd) {
      if (fgets (buf, 256, fd)) {
         if (strstr (buf, tag_string)) {
            str = &buf[strlen (tag_string)];
            if (sscanf (str, "%d.%d", &major_version, &minor_version)) {
               read_mode = major_version;
               argus_parse_init (fd, read_mode);
               init ();
               retn = read_file (fd, read_mode, bpfcode);
            } else
               fprintf (stderr, "Argus file format error: %s", buf);

         } else
            fprintf (stderr, "Not Argus file: %s", buf);
      }
      fclose (fd);
   }
   return (retn);
}


#define CLIENTTIMEOUT  1

read_remote (fd, read_mode, bpfcode)
int fd;
int read_mode;
struct bpf_program *bpfcode;
{
   struct writeStruct ptr;
   register int cnt, size, retn;
   extern void clientTimeout ();
   struct timeval wait, now, timeoutValue;
   fd_set readmask;
   int width = ulimit (4, NULL);

   switch (read_mode) {
      default: size = sizeof (struct writeStruct) ; break;
   }

   bzero (&ptr, sizeof (struct writeStruct));

   if (gettimeofday (&now, NULL) == 0) {
      timeoutValue = now;
      timeoutValue.tv_sec += CLIENTTIMEOUT;
      wait.tv_sec = 0; wait.tv_usec = 250000;
   
      FD_ZERO (&readmask);
      FD_SET (fd, &readmask);

      for (;;) {
         if ((retn = select (width, &readmask, NULL, NULL, &wait)) >= 0) {
            if (FD_ISSET (fd, &readmask))
               if ((cnt = read (fd, &ptr, size)) >= 0)
                  if (cnt == size)
                     handle_datum (&ptr, read_mode, bpfcode);
   
            gettimeofday (&now, NULL);
   
            if (now.tv_sec > timeoutValue.tv_sec) {
               (void) clientTimeout ();
               now = timeoutValue;
               timeoutValue.tv_sec += CLIENTTIMEOUT;
            }
            FD_SET (fd, &readmask);
         } else
            if (errno != EINTR)
               break;
      }

      if (retn < 0)
         perror ("select");
   }
}


read_file (fd, read_mode, bpfcode)
FILE *fd;
int read_mode;
struct bpf_program *bpfcode;
{
   struct writeStruct ptr;
   register int cnt, size, retn = 1;

   switch (read_mode) {
      default: size = sizeof (struct writeStruct) ; break;
   }

   while ((cnt = fread (&ptr, size, 1, fd)) > 0) {
      retn = 0;
      handle_datum (&ptr, read_mode, bpfcode);
   }

   return (retn);
}


struct tm tm_lasttime;
struct tm tm_startime;

version_process (ptr, read_mode)
struct writeStruct *ptr;
int read_mode;
{
   unsigned int status;
   struct tm *tm;

   if (tm = localtime (&ptr->startime.tv_sec))
      bcopy ((char *) tm, (char *)&tm_startime, sizeof (struct tm));
   if (tm = localtime (&ptr->lasttime.tv_sec))
      bcopy ((char *) tm, (char *)&tm_lasttime, sizeof (struct tm));
   
   if (status = (ptr->status & PROTOMASK)) {
      switch (status) {
         case   IPPROTO:
            switch (ptr->status & IPPROTOMASK) {
               case  TCPPROTO: process_tcp (ptr); break;
               case  UDPPROTO: process_udp (ptr); break;
               case ICMPPROTO: process_icmp (ptr); break;
                      default: process_ip (ptr); break;
            }
            break;
      }
   }
}


in_queue (queue, object)
struct QUEUE *queue;
struct NET_OBJECT *object;
{
   register int retn = 0;
   register struct NET_OBJECT *ptr;

   if (ptr = (struct NET_OBJECT *) queue->start)
      for ( ; ptr; ptr = (struct NET_OBJECT *) ptr->queue.nxt)
         if (ptr = object) {
            retn = 1; break;
         }

   return (retn);
}



add_to_queue (queue, object)
struct QUEUE *queue;
struct NET_OBJECT *object;
{
   register int i;
   register struct QUEUE_HEADER *head = queue->start;

   object->queue.queue = queue;
   queue->count++;

   if (head) {
      object->queue.prv = head->prv;
      if (object->queue.prv)
         (object->queue.prv)->nxt = &object->queue;

      object->queue.nxt = head;
   }

   queue->start = &object->queue;
}


remove_from_queue (object)
struct NET_OBJECT *object;
{
   register struct QUEUE *queue;
   register struct QUEUE_HEADER *prv, *nxt;

   if (object) {
      if (queue = object->queue.queue) {
         queue->count--;
         object->queue.queue = NULL;
   
         nxt = object->queue.nxt;
   
         if (prv = object->queue.prv)
            prv->nxt = nxt;
   
         if (nxt)
            nxt->prv = prv;
   
         if (queue->start == &object->queue)
            queue->start = nxt;
      }

      object->queue.prv = NULL;
      object->queue.nxt = NULL;
   }
}




#include <netdb.h>

get_server_socket (ip)
struct IP_ENTRY *ip;
{
   register int retn = 0;
   struct sockaddr_in server;
   struct servent *sp;
   struct hostent *hp;
   int s;

   if (!portnum) {
      if (sp = getservbyname ("monitor", "tcp")) portnum = sp->s_port;
      else portnum = 561;
   }

   if (hp = gethostbyaddr ((char *)&ip->addr, sizeof (ip->addr), AF_INET)) {
      bzero ((char *) &server, sizeof (server));
      bcopy (hp->h_addr, (char *)&server.sin_addr, hp->h_length);
      server.sin_family = hp->h_addrtype;
      server.sin_port = portnum;
      fprintf (stderr, "%s: Trying %s port %d\n", progname, (hp->h_name) ?
                  (hp->h_name) : intoa (ip->addr), portnum); 
      if ((s = socket (AF_INET, SOCK_STREAM, 0)) >= 0)
         if ((connect (s, (struct sockaddr *)&server, sizeof(server))) >= 0)
            if (retn = s)
               fprintf (stderr, "%s: connected ", progname);
   }

   return (retn);
}


new_file_arg (ptr)
char *ptr;
{
   register int retn = 0;
   register struct FILE_ENTRY *file, *list, *tmp;

   if (ptr) {
      if (file = (struct FILE_ENTRY *) calloc (1, sizeof(struct FILE_ENTRY))) {
         if (list = input_file_list) {
            while (list->nxt) list = list->nxt;
            list->nxt = file;
         } else
            input_file_list = file;

         file->str = strdup (ptr);
         retn = 1;
      }
   }
   return (retn);
}


remote_host_arg (ptr)
char *ptr;
{
   register int retn = 0;
   register struct IP_ENTRY *addr;
   u_long alist, **name;

   if ((alist = (u_long) inet_addr (ptr)) == (u_long) -1) {
      if (name = (u_long **) pcap_nametoaddr (ptr)) {
         while (*name) {
            if (addr = (struct IP_ENTRY *) calloc (1, 
                                            sizeof (struct IP_ENTRY))) {
               addr->nxt = remote_host_list;
               remote_host_list = addr;
               addr->addr = **name;
               *name++;
               retn = 1;
            }
         }
      }
   } else
      if (addr = (struct IP_ENTRY *) calloc (1, sizeof (struct IP_ENTRY))) {
         addr->nxt = remote_host_list;
         remote_host_list = addr;
         addr->addr = alist;
         retn = 1;
      }

   return (retn);
}


static int firstWrite = 1;

writeNewLogfile (ptr)
struct writeStruct *ptr;
{
   int i, retn;
   struct stat buf;
   FILE *fd;

   if (wfile) {
      if (strcmp (wfile, "-")) {
         if (stat (wfile, &buf) < 0)
            if (errno == ENOENT)
               if (fd = fopen (wfile, "a")) {
                  fprintf (fd, "Argus Version %d.%d\n",
                                           VERSION_MAJOR, VERSION_MINOR);
                  fclose (fd);
               } else
                  perror ("fopen");

         if (fd = fopen (wfile, "a")) {
            fwrite (ptr, sizeof (struct writeStruct), 1, fd);
            fclose (fd);
         }
      } else {
         if (firstWrite) {
            fprintf (stdout, "Argus Version %d.%d\n",
                                           VERSION_MAJOR, VERSION_MINOR);
            fflush (stdout);
            firstWrite = 0;
         }
         fwrite (ptr, sizeof (struct writeStruct), 1, stdout);
         fflush (stdout);
      }
   }
}

