
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_ip.c - routines related to ip event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */


#include <stdlib.h>
#include <unistd.h>


#include <compat.h>
#include <pcap.h>
#include <interface.h>

#include <argus.h>
#include <cons_ip.h>

#define TRUE    1
#define FALSE   0

static int ip_eol = 0;
static int ip_nop = 0;

int ip_options = 0;
int ip_ts = 0;
int ip_rr = 0;
int ip_sec = 0;
int ip_lsrr = 0;
int ip_ssrr = 0;
int ip_satid = 0;

static void new_ip_hash_entry ();
static void delete_ip ();
static void remove_ip_hash_entry ();
static void cons_ip ();
static int parse_options ();

void
cons_ip_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &ip_display_list, sizeof (ip_display_list));
   bzero ((char *) &ip_hash_table, sizeof (ip_hash_table));

   if (ptr = (struct HASH_TABLE_HEADER **) calloc (IPSEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      ip_hash_table.hash_array = ptr;
      ip_hash_table.size = IPSEQ_HASHSIZE;
   } else
      perror ("cons_ip_init: calloc");

   ip_display_list.timerRoutine = check_ip_timeouts;
   ip_display_list.logRoutine   = log_ip_connection;
   iptimeout = iptimeout ? iptimeout : IPTIMEOUT;
}


void argus_ip_handler (ep, ip, length, tvp)
struct ether_header *ep;
struct ip *ip;
int length;
struct timeval *tvp;
{
   int hlen, len, optionlen;
   unsigned char *cp;
   u_char buf[1024];
   u_char *abuf = buf;
   extern int snaplen;

   if (length >= sizeof (struct ip)) {

#ifdef ALIGN
   if ((int) ip & (sizeof (long) - 1)) {
      bcopy ((char *) ip, (char *) abuf, min (length, snaplen));
      ip = (struct ip *) abuf;
   }
#endif

      hlen = ip->ip_hl << 2;
      NTOHS(ip->ip_len); NTOHS(ip->ip_off);
      len = ip->ip_len - hlen;
   
      optionlen = (hlen - sizeof (struct ip));
      if (optionlen) {
         ip_options = 1;
         if (parse_options ((unsigned char *) (ip + 1), optionlen))
            return;
      } else {
         ip_options = ip_eol = ip_nop = ip_ts = 0;
         ip_rr = ip_sec = ip_lsrr = ip_ssrr = ip_satid = 0;
      }
   
      if ((ip->ip_off & 0x1fff) == 0) {
         cp = (unsigned char *)ip + hlen;
         switch (ip->ip_p) {
            case IPPROTO_UDP:
               cons_udp (ep, (struct udphdr *) cp, len, ip, tvp);
               break;
   
            case IPPROTO_ICMP:
               cons_icmp (ep, (struct icmp *) cp, len, ip, tvp);
               break;
   
            case IPPROTO_TCP:
               cons_tcp (ep, (struct tcphdr *) cp, len, ip, tvp);
               break;
   
            default:
               cons_ip (ep, ip, len, tvp); 
               break;
         }
      }
   }
}

static int
parse_options (ptr, len)
unsigned char *ptr;
int len;
{
   int retn = 0, offset = 0;

   for (; len > 0; ptr += offset, len -= offset) {
      switch (*ptr) {
         case IPOPT_EOL:      ip_eol++; break;
         case IPOPT_NOP:      ip_nop++; break;
         case IPOPT_TS:       ip_ts++; break;
         case IPOPT_RR:       ip_rr++; break;
         case IPOPT_SECURITY: ip_sec++; break;
         case IPOPT_LSRR:     ip_lsrr++; break;
         case IPOPT_SSRR:     ip_ssrr++; break;
         case IPOPT_SATID:    ip_satid++; break;
         default:             retn++; break;
      }
      if (!retn) {
         if ((*ptr == IPOPT_EOL) || (*ptr == IPOPT_NOP))
            offset = 1;
         else {
            offset = ptr[1];
            if (!(offset && (offset <= len))) {
               retn++; break;
            }
         }
      }
   }

   return (retn);
}


static void 
cons_ip (ep, ip, length, tvp)
struct ether_header *ep;
struct ip *ip;
int length;
struct timeval *tvp;
{
   struct IP_OBJECT *ih;
   int rev, status;
   struct tha tha;

   if (length && ip) {
      create_ip_tha (&tha, ip, &rev);
 
      if (ih = (struct IP_OBJECT *) find_hash_entry (&ip_hash_table, &tha))
         update_ip_record (ih, ep, length, tvp, &rev);
      else
         new_ip_hash_entry (ip, ep, rev, tvp, &tha, length);
   }
}

create_ip_tha (tha, ip, rev)
struct tha *tha;
struct ip *ip;
int *rev;
{
   bzero (tha, sizeof (*tha));
   if (ip->ip_src.s_addr < ip->ip_dst.s_addr) {
      tha->src = ip->ip_dst, tha->dst = ip->ip_src;
      *rev = 1;
   } else {
      tha->src = ip->ip_src, tha->dst = ip->ip_dst;
      *rev = 0;
   }
   tha->port  = ip->ip_p;
}


update_ip_record (ih, ep, length, tvp, rev)
struct IP_OBJECT *ih;
struct ether_header *ep;
int length;
struct timeval *tvp;
int rev;
{
   struct ipcb *ipCb;

   ipCb = &ih->ip_cb;
   ipCb->lasttime = *tvp; ih->qhdr.last_time = *tvp;
   if (ipCb->rev == rev) {
      ipCb->src.count++;
      ipCb->src.bytes += length;
   } else {
      ipCb->dst.count++;
      ipCb->dst.bytes += length;
   }

   if (ipCb->status & MODIFIED) {
      if (!(ipCb->status & MULTIADDR)) {
         if (rev) {
            if (bcmp ((char *)&ep->ether_dhost,
                       (char *)&ih->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_shost,
                       (char *)&ih->link.phys.etherdst, 6))
               ipCb->status |= MULTIADDR;
         } else
            if (bcmp ((char *)&ep->ether_shost,
                       (char *)&ih->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_dhost,
                       (char *)&ih->link.phys.etherdst, 6))
               ipCb->status |= MULTIADDR;
         }
   } else {
      if (rev) {
         bcopy ((char *)&ep->ether_dhost, (char *)&ih->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_shost, (char *)&ih->link.phys.etherdst, 6);
      } else {
         bcopy ((char *)&ep->ether_shost, (char *)&ih->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&ih->link.phys.etherdst, 6);
      }
   }

   if ((ipCb->src.count + ipCb->dst.count) == 2)
      ipCb->status |= CON_ESTABLISHED;
   update_queue_status ((struct QUEUE_HEADER *) &ih->qhdr);
   ipCb->status |= MODIFIED;

   if (ip_options) {
      if (ip_rr)    ipCb->status |= RECORDROUTE;
      if (ip_ts)    ipCb->status |= TIMESTAMP;
      if (ip_sec)   ipCb->status |= SECURITY;
      if (ip_lsrr)  ipCb->status |= LSRCROUTE;
      if (ip_ssrr)  ipCb->status |= SSRCROUTE;
      if (ip_satid) ipCb->status |= SATNETID;
   }

   if (((tvp->tv_sec - ih->ip_cb.startime.tv_sec) > iptimeout) ||
         (((ipCb->src.count == 1) && (ipCb->dst.count == 1)) &&
              !(ipCb->status & LOGGED)))

      log_ip_connection (ih, tvp, CON_ESTABLISHED);
}


int ipdeletecount = 0;


static void
delete_ip (ih)
struct IP_OBJECT *ih;
{
   ipdeletecount++;
   remove_hash_table_entry (&ip_hash_table, &ih->addr);
   free (ih);
}


static void
new_ip_hash_entry (ip, ep, rev, tvp, tha)
struct ip *ip;
struct ether_header *ep;
int rev;
struct timeval *tvp;
struct tha *tha;
{
   struct IP_OBJECT *ptr = NULL;
   struct ipcb *ipCb;

   if (ptr = (struct IP_OBJECT *) calloc (1, sizeof (*ptr))) {
      ipCb = &ptr->ip_cb;
      ipCb->status = IPPROTO | IP_INIT | MODIFIED;
      if (dflag) ipCb->status |= DETAIL;
      ipCb->startime = *tvp;
      ipCb->lasttime = *tvp;
      if (ipCb->rev = rev) {
         ipCb->status |= REVERSE;
         bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.etherdst, 6);
      } else {
         bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
      }
      ptr->qhdr.last_time = *tvp;
      bcopy ((char *) tha, (char *) &ptr->addr, sizeof (struct tha));

      if (ip_options) {
         if (ip_rr)    ipCb->status |= RECORDROUTE;
         if (ip_ts)    ipCb->status |= TIMESTAMP;
         if (ip_sec)   ipCb->status |= SECURITY;
         if (ip_lsrr)  ipCb->status |= LSRCROUTE;
         if (ip_ssrr)  ipCb->status |= SSRCROUTE;
         if (ip_satid) ipCb->status |= SATNETID;
      }  

      ipCb->src.count++;
      ipCb->src.bytes += ip->ip_len - (ip->ip_hl << 2);

      if (add_to_queue (&ip_display_list,(struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&ip_hash_table, tha, ptr);
         if (dflag) {
            log_ip_connection (ptr, tvp, IP_INIT);
            ipCb->status &= ~LOGGED;
         }
      } else
         delete_ip (ptr);
   }
}


#include <sys/stat.h>
#include <sys/errno.h>
 
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;
 
extern int clienttags;
extern int clients[];
 
 
void
log_ip_connection (ptr, tvp, state)
struct IP_OBJECT *ptr;
struct timeval *tvp;
int state;
{
   struct writeStruct output;

   if (clienttags || wflag) {
      if ((ptr->ip_cb.src.count || ptr->ip_cb.dst.count) &&
                  (ptr->ip_cb.status &= ~MODIFIED)) {
         output.startime  =  ptr->ip_cb.startime;
         output.lasttime  =  ptr->ip_cb.lasttime;
         output.status    =  ptr->ip_cb.status | state;
         ptr->ip_cb.status &= ~MODIFIED;
         ptr->ip_cb.status &= ~MULTIADDR;
         output.addr      =  ptr->addr;
         bcopy ((char *)&ptr->link.phys.ethersrc, (char *)&output.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst, (char *)&output.etherdst, 6);
         output.src_count = ptr->ip_cb.src.count - ptr->ip_cb.src.lastcount;
         output.dst_count = ptr->ip_cb.dst.count - ptr->ip_cb.dst.lastcount;
         output.src_bytes = ptr->ip_cb.src.bytes - ptr->ip_cb.src.lastbytes;
         output.dst_bytes = ptr->ip_cb.dst.bytes - ptr->ip_cb.dst.lastbytes;
         if ((output.src_count || output.src_bytes) ||
                        (output.dst_count || output.dst_bytes)) {
            if (ptr->ip_cb.status & state) {
               writeOutData (&output);
               ptr->ip_cb.status |= LOGGED;
            }

            if (!(dflag && (state & UDP_INIT)) &&
                           (ptr->ip_cb.status & LOGGED)) {
               ptr->ip_cb.src.lastcount = ptr->ip_cb.src.count;
               ptr->ip_cb.src.lastbytes = ptr->ip_cb.src.bytes;
               ptr->ip_cb.dst.lastcount = ptr->ip_cb.dst.count;
               ptr->ip_cb.dst.lastbytes = ptr->ip_cb.dst.bytes;
               ptr->ip_cb.startime = *tvp; ptr->ip_cb.lasttime = *tvp;
            }
         }
      }  
   }

   ptr->qhdr.logtime = *tvp;
}


void
check_ip_timeouts (ptr, tvp)
struct IP_OBJECT *ptr;
struct timeval *tvp;
{
   int srccounts, dstcounts;

   if ((tvp->tv_sec - ptr->ip_cb.lasttime.tv_sec) > iptimeout) {
      if ((ptr->ip_cb.src.count || ptr->ip_cb.dst.count) &&
                                    !(ptr->ip_cb.status & LOGGED)) {
         ptr->ip_cb.status |= TIMED_OUT;
         log_ip_connection (ptr, tvp, TIMED_OUT);
      }
      remove_from_queue (&ip_display_list,
                             (struct QUEUE_HEADER *) &ptr->qhdr);
      delete_ip (ptr);

   } else {
      if ((tvp->tv_sec - ptr->ip_cb.startime.tv_sec) > iptimeout) {
         srccounts = (ptr->ip_cb.src.count - ptr->ip_cb.src.lastcount);
         dstcounts = (ptr->ip_cb.dst.count - ptr->ip_cb.dst.lastcount);
         if ((srccounts || dstcounts) &&
                    (ptr->ip_cb.status & CON_ESTABLISHED)) {
            log_ip_connection (ptr, tvp, CON_ESTABLISHED);
            ptr->qhdr.last_time = *tvp;
            update_queue_status ((struct QUEUE_HEADER *) &ptr->qhdr);
         }
      }
   }
}
