
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_tcp.c - handle tcp specific protocol state machine,
 *              and routines related to tcp event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */

#define CONS_TCP

#include <stdlib.h>
#include <unistd.h>

#include <compat.h>
#include <pcap.h>
#include <interface.h>

#include <argus.h>
#include <cons_tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp.h>
/*#include <netinet/tcp_fsm.h>*/
#include "tcp_fsm.h"

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;


static void new_tcp_hash_entry ();
static void delete_tcp ();
static int update_seq ();
static int tcp_state ();

void
cons_tcp_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &tcp_display_list, sizeof (tcp_display_list));
   bzero ((char *) &tcp_timeout_list, sizeof (tcp_timeout_list));
   bzero ((char *) &tcp_hash_table, sizeof (tcp_hash_table));
   if (ptr = (struct HASH_TABLE_HEADER **) calloc (TSEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      tcp_hash_table.hash_array = ptr;
      tcp_hash_table.size = TSEQ_HASHSIZE;
   } else
      perror ("cons_tcp_init: calloc");

   tcp_display_list.timerRoutine = check_tcp_timeouts;
   tcp_display_list.logRoutine =   log_tcp_connection;
   tcp_timeout_list.timerRoutine = check_tcp_timeouts;
   tcp_timeout_list.logRoutine =   log_tcp_connection;

   tcptimeout = tcptimeout ? tcptimeout : TCPTIMEOUT;
}


void
cons_tcp (ep, tp, length, ip, tvp)
struct ether_header *ep;
struct tcphdr *tp;
int length;
struct ip *ip;
struct timeval *tvp;
{
   struct TCP_OBJECT *th;
   int rev = 0;
   struct tha tha;

   if ((length >= sizeof(struct tcphdr)) && (tp)) {
      create_tcp_tha (&tha, ip, tp, &rev);

      if (th = (struct TCP_OBJECT *) find_hash_entry (&tcp_hash_table, &tha))
         update_tcp_record (th, ep, tp, length, tvp, rev, tp->th_flags);
      else
         new_tcp_hash_entry (tp, ep, rev, tvp, ip, &tha,
                       length - (tp->th_off<<2));
   }
}


create_tcp_tha (tha, ip, tp, rev)
struct tha *tha;
struct ip *ip;
struct tcphdr *tp;
int *rev;
{

   if (tp->th_dport > tp->th_sport || (tp->th_sport == tp->th_dport &&
        ip->ip_src.s_addr < ip->ip_dst.s_addr)) {
      tha->src = ip->ip_dst, tha->dst = ip->ip_src;
      tha->port = tp->th_dport << 16 | tp->th_sport;
      *rev = 1;

   } else {
      tha->src = ip->ip_src, tha->dst = ip->ip_dst;
      tha->port = tp->th_sport << 16 | tp->th_dport;
      *rev = 0;
   }
}


update_tcp_record (th, ep, tp, length, tvp, rev, flags)
struct TCP_OBJECT *th;
struct ether_header *ep;
struct tcphdr *tp;
int length;
struct timeval *tvp;
int rev;
u_char flags;
{
   struct tcpcb *tcpCb;

   tcpCb = &th->tcp_cb;
   tcpCb->lasttime = *tvp; th->qhdr.last_time = *tvp;
   if (tcpCb->rev == rev)
      tcpCb->src.count++;
   else
      tcpCb->dst.count++;

   if (tcpCb->status & MODIFIED) {
      if (!(tcpCb->status & MULTIADDR)) {
         if (rev) {
            if (bcmp ((char *)&ep->ether_dhost,
                            (char *)&th->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_shost,
                            (char *)&th->link.phys.etherdst, 6))
               tcpCb->status |= MULTIADDR;
         } else {
            if (bcmp ((char *)&ep->ether_shost,
                            (char *)&th->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_dhost,
                            (char *)&th->link.phys.etherdst, 6))
               tcpCb->status |= MULTIADDR;
         }
      }
   } else {
      if (rev) {
         bcopy ((char *)&ep->ether_dhost, (char *)&th->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_shost, (char *)&th->link.phys.etherdst, 6);
      } else {
         bcopy ((char *)&ep->ether_shost, (char *)&th->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&th->link.phys.etherdst, 6);
      }
   }

   if (update_seq (tp, th, rev, length - (tp->th_off << 2))) {
      update_queue_status ((struct QUEUE_HEADER *) &th->qhdr);
      tcpCb->status |= MODIFIED;
    
      switch (tcp_state (flags, th, tvp)) {
         case TCPS_LISTEN:
            if (th->qhdr.queue == &tcp_display_list)
               log_tcp_connection (th, tvp, 0L);
            remove_from_queue (th->qhdr.queue,
                                 (struct QUEUE_HEADER *) &th->qhdr);
            delete_tcp (th);
            th = NULL;
            break;
               
         case TCPS_CLOSED:
         case TCPS_TIME_WAIT:
            if (th->qhdr.queue == &tcp_display_list) {
               if (!(th->tcp_cb.status & RESET))
                  th->tcp_cb.status |= NORMAL_CLOSE;

               log_tcp_connection (th, tvp, 0L);
               remove_from_queue (&tcp_display_list,
                                 (struct QUEUE_HEADER *) &th->qhdr);
               if (!(add_to_queue (&tcp_timeout_list,
                                 (struct QUEUE_HEADER *) &th->qhdr))) {
                  closed++;
                  delete_tcp (th);
               }
            }
            break;
      }
   }
}


static int
update_seq (tp, th, rev, len)
struct tcphdr *tp;
struct TCP_OBJECT *th;
int rev, len;
{
   int retn = 1;
   struct TCP_CON_OBJECT *src, *dst;
   tcp_seq seq = tp->th_seq + len;
   u_char flags = tp->th_flags;

   if (th->tcp_cb.rev == rev) {
      src = (struct TCP_CON_OBJECT *) &th->tcp_cb.src;
      dst = (struct TCP_CON_OBJECT *) &th->tcp_cb.dst;
   } else {
      src = (struct TCP_CON_OBJECT *) &th->tcp_cb.dst;
      dst = (struct TCP_CON_OBJECT *) &th->tcp_cb.src;
   }

   if (!src->seq_base)
      src->seq_base = tp->th_seq;

   if ((tp->th_seq < src->seq) && dst->win) {
      src->retrans += len;
      th->tcp_cb.status |= PKTS_RETRANS;
   } else
      src->seq = seq;
 
   if (tp->th_seq < dst->ack)
      src->strays++;
 
   if (!(src->win = tp->th_win) && !(flags & (TH_FIN|TH_RST)))
      th->tcp_cb.status |= SRC_WINDOW_SHUT;

   return (retn);
}


static int
tcp_state (flags, th, tvp)
u_char flags;
struct TCP_OBJECT *th;
struct timeval *tvp;
{
   int state = th->tcp_cb.t_state;
 
   if ((flags &= ~TH_PUSH) & TH_RST) {
      th->tcp_cb.status |= RESET;
      state = TCPS_CLOSED;
   } else {
      switch (state) {
         case TCPS_LISTEN:
         case TCPS_SYN_SENT:
            if (flags == (TH_SYN|TH_ACK)) {
               state = TCPS_SYN_RECEIVED;
               th->tcp_cb.status |= SAW_SYN_SENT;
               if (dflag) log_tcp_connection (th, tvp, DETAIL);
            } else if (flags & TH_ACK) {
               state = TCPS_ESTABLISHED;
               th->tcp_cb.status |= CON_ESTABLISHED;
               if (dflag) log_tcp_connection (th, tvp, DETAIL);
            }
 
            if (flags & TH_FIN)
               state = TCPS_CLOSING;
            break;
 
         case TCPS_SYN_RECEIVED:
            if (flags == (TH_FIN|TH_ACK))
               state = TCPS_FIN_WAIT_1;
            else if (!(flags & TH_SYN) && (flags & TH_ACK)) {
               state = TCPS_ESTABLISHED;
               th->tcp_cb.status |= CON_ESTABLISHED;
               if (dflag) log_tcp_connection (th, tvp, DETAIL);
            }
            break;
 
         case TCPS_ESTABLISHED:
            if (flags == (TH_FIN|TH_ACK))
               state = TCPS_FIN_WAIT_1;
               th->tcp_cb.status |= CLOSE_WAITING;
            break;
 
         case TCPS_CLOSE_WAIT:
         case TCPS_FIN_WAIT_1:
            if ((flags & (TH_FIN)))
               state = TCPS_CLOSING;
            else if (flags & TH_SYN)
               state = TCPS_LISTEN;
            else if (flags & TH_ACK)
               state = TCPS_FIN_WAIT_2;
            break;
 
         case TCPS_LAST_ACK:
         case TCPS_FIN_WAIT_2:
            if (flags == (TH_FIN|TH_ACK))
               state = TCPS_CLOSING;
            else

         case TCPS_CLOSING:
         case TCPS_TIME_WAIT:
            if (flags & TH_SYN)
               state = TCPS_LISTEN;
            else if (flags & TH_ACK)
               state = TCPS_CLOSED;
            break;

         case TCPS_CLOSED:
            if (flags & TH_SYN)
               state = TCPS_LISTEN;
            break;
      }
   }

   if (state != TCPS_LISTEN)
      th->tcp_cb.t_state = state;
   return (state);
}


int tcpdeletecount = 0;

static void
delete_tcp (th)
struct TCP_OBJECT *th;
{
   tcpdeletecount++;
   remove_hash_table_entry (&tcp_hash_table, &th->addr);
   free (th);
}


      
static void
new_tcp_hash_entry (tp, ep, rev, tvp, ip, tha, len)
struct tcphdr *tp;
struct ether_header *ep;
int rev;
struct timeval *tvp;
struct ip *ip;
struct tha *tha;
int len;
{
   struct TCP_OBJECT *ptr = NULL;
   struct tcpcb *tcpCb;
   u_char flags = tp->th_flags;
   u_char etherbuf[6];

   if (!(flags & TH_RST)) {
      if (ptr = (struct TCP_OBJECT *) calloc (1, sizeof (*ptr))) {
         tcpCb = &ptr->tcp_cb;
         tcpCb->status = TCPPROTO;
         if (ip) tcpCb->status |= IPPROTO;
         tcpCb->startime = *tvp;
         tcpCb->lasttime = *tvp;
         tcpCb->status |= MODIFIED;
         if (tcpCb->rev = rev) {
            tcpCb->status |= REVERSE;
            bcopy ((char *)&ep->ether_dhost,
                       (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost,
                       (char *)&ptr->link.phys.etherdst, 6);
         } else {
            bcopy ((char *)&ep->ether_shost,
                       (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost,
                       (char *)&ptr->link.phys.etherdst, 6);
         }
         bcopy ((char *) tha, (char *) &ptr->addr, sizeof (struct tha));

         ptr->qhdr.last_time = *tvp;
         
         if (ip_options) { 
            if (ip_ts)   tcpCb->status |= TIMESTAMP;
            if (ip_sec)  tcpCb->status |= SECURITY; 
            if (ip_lsrr) tcpCb->status |= LSRCROUTE; 
            if (ip_ssrr) tcpCb->status |= SSRCROUTE; 
            if (ip_rr)   tcpCb->status |= RECORDROUTE; 
            if (ip_satid) tcpCb->status |= SATNETID; 
         }

         switch (flags) {
            case (TH_SYN):
               tcpCb->t_state = TCPS_SYN_SENT;
               tcpCb->status |= SAW_SYN;
               tcpCb->src.count = 1;
               tcpCb->src.seq_base = tp->th_seq; 
               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            case (TH_SYN|TH_ACK):
               tcpCb->t_state = TCPS_SYN_RECEIVED;
               tcpCb->status |= SAW_SYN_SENT;
               tcpCb->rev = rev ? 0 : 1;
               tcpCb->status ^= REVERSE;
               bcopy ((char *)&ptr->link.phys.ethersrc, (char *) &etherbuf, 6);
               bcopy ((char *)&ptr->link.phys.etherdst,
                      (char *)&ptr->link.phys.ethersrc, 6);
               bcopy ((char *) &etherbuf, (char *)&ptr->link.phys.ethersrc, 6);

               tcpCb->dst.count = 1;
               tcpCb->dst.seq_base = tp->th_seq;
               tcpCb->src.seq_base = tp->th_ack - 1;

               tcpCb->status |= SAW_SYN;
               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            case (TH_ACK):
            case (TH_PUSH):
            case (TH_PUSH|TH_ACK):
            case (TH_URG|TH_ACK):
            case (TH_PUSH|TH_URG|TH_ACK):
               tcpCb->src.count = 1;
               tcpCb->src.seq_base = tp->th_seq - len;
               tcpCb->src.seq = tp->th_seq;
               tcpCb->t_state = TCPS_ESTABLISHED;
               tcpCb->status |= CON_ESTABLISHED;
               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            default:
               tcpCb->t_state = TCPS_CLOSING;
               break;
         }

         if (tcpCb->t_state == TCPS_CLOSING) {
            if (add_to_queue (&tcp_timeout_list,
                           (struct QUEUE_HEADER *) &ptr->qhdr))
               (void) add_hash_table_entry (&tcp_hash_table, tha, ptr);

            else
               closed++;
         } else
            if (add_to_queue (&tcp_display_list,
                           (struct QUEUE_HEADER *) &ptr->qhdr))
               (void) add_hash_table_entry (&tcp_hash_table, tha, ptr);

            else
               delete_tcp (ptr);
      }
   }
}


#include <sys/stat.h>
#include <sys/errno.h>

extern char *sys_errlist[];
extern int errno;

void
log_tcp_connection (ptr, tvp, state)
struct TCP_OBJECT *ptr;
struct timeval *tvp;
int state;
{
   unsigned int status = 0;
   int src_count, dst_count, src_bytes, dst_bytes;
   struct TCP_CON_OBJECT *src, *dst;
   struct writeStruct output;

   if (clienttags || wflag) {
      bzero ((char *) &output, sizeof (struct writeStruct));
      ptr->tcp_cb.status |= state;
      status = ptr->tcp_cb.status;
      ptr->tcp_cb.status &= ~MODIFIED;
      ptr->tcp_cb.status &= ~MULTIADDR;
      src = (struct TCP_CON_OBJECT *) &ptr->tcp_cb.src;
      dst = (struct TCP_CON_OBJECT *) &ptr->tcp_cb.dst;

      if (dflag & (!(status & (NORMAL_CLOSE|RESET|TIMED_OUT))))
         output.startime =  ptr->qhdr.logtime;
      else
         output.startime =  ptr->tcp_cb.startime;

      output.lasttime =  ptr->tcp_cb.lasttime;
      output.status   =  status;
      bcopy ((char *)&ptr->link.phys.ethersrc, (char *)&output.ethersrc, 6);
      bcopy ((char *)&ptr->link.phys.etherdst, (char *)&output.etherdst, 6);
      output.addr     =  ptr->addr;
      output.src_count = src->count;
      output.dst_count = dst->count;

      if (src->seq && src->seq_base)
         if (output.src_count > 0)
            output.src_bytes = (src->seq - src->seq_base) - 1;

      if (dst->seq && dst->seq_base)
         if (output.dst_count > 0)
            output.dst_bytes = (dst->seq - dst->seq_base) - 1;

      if (dflag && (status & TCPS_ESTABLISHED) &&
              (!(status & (NORMAL_CLOSE|RESET|TIMED_OUT)))) {
         src_count = output.src_count;
         dst_count = output.dst_count;
         src_bytes = output.src_bytes;
         dst_bytes = output.dst_bytes;
         output.src_count = src_count - src->lastcount;
         output.dst_count = dst_count - dst->lastcount;
         output.src_bytes = src_bytes - src->lastbytes;
         output.dst_bytes = dst_bytes - dst->lastbytes;
         src->lastcount = src_count;
         src->lastbytes = src_bytes;
         dst->lastcount = dst_count;
         dst->lastbytes = dst_bytes;
      }

      if ((output.src_count || output.dst_count) ||
          (output.src_bytes || output.dst_bytes))
         writeOutData (&output);

      ptr->tcp_cb.status |= LOGGED;
      ptr->tcp_cb.status ^= state;
   }
   ptr->qhdr.logtime = *tvp;
}


void
check_tcp_timeouts (ptr, tvp)
struct TCP_OBJECT *ptr;
struct timeval *tvp;
{
   switch (ptr->tcp_cb.t_state) {
      case TCPS_SYN_SENT:
      case TCPS_SYN_RECEIVED:
      case TCPS_ESTABLISHED:
      case TCPS_CLOSE_WAIT:
      case TCPS_FIN_WAIT_1:
      case TCPS_LAST_ACK:
      case TCPS_FIN_WAIT_2:
      case TCPS_CLOSING:
      case TCPS_TIME_WAIT:
         ptr->tcp_cb.status |= TIMED_OUT;
         if (ptr->qhdr.queue == &tcp_display_list)
            log_tcp_connection (ptr, tvp, 0L);
         break;
   }

   remove_from_queue (ptr->qhdr.queue, (struct QUEUE_HEADER *)&ptr->qhdr);
   closed++;
   delete_tcp (ptr);
}
