
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * argus_util.c - supports connection tracking 
 *
 * Useful routines to support argus package.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 */


#include <stdlib.h>
#include <unistd.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>

#include <compat.h>
#include <pcap.h>
#include <pcap-int.h>
#include <interface.h>
#include <addrtoname.h>

#include <argus.h>
#include <cons_tcp.h>

extern fd_set readmask, writemask, exceptmask;

struct callback {
   pcap_handler function;
   int type;
};

static struct callback callbacks[] = {
   { cons_ether_packet,  DLT_EN10MB },
   { cons_fddi_packet,   DLT_FDDI },
   { NULL,               DLT_SLIP },
   { NULL,               DLT_PPP },
   { NULL,               DLT_NULL },
   { NULL,               NULL },
};


extern int lfd;
extern pcap_t *pd;

static void check_all_timeouts ();
static void wrapup ();

void
argus_loop (p, callback)
pcap_t *p;
pcap_handler callback;
{
   int pfd = 0, width = ulimit (4, NULL);
   struct timeval wait, tvp;
   extern struct timeval globaltvp;
   extern char *rfile;
   
   wait.tv_sec = 1; wait.tv_usec = 0;

   if (p && callback) {
      if ((pfd = p->fd) >= 0) {
         if (lfd >= 0) FD_SET (lfd, &readmask);
         if (pfd >= 0) FD_SET (pfd, &readmask);

         while (select (width, &readmask, NULL, NULL, &wait) >= 0) {
            if (FD_ISSET (pfd, &readmask))
               pcap_read (p, -1, callback, (u_char *) NULL);

            if ((lfd >= 0) && FD_ISSET (lfd, &readmask))
               check_client_status (lfd);

            gettimeofday (&tvp, NULL);

            if (updatetime (&tvp)) check_all_timeouts (&tvp);
            if (lfd >= 0) FD_SET (lfd, &readmask);
            if (pfd) FD_SET (pfd, &readmask);
         }
      } else
         pcap_offline_read (p, -1, callback, (u_char *) NULL);
   }
}

int reporttime = 0;

static void
check_all_timeouts (tvp)
struct timeval *tvp;
{
   extern int debug_interval;
   struct tm *tm;
   char buf[64];

   check_timeouts (&tcp_timeout_list, tvp, 60);
   check_timeouts (&tcp_display_list, tvp, tcptimeout);
   check_timeouts (&udp_display_list, tvp, udptimeout);
   check_timeouts ( &ip_display_list, tvp,  iptimeout);

   if (debugflag) {
      if (tvp->tv_sec >= reporttime) {
         extern int tcpdeletecount, udpdeletecount, ipdeletecount;
         extern pcap_t *pd;
         struct pcap_stat stat;

         reporttime = tvp->tv_sec + debug_interval;
         strftime (buf, 64, "%T", localtime (&tvp->tv_sec));
         if (pcap_stats (pd, &stat) >= 0) {
            (void) fprintf (stderr, "%s ", buf);
            (void) fprintf (stderr, "pkts %8dr %5dd",
                 stat.ps_recv, stat.ps_drop);
            (void) fprintf (stderr, " tcp %4da %4dt %5dd",
                 tcp_display_list.count, tcp_timeout_list.count,
                 tcpdeletecount);
            (void) fprintf (stderr, "  udp %4dc %5dd ",
                 udp_display_list.count, udpdeletecount);
            (void) fprintf (stderr, "  ip %3dc %4dd\n",
                 ip_display_list.count, ipdeletecount);
         }
      }
   }
}


static void
wrapup (pd)
pcap_t *pd;
{
   struct pcap_stat stat;
   extern char *rfile;
   extern int totalPktsRcv;

   if (rfile) {
      stat.ps_recv = totalPktsRcv;
      stat.ps_drop = 0;
   } else
      if (pcap_stats (pd, &stat) < 0)
         (void) fprintf (stderr, "pcap_stats: %s\n", pcap_geterr (pd));

   (void) fprintf (stderr, "\n%d packets recv'd by filter\n", stat.ps_recv);
   (void) fprintf (stderr, "%d packets dropped by kernel\n", stat.ps_drop);

   pcap_close (pd);
   close_clients ();
}

pcap_handler
lookup_pcap_callback (int type)
{
   pcap_handler retn = NULL;
   struct callback *callback;

   for (callback = callbacks; callback->function; ++callback)
      if (type == callback->type) {
         retn = callback->function;
         break;
      }

   return (retn);
}


void
cleanup ()
{
   struct timeval tvp;

   gettimeofday (&tvp, NULL);
   udptimeout = -1; tcptimeout = -1; iptimeout = -1;
   check_timeouts (&tcp_display_list, tvp, tcptimeout);
   check_timeouts (&udp_display_list, tvp, udptimeout);
   check_timeouts ( &ip_display_list, tvp,  iptimeout);
   
   if (pd != NULL) wrapup (pd);
   exit (0);
}

void
usr1sig ()
{
   debugflag = (debugflag++ > 30) ? 30 : debugflag;
   fprintf (stderr, "argus: debug enabled level %d\n", debugflag);
}

void
usr2sig ()
{
   debugflag = 0;
   fprintf (stderr, "argus: debug disabled\n");
}

void
usage(progname)
char *progname;
{
   fprintf (stderr, "Version %d.%d\n", VERSION_MAJOR, VERSION_MINOR);
   fprintf (stderr, "usage: %s [-bhOp][-d detail-interval]", progname);
   fprintf (stderr, "[-D debug-level][-r tcpdump-file]\n");
   fprintf (stderr, "           [-w argus-file]");
   fprintf (stderr, "[-U udp-timeout][-T tcp-timeout][-I ip-timeout]\n");
   fprintf (stderr, "           [-P port] [-i interface] expression\n");
   fprintf (stderr, "options: b - print filter definition\n");
   fprintf (stderr, "         d - log detailed events on sec interval\n");
   fprintf (stderr, "         D - set debug level\n");
   fprintf (stderr, "         h - print help\n");
   fprintf (stderr, "         i - specify network interface\n");
   fprintf (stderr, "         I - set IP transaction timer\n");
   fprintf (stderr, "         O - turn off filter optimizer\n");
   fprintf (stderr, "         p - don't go into promiscuous mode\n");
   fprintf (stderr, "         P - specify tcp remote access port\n");
   fprintf (stderr, "         r - read from tcpdump(1) packet file\n");
   fprintf (stderr, "         T - set TCP transaction timer\n");
   fprintf (stderr, "         U - set UDP transaction timer\n");
   fprintf (stderr, "         w - write output to logfile, or stdout '-'\n");
   exit (-1);
}


struct HASH_TABLE_HEADER *
add_hash_table_entry (table, tha, ptr)
struct HASH_TABLE *table;
struct tha *tha;
struct OBJECT *ptr;
{
   struct HASH_TABLE_HEADER *retn = NULL;
   struct HASH_TABLE_HEADER *entry = NULL, *start = NULL;
   unsigned short hash = 0;
   int i, len;

   for (i = 0, len = (sizeof (*tha) / sizeof (hash)); i < len; i++)
      hash += ((unsigned short *)tha)[i];

   if (entry = table->hash_array[hash % table->size]) {
      start = entry;
      do {
         if (!(bcmp ((char *)tha, (char *)&entry->tha, sizeof (*tha)))) {
            retn = entry;
            break;
         }
         entry = entry->nxt;
      } while (entry != start);
   }
   if (!(retn)) {
      if (retn = (struct HASH_TABLE_HEADER *) calloc (1, sizeof (*retn))) {
         retn->object = ptr;
         retn->hash = hash;
         bcopy ((char *)tha, (char *)&retn->tha, sizeof (*tha));
         if (start) {
            retn->nxt = start;
            retn->prv = start->prv;
            retn->prv->nxt = retn;
            retn->nxt->prv = retn;
         } else 
            retn->prv = retn->nxt = retn;
         table->hash_array[hash % table->size] = retn;
      } else perror ("add_hash_table_entry: calloc");
   } else aerror ("add_hash_table_entry: already in queue", 0L);

   return (retn);
}


struct OBJECT *
find_hash_entry (table, tha)
struct HASH_TABLE *table;
struct tha *tha;
{
   struct OBJECT *retn = NULL;
   struct HASH_TABLE_HEADER *entry, *start;
   unsigned short hash = 0;
   int i, len;

   for (i = 0, len = (sizeof (*tha) / sizeof (hash)); i < len; i++)
      hash += ((unsigned short *)tha)[i];

   if (entry = table->hash_array[hash % table->size]) {
      start = entry;
      do {
         if (entry) {
            if (!(bcmp ((char *)tha, (char *)&entry->tha, sizeof (*tha)))) {
               retn = entry->object;
               break;
            } else
               entry = entry->nxt;
         } else {
            aerror ("find_hash_entry: bad entry pointer", 0L);
         }
      } while (entry != start);
   }

   return (retn);
}


remove_hash_table_entry (table, tha)
struct HASH_TABLE *table;
struct tha *tha;
{
   struct HASH_TABLE_HEADER *retn = NULL;
   struct HASH_TABLE_HEADER *entry, *start;
   unsigned short hash = 0;
   int i, len;

   for (i = 0, len = (sizeof (*tha) / sizeof (hash)); i < len; i++)
      hash += ((unsigned short *)tha)[i];

   if (entry = table->hash_array[hash % table->size]) {
      start = entry;
      do {
         if (!(bcmp ((char *)tha, (char *)&entry->tha, sizeof (*tha)))) {
            retn = entry;
            break;
         }
         entry = entry->nxt;
      } while (entry != start);
   }

   if (retn) {
      retn->prv->nxt = retn->nxt;
      retn->nxt->prv = retn->prv;
      if (retn == start) {
         if (retn == retn->nxt)
            table->hash_array[hash % table->size] = NULL;
         else
            table->hash_array[hash % table->size] = retn->nxt;
      }
      free (retn);
   }
}


add_to_queue (queue, ptr)
struct QUEUE *queue;
struct QUEUE_HEADER *ptr;
{
   int retn = 0;
   struct QUEUE_HEADER *head;

   if (queue && ptr) {
      if (!(ptr->queue)) {
         ptr->queue = queue;
         if (head = queue->start) {
            if (head->prv && head->nxt) {
               ptr->nxt = head;
               ptr->prv = head->prv;
               ptr->nxt->prv = ptr;
               ptr->prv->nxt = ptr;
            } else aerror ("add_to_queue: head pointers bad", 0L);
         } else {
            ptr->prv = ptr;
            ptr->nxt = ptr;
         }
         queue->start = ptr;
         queue->count++;
         retn = 1;
      } else
         aerror ("add_to_queue: ptr->queue not NULL", 0L);
   } else
      aerror ("add_to_queue: bad parameters", 0L);

   return (retn);
}

void
remove_from_queue (queue, ptr)
struct QUEUE *queue;
struct QUEUE_HEADER *ptr;
{
   struct QUEUE *thisqueue;

   if (queue && ptr) {
      if (ptr->queue && (ptr->queue == queue)) {
         if (--queue->count > 0) {
            if (ptr->prv) ptr->prv->nxt = ptr->nxt; 
            else aerror ("remove_from_queue: ptr->prv is null", 0L);
            if (ptr->nxt) ptr->nxt->prv = ptr->prv; 
            else aerror ("remove_from_queue: ptr->nxt is null", 0L);
   
            if (ptr == queue->start)
                  queue->start = ptr->nxt;
         } else
            queue->start = NULL;
      } else if (ptr->queue)
                aerror ("remove_from_queue: remove from wrong queue", 0L);
             else
                aerror ("remove_from_queue: object not in queue", 0L);

      ptr->queue = NULL;
      ptr->prv = NULL;
      ptr->nxt = NULL;
   } else
      aerror ("remove_from_queue: bad parameters", 0L);
}


void
update_queue_status (ptr)
struct QUEUE_HEADER *ptr;
{
   struct QUEUE *queue;

   if (ptr) { 
      queue = ptr->queue;
      if (queue) {
         if (ptr != queue->start) {
            remove_from_queue (queue, ptr);
            if (!(add_to_queue (queue, ptr)))
               aerror ("update_queue_status: add_to_queue error", NULL);
         }
      } else aerror ("update_queue_status: queue is NULL", 0L);
   } else aerror ("update_queue_status: ptr is NULL", 0L);
}


#include <malloc.h> 
#include <string.h> 

char *
copy_argv (argv)
char **argv;
{
   char **p;
   int len = 0;
   char *buf = NULL, *src, *dst;

   p = argv;
   if (*p == 0) return 0;

   while (*p) len += (int) strlen (*p++) + 1;

   if (buf = calloc (1, len)) {
      p = argv;
      dst = buf;
      while ((src = *p++) != NULL) {
         while ((*dst++ = *src++) != '\0') ;
         dst[-1] = ' ';
      }

      dst[-1] = '\0';
   }
   return buf;
}

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

char *
read_infile (char *fname)
{
   int fd;
   char *p;
   struct stat buf;
 
   if ((fd = open(fname, O_RDONLY)) >= 0) {
      if (fstat(fd, &buf) < 0)
         aerror ("can't state '%s'", (int) fname);
 
      if (p = calloc (1, (u_int) buf.st_size))
         if (read (fd, p, (unsigned int)buf.st_size) != buf.st_size)
            aerror ("problem reading '%s'", (int) fname);
 
   } else
      aerror ("can't open '%s'", (int) fname);

   return (p);
}

unsigned short udpPorts [128];
unsigned short *udpServicePorts = NULL;

#define MAXSTRLEN  1024

read_udp_services (file)
char *file;
{
   FILE *fd;
   int port, i = 0;
   char *ptr = NULL, buffer[MAXSTRLEN];

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         bzero ((char *) udpPorts, sizeof (udpPorts));
         while ((i < 128) && fgets (buffer, MAXSTRLEN, fd))
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!'))
               if ((ptr = strtok (buffer, " \t")) && (*ptr != '\n'))
                  if ((ptr = strtok (NULL, " \t")) && (*ptr != '\n'))
                     if (strstr (ptr, "udp"))
                        if ((sscanf (ptr, "%d", &port)) == 1)
                           udpPorts[i++] = (unsigned short) port;
         if (i)
            udpServicePorts = udpPorts;

         fclose (fd);
      }
   }
}

#include <varargs.h>

void
aerror (fmt, va_alist)
char *fmt;
va_dcl
{
   va_list ap;

   (void) fprintf (stderr, "%s: ", program_name);
   va_start (ap);
   (void) vfprintf (stderr, fmt, ap);
   va_end (ap);
   if (*fmt) {
      fmt += (int) strlen (fmt);
      if (fmt[-1] != '\n')
         (void) fputc ('\n', stderr);
   }
   exit (1);
}

#ifdef SOLARIS
bcopy (s1, s2, len)
char *s1, *s2;
int len;
{
   memmove (s2, s1, len);
}

bcmp (s1, s2, len)
char *s1, *s2;
int len;
{
   return (memcmp (s1, s2, len));
}

bzero (s1, len)
char *s1;
int len;
{
   memset (s1, 0, len);
}
#endif
