
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_sockets.c - supports connection tracking 
 *
 * Used to handle remote client sockets.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 */

#include <stdlib.h>
#include <unistd.h>

#include <sys/socket.h>
#include <netinet/in.h>

#include <memory.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/errno.h>
 
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

#include <compat.h>
#include <pcap.h>
#include <interface.h>
#include <argus.h>

#define STDOUT           1
#define MONITORPORTNUM   561

static void close_socket ();
extern fd_set readmask, writemask, exceptmask;

int clienttags = 0;
static int clientsfd [MAX_LISTEN];
static struct sockaddr *clientsock[MAX_LISTEN];


void
cons_socket_init ()
{
   int i;
   for (i = 0; i < MAX_LISTEN; i++) {
      clientsfd[i] = -1;
      clientsock[i] = NULL;
   }
}


int firstWrite = 1;

void
writeOutData (output)
struct writeStruct *output;
{
   int i, fd;
   struct stat buf;
   char buffer[64];

   if (wfile && wflag) {
      if (strcmp (wfile, "-")) {
         if ((fd = open (wfile, O_RDWR | O_APPEND | O_CREAT, 0x1a4)) >= 0) {
            if (fstat (fd, &buf) == 0) {
               if (buf.st_size == 0) {
                  sprintf (buffer, "Argus Version %d.%d\n",
                                           VERSION_MAJOR, VERSION_MINOR);
                  if ((write (fd, buffer, strlen(buffer))) < 0)
                     perror ("write");
               }
            } else {
               perror ("fstat");
               close (fd); unlink (wfile);
               return;
            }
            if ((write (fd, output, sizeof (*output))) < 0)
               perror ("write");
            close (fd);
         } else
            perror ("open");

      } else {
         if (firstWrite) {
            fprintf (stdout, "Argus Version %d.%d\n",
                                           VERSION_MAJOR, VERSION_MINOR);
            firstWrite = 0;
         }
         if (!(fwrite (output, sizeof (struct writeStruct), 1, stdout)))
            perror ("fwrite");
         fflush (stdout);
      }
   }

   if (clienttags)
      for (i = 0; i < MAX_LISTEN; i++)
         if (clientsfd[i] != -1)
            if ((write (clientsfd[i], (char *) output,
                                sizeof(struct writeStruct))) <= 0)
               if ((errno == ENXIO) || (errno == EPIPE))
                  close_socket (i);
}


establish_listen (port)
int port;
{
   int s = -1;
   struct sockaddr_in sin;
   struct servent *sp;

   if (port) {
      if (sp = getservbyname ("monitor", "tcp"))
         sin.sin_port = sp->s_port;
      else 
         sin.sin_port = (u_short) port;
   
      sin.sin_addr.s_addr = INADDR_ANY;
      if (s = socket (AF_INET, SOCK_STREAM, 0)) {
         if ((fcntl (s, F_SETFL, O_NDELAY)) >= 0) {
            if (!(bind (s, (struct sockaddr *)&sin, sizeof(sin)))) {
               listen (s, MAX_LISTEN);
               FD_SET (s, &readmask);
               FD_SET (s, &writemask);
               FD_SET (s, &exceptmask);
            } else
               perror ("bind");
         }
      } else
         perror ("bind");
   }
   return (s);
}

#include <string.h>

void
check_client_status (lfd)
int lfd;
{
   struct sockaddr from;
   int fd, len = sizeof (from), i;
   char buf [1024];

   if ((fd = accept (lfd, (struct sockaddr *)&from, &len)) > 0) {
      if (tcp_wrapper (fd, &from) >= 0) {
         sprintf (buf, "Argus Version %d.%d",
                                  VERSION_MAJOR, VERSION_MINOR);
         if (dflag) strcat (buf, " detail mode\n");
         else strcat (buf, "\n");
            
         if (write (fd, buf, (unsigned int) strlen (buf)) <= 0) {
            close (fd);
            return;
         } else {
            for (i = 0; i < MAX_LISTEN; i++) {
               if (clientsfd[i] == -1) {
                  clientsfd[i] = fd;
                  if (clientsock[i] = (struct sockaddr *) calloc (1, 
                         sizeof (struct sockaddr)))
                     bcopy ((char *)&from, (char *) clientsock[i],
                                           sizeof (struct sockaddr));
                  clienttags |= (1 << i);
                  break;
               }
            }
            if (i == MAX_LISTEN)
               close (fd);
         }

      } else
         close (fd);
   }
}

static void
close_socket (i)
int i;
{
   close (clientsfd[i]);
   clientsfd[i] = -1;
   free (clientsock[i]);
   clientsock[i] = NULL;
   clienttags &= ~(1 << i);
}


void
close_clients ()
{
   extern int lfd;
   int i;

   for (i = 0; i < MAX_LISTEN; i++)
      if (clientsfd[i] != -1)
         close_socket (i);

   clienttags = 0;
   close (lfd);
}
