
/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */


/*
 * cons_udp.c - handle udp specific protocol state machine,
 *              and routines related to udp event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */


#include <stdlib.h>
#include <unistd.h>

#include <compat.h>
#include <pcap.h>
#include <interface.h>

#include <argus.h>
#include <cons_udp.h>

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;

static void new_udp_hash_entry ();
static void delete_udp ();
static void remove_udp_hash_entry ();



void
cons_udp_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &udp_display_list, sizeof (udp_display_list));
   bzero ((char *) &udp_hash_table, sizeof (udp_hash_table));

   if (ptr = (struct HASH_TABLE_HEADER **) calloc (USEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      udp_hash_table.hash_array = ptr;
      udp_hash_table.size = USEQ_HASHSIZE;
   } else
      perror ("cons_udp_init: calloc");

   udptimeout = udptimeout ? udptimeout : UDPTIMEOUT; 
   udp_display_list.timerRoutine = check_udp_timeouts;
   udp_display_list.logRoutine   = log_udp_connection;
}

void
cons_udp (ep, up, length, ip, tvp)
struct ether_header *ep;
struct udphdr *up;
int length;
struct ip *ip;
struct timeval *tvp;
{
   struct UDP_OBJECT *uh;
   int rev;
   struct tha tha;

   if ((length >= sizeof(struct udphdr)) && (up)) {
      create_udp_tha (&tha, ip, up, &rev);

      if (uh = (struct UDP_OBJECT *) find_hash_entry (&udp_hash_table, &tha))
         update_udp_record (uh, ep, up, (int)up->uh_ulen, tvp, rev);
      else
         new_udp_hash_entry (up, ep, rev, tvp, ip, &tha, (int)up->uh_ulen);
   }
}


create_udp_tha (tha, ip, up, rev)
struct tha *tha;
struct ip *ip;
struct udphdr *up;
int *rev;
{
   adjust_group_port (&up->uh_sport, &up->uh_dport);

   if ((up->uh_dport > up->uh_sport) || ((up->uh_dport == up->uh_sport) &&
            (ip->ip_src.s_addr < ip->ip_dst.s_addr))) {
      tha->src = ip->ip_dst, tha->dst = ip->ip_src;
      tha->port = up->uh_dport << 16 | up->uh_sport;
      *rev = 1;

   } else {
      tha->src = ip->ip_src, tha->dst = ip->ip_dst;
      tha->port = up->uh_sport << 16 | up->uh_dport;
      *rev = 0;
   }
}

adjust_group_port (sport, dport)
unsigned short *sport, *dport;
{
   int is_src_server = 0, is_dst_server = 0;
   extern unsigned short *udpServicePorts;
   unsigned short *ptr = udpServicePorts, *end = NULL;
 
   if (ptr && *ptr) {
      do {
         if (*sport == *ptr) is_src_server = 1;
         if (*dport == *ptr) is_dst_server = 1;
         ptr++;
      } while (*ptr);

      if ((is_src_server || is_dst_server) &&
                              !(is_src_server && is_dst_server)) {
         if (is_src_server) *dport = 0;
         if (is_dst_server) *sport = 0;
      }
   }
}


update_udp_record (uh, ep, up, length, tvp, rev)
struct UDP_OBJECT *uh;
struct ether_header *ep;
struct udphdr *up;
int length;
struct timeval *tvp;
int rev;
{
   struct udpcb *udpCb;

   udpCb = &uh->udp_cb;
   udpCb->lasttime = *tvp; uh->qhdr.last_time = *tvp;
   if (udpCb->rev == rev) {
      udpCb->src.count++;
      udpCb->src.bytes += length;
   } else {
      udpCb->dst.count++;
      udpCb->dst.bytes += length;
   }

   if (udpCb->status & MODIFIED) {
      if (!(udpCb->status & MULTIADDR)) {
         if (rev) {
            if (bcmp ((char *)&ep->ether_dhost,
                    (char *)&uh->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_shost,
                    (char *)&uh->link.phys.etherdst, 6))
               udpCb->status |= MULTIADDR;
         } else
            if (bcmp ((char *)&ep->ether_shost,
                    (char *)&uh->link.phys.ethersrc, 6) ||
                bcmp ((char *)&ep->ether_dhost,
                    (char *)&uh->link.phys.etherdst, 6))
               udpCb->status |= MULTIADDR;
      }

   } else {
      if (rev) {
         bcopy ((char *)&ep->ether_dhost, (char *)&uh->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_shost, (char *)&uh->link.phys.etherdst, 6);
      } else {
         bcopy ((char *)&ep->ether_shost, (char *)&uh->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&uh->link.phys.etherdst, 6);
      }
   }

   if ((udpCb->src.count + udpCb->dst.count) == 2)
      udpCb->status |= CON_ESTABLISHED;

   update_queue_status ((struct QUEUE_HEADER *) &uh->qhdr);
   udpCb->status |= MODIFIED;

   if (((tvp->tv_sec - udpCb->startime.tv_sec) > udptimeout) ||
         (((udpCb->src.count == 1) && (udpCb->dst.count == 1)) &&
              !(udpCb->status & LOGGED)))
      log_udp_connection (uh, tvp, CON_ESTABLISHED);
}
 

int udpdeletecount = 0;

static void
delete_udp (uh)
struct UDP_OBJECT *uh;
{
   udpdeletecount++;
   remove_hash_table_entry (&udp_hash_table, &uh->addr);
   free (uh);
}


static void
new_udp_hash_entry (up, ep, rev, tvp, ip, tha, len)
struct udphdr *up;
struct ether_header *ep;
int rev;
struct timeval *tvp;
struct ip *ip;
struct tha *tha;
int len;
{
   struct UDP_OBJECT *ptr = NULL;
   struct udpcb *udpCb;

   if (ptr = (struct UDP_OBJECT *) calloc (1, sizeof (*ptr))) {
      udpCb = &ptr->udp_cb;
      udpCb->status = UDPPROTO | UDP_INIT | MODIFIED;
      if (ip) udpCb->status |= IPPROTO;
      if (dflag) udpCb->status |= DETAIL;
      udpCb->startime = *tvp;
      udpCb->lasttime = *tvp;
      if (udpCb->rev = rev) {
         udpCb->status |= REVERSE;
         bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.etherdst, 6);

      } else {
         bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
      }

      ptr->qhdr.last_time = *tvp;
      bcopy ((char *) tha, (char *) &ptr->addr, sizeof (*tha));

      if (ip_options) { 
         if (ip_rr)   udpCb->status |= RECORDROUTE; 
         if (ip_ts)   udpCb->status |= TIMESTAMP;
         if (ip_sec)  udpCb->status |= SECURITY; 
         if (ip_lsrr) udpCb->status |= LSRCROUTE; 
         if (ip_ssrr) udpCb->status |= SSRCROUTE; 
         if (ip_satid) udpCb->status |= SATNETID; 
      }

      udpCb->src.count++;
      udpCb->src.bytes += len;

      if (add_to_queue (&udp_display_list, (struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&udp_hash_table, tha, ptr);
         if (dflag) {
            log_udp_connection (ptr, tvp, UDP_INIT);
            udpCb->status &= ~LOGGED;
         }
      } else
         delete_udp (ptr);
   }
}


#include <sys/stat.h>
#include <sys/errno.h>

void
log_udp_connection (ptr, tvp, state)
struct UDP_OBJECT *ptr;
struct timeval *tvp;
int state;
{
   struct writeStruct output;

   if (clienttags || wflag) {
      if ((ptr->udp_cb.src.count || ptr->udp_cb.dst.count) &&
              (ptr->udp_cb.status & MODIFIED)) {
         output.startime =  ptr->udp_cb.startime;
         output.lasttime =  ptr->udp_cb.lasttime;
         output.status   =  ptr->udp_cb.status | state;
         ptr->udp_cb.status &= ~MODIFIED;
         ptr->udp_cb.status &= ~MULTIADDR;
         output.addr     =  ptr->addr;
         bcopy ((char *)&ptr->link.phys.ethersrc, (char *)&output.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst, (char *)&output.etherdst, 6);
         output.src_count = ptr->udp_cb.src.count - ptr->udp_cb.src.lastcount;
         output.dst_count = ptr->udp_cb.dst.count - ptr->udp_cb.dst.lastcount;
         output.src_bytes = ptr->udp_cb.src.bytes - ptr->udp_cb.src.lastbytes;
         output.dst_bytes = ptr->udp_cb.dst.bytes - ptr->udp_cb.dst.lastbytes;
         if ((output.src_count || output.src_bytes) || 
                        (output.dst_count || output.dst_bytes)) {
            if (ptr->udp_cb.status & state) {
               writeOutData (&output);
               ptr->udp_cb.status |= LOGGED;
            }

            if (!(dflag && (state & UDP_INIT)) &&
                            (ptr->udp_cb.status & LOGGED)) {
               ptr->udp_cb.src.lastcount = ptr->udp_cb.src.count;
               ptr->udp_cb.src.lastbytes = ptr->udp_cb.src.bytes;
               ptr->udp_cb.dst.lastcount = ptr->udp_cb.dst.count;
               ptr->udp_cb.dst.lastbytes = ptr->udp_cb.dst.bytes;
               ptr->udp_cb.startime = *tvp; ptr->udp_cb.lasttime = *tvp;
            }
         }
      }
   }

   ptr->qhdr.logtime = *tvp;
}
   

void
check_udp_timeouts (ptr, tvp)
struct UDP_OBJECT *ptr;
struct timeval *tvp;
{
   int srccounts, dstcounts;

   if ((tvp->tv_sec - ptr->udp_cb.lasttime.tv_sec) > udptimeout) {
      if ((ptr->udp_cb.src.count || ptr->udp_cb.dst.count) &&
                                    !(ptr->udp_cb.status & LOGGED)) {
         ptr->udp_cb.status |= TIMED_OUT;
         log_udp_connection (ptr, tvp, TIMED_OUT);
      }
      remove_from_queue (&udp_display_list,
                          (struct QUEUE_HEADER *) &ptr->qhdr);
      delete_udp (ptr);
   } else {
      if ((tvp->tv_sec - ptr->udp_cb.startime.tv_sec) > udptimeout) {
         srccounts = (ptr->udp_cb.src.count - ptr->udp_cb.src.lastcount);
         dstcounts = (ptr->udp_cb.dst.count - ptr->udp_cb.dst.lastcount);
         if ((srccounts || dstcounts) &&
                              (ptr->udp_cb.status & CON_ESTABLISHED)) {
            log_udp_connection (ptr, tvp, CON_ESTABLISHED);
            ptr->qhdr.last_time = *tvp;
            update_queue_status ((struct QUEUE_HEADER *) &ptr->qhdr);
         }
      }
   }
}
