

/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Use in source and binary forms, with or without modification, is
 * permitted provided that source code modifications retain all
 * perintent copyright notices and this paragraph in its entirety.
 * This distribution includes software developed by Carnegie Mellon
 * University, the Software Engineering Institute and the CERT
 * Coordination Center.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Carter Bullard
 * CERT Coordiantion Center
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 *
 * dnstats - calculate DNS query stats on a server and client basis.
 *
 * written by Carter Bullard
 * Network Flow Sciences
 * Pittsburgh, PA.
 * November 1997.
 *
 */


#define ARGUS_CLIENT

#include <unistd.h>

#include <compat.h>

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#ifndef SOLARIS
#include <strings.h>
#endif
#include <string.h>
#include <time.h>
 
#include <netdb.h>

#include <math.h>
#include <signal.h>

#include <policy.h>

#include <netinet/in.h>

#include <argus_util.h>
#include <argus_parse.h>
#include <argus_client.h>
#include <services.h>

#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>
#include <nametoaddr.h>
#include <cons_icmp.h>



char *appOptstring = NULL;
extern int major_version, minor_version;

#define SRC  1
#define DST  2

struct QUEUE  srv_objects;
struct QUEUE  net_objects;
struct QUEUE  networks;

struct NETWORK        *net_hash_seq [TSEQ_HASHSIZE];
struct NET_OBJECT    *net_obj_hash_seq [TSEQ_HASHSIZE];
struct SRV_OBJECT *server_obj_hash_seq [TSEQ_HASHSIZE];
struct NET_OBJECT *client_obj_hash_seq [TSEQ_HASHSIZE];

extern arg_uint32 ipaddrtonetmask ();
extern arg_uint32 thisnet, localnet, localnetnumber, netmask;

struct NET_OBJECT *client_object_hash ();
struct SRV_OBJECT *server_object_hash ();
struct NET_OBJECT *net_object_hash ();
struct SRV_OBJECT *new_srv_object_hash ();
struct NET_OBJECT *new_net_object_hash ();

struct NETWORK *network_hash ();
struct NETWORK *new_network_hash_entry ();
struct NETWORK  this_phys_net;

int server_compare ();
int net_obj_compare ();

extern char *progname;

int argus_parse_complete ();
struct NET_OBJECT total_net_obj;

init ()
{
   network_db_init ();
   bzero ((char *) &net_objects, sizeof (struct QUEUE));
   bzero ((char *) &networks, sizeof (struct QUEUE));
   bzero ((char *) &this_phys_net, sizeof (struct NETWORK));
   bzero ((char *) &total_net_obj, sizeof (struct NET_OBJECT));
   (void) signal (SIGPIPE, SIG_IGN);
   (void) signal (SIGHUP,  (void (*)()) argus_parse_complete);
   (void) signal (SIGINT,  (void (*)()) argus_parse_complete);
   (void) signal (SIGTERM, (void (*)()) argus_parse_complete);
   (void) signal (SIGQUIT, (void (*)()) argus_parse_complete);

   dflag = sflag = 1;
}

int total_servers = 0;
int total_clients = 0;

argus_parse_complete ()
{
   struct PORT *ptr, *startPtr;
   putchar ('\n');

   if (total_stats.cons) {
      printf ("Total Nets         %6d\n", total_nets);
      printf ("Total Servers      %6d\n", total_servers);
      printf ("Total Clients      %6d\n", total_clients);
      print_stats_data ("Total DNS Queries  ", &total_stats);

      sort_servers (progname, &srv_objects);

   } else
      printf ("No connections seen\n");
   exit (0);
}


void clientTimeout () {}
parse_arg (argc, argv)
int argc;
char **argv;
{}

void
usage (ptr)
char *ptr;
{
   fprintf (stderr, "usage: %s [-bcfghlmnux] ", ptr);
   fprintf (stderr, "[-C access-file] [-d debug-interval] [-F file] [-P port]\n");
   fprintf (stderr, "          [-r input_file] [-s Services_file] [-S argus_server]\n");
   fprintf (stderr, "          [-t time_range] [-w output_file] expression.\n");
   fprintf (stderr, "options: b - dump packet-matching code.\n");
   fprintf (stderr, "         c - print packet and byte counts.\n");
   fprintf (stderr, "         C - specify Cisco access-file.\n");
   fprintf (stderr, "         d - specify detail mode, interval in secs.\n");
   fprintf (stderr, "         f - print local hostnames only\n");
   fprintf (stderr, "         F - use file to define filter expression.\n");
   fprintf (stderr, "         g - print difference in start and last time values.\n");
   fprintf (stderr, "         h - print help.\n");
   fprintf (stderr, "         l - print last time values [default is start time].\n");
   fprintf (stderr, "         m - print MAC addresses.\n");
   fprintf (stderr, "         n - don't convert numbers to names.\n");
   fprintf (stderr, "         P - specify remote argus port.\n");
   fprintf (stderr, "         r - read Argus input file.\n");
   fprintf (stderr, "         S - specify remote argus host.\n");
   fprintf (stderr, "         s - specify remote argus host.\n");
   fprintf (stderr, "         t - specify time_range for search.\n         ");
   fprintf (stderr, "      format:    ");
   fprintf (stderr, "timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                          timeSpecification: ");
   fprintf (stderr, "[mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             ");
   fprintf (stderr, " mm/dd[/yy]\n");
   fprintf (stderr, "         u - print msecs timestamps (*)\n");
   fprintf (stderr, "         w - write Argus output file.\n");
   fprintf (stderr, "         x - flip ports on input (*)\n");
   exit(1);
}
dnsProcess (ptr, seconds)
struct writeStruct *ptr;
double seconds;
{
   register struct SRV_OBJECT *server_obj;
   register struct NET_OBJECT *client_obj;
   register struct NETWORK *net;
   u_long client, server;

   client = ptr->addr.src.s_addr;
   server = ptr->addr.dst.s_addr;

   do_stats (ptr, NULL, 0, seconds);

   if (server_obj = server_object_hash (server)) {
      do_stats (ptr, server_obj, DST, seconds);
   }

   if (client_obj = client_object_hash (server_obj, client)) {
      if (net = network_hash (client)) {
         if (!(client_obj->net)) {
            client_obj->net = net;
            add_obj_to_net (client_obj, net);
         }

         if (!(client_obj->group))
            client_obj->group = &net->net;
      }
      do_stats (ptr, client_obj, SRC, seconds);
   }
}

process_udp (ptr)
struct writeStruct *ptr;
{
   double seconds;

   if (((unsigned short *) &ptr->addr.port)[1] == 53) {
      if ((ptr->src_count == 1) && (ptr->dst_count == 1)) {
         if (ptr->status & (CON_ESTABLISHED)) {
            seconds = (double)(((double)(ptr->lasttime.tv_sec-ptr->startime.tv_sec))
                     + ((ptr->lasttime.tv_usec - ptr->startime.tv_usec)/1000000.0));

            if (seconds < 180)
               dnsProcess (ptr, seconds);
         }
      }
   }
}


process_man (ptr) struct writeStruct *ptr; { }
process_icmp (ptr) struct writeStruct *ptr; { } 
process_tcp (ptr) struct writeStruct *ptr; { }
process_ip (ptr) struct writeStruct *ptr; { }
process_arp (ptr) struct writeStruct *ptr; { }


do_stats (ptr, net_obj, index, seconds)
struct writeStruct *ptr;
struct NET_OBJECT *net_obj;
int index;
double seconds;
{
   int i; 
   struct STATISTICS *stats;

   if (net_obj) {
      if (net_obj->group)
         do_stats (ptr, net_obj->group, index, seconds);

      switch (index) {
         case SRC:  stats = &net_obj->src_stats; break;
         case DST:  stats = &net_obj->dst_stats; break;
         default:   break;
      }
   } else
      stats = &total_stats;

   do_particular_stats (ptr, stats, seconds);

   if ((index == DST) || !(net_obj))
      do_port_stats (ptr, net_obj, seconds);
}


do_port_stats (ptr, net_obj, seconds)
struct writeStruct *ptr;
struct NET_OBJECT *net_obj;
double seconds;
{
   struct PORT *ports, *portStart, *port;
   struct STATISTICS *stats = NULL;
   unsigned short dstPort, srcPort;
   struct NET_OBJECT *object;

   srcPort = ((unsigned short *) &ptr->addr.port)[0];
   dstPort = ((unsigned short *) &ptr->addr.port)[1];

   object = (net_obj) ? net_obj : &total_net_obj;

   ports = portStart = object->ports;
   if (ports) {
      do  {
         if (ports->port >= dstPort)
            break;
         ports = ports->nxt;
      } while (ports != portStart);

      if (ports->port == dstPort)
         stats = &ports->stats;
   }

   if (!stats)
      if (port = (struct PORT *) calloc (1, sizeof (struct PORT))) {
         if (!ports) {
            port->prv = port;
            port->nxt = port;
            object->ports = port;
         } else {
            port->nxt = ports;
            port->prv = ports->prv;
            ports->prv = port;
            port->prv->nxt = port;

         }
         stats = &port->stats;
         port->port = dstPort;
         if (object->ports->port > dstPort) object->ports = port;
      }

   if (stats)
      do_particular_stats (ptr, stats, seconds);
}


do_particular_stats (ptr, stats, seconds) 
struct writeStruct *ptr;
struct STATISTICS *stats;
double seconds;
{
   double bppkt = 0.0;
   int src_count, dst_count, src_bytes, dst_bytes;
   
   src_count = (ptr->src_count > 0) ? ptr->src_count : 0;
   dst_count = (ptr->dst_count > 0) ? ptr->dst_count : 0;
   src_bytes = (ptr->src_bytes > 0) ? ptr->src_bytes : 0;
   dst_bytes = (ptr->dst_bytes > 0) ? ptr->dst_bytes : 0;

   stats->cons++;
   stats->secs += seconds;
   stats->secs_sqrd += pow (seconds, 2.0);

   stats->src.pkts += src_count; stats->dst.pkts += dst_count;
   stats->src.bytes += src_bytes; stats->dst.bytes += dst_bytes;
   
   stats->src.pkts_sqrd += pow (src_count, 2.0);
   stats->dst.pkts_sqrd += pow (dst_count, 2.0);
   
   stats->src.bytes_sqrd += pow (src_bytes, 2.0);
   stats->dst.bytes_sqrd += pow (dst_bytes, 2.0);
   
   if (src_count) {
      double bppkt = (double) src_bytes/src_count;
      stats->src.bytes_per_pkt += bppkt;
      stats->src.bytes_per_pkt_sqrd += pow (bppkt, 2.0);
   }
   if (dst_count) {
      double bppkt = (double) ptr->dst_bytes/ptr->dst_count;
      stats->dst.bytes_per_pkt += bppkt;
      stats->dst.bytes_per_pkt_sqrd += pow (bppkt, 2.0);
   }
}



#include <nametoaddr.h>

#include <ctype.h>

char output_string [256];


struct NET_OBJECT *client_object_hash (server, ip_addr)
struct SRV_OBJECT *server;
u_long ip_addr;
{
   register struct NET_OBJECT *net_obj = NULL;
   register struct IP_ENTRY *ip = NULL;
   register int i, n, found = 0;
   register u_long hash = 0;
   register u_char *ptr;

   for (i = 0, ptr = (u_char *) &ip_addr; i < sizeof (u_long); i++)
      hash += *ptr++;

   if (net_obj = server->obj_hash_seq [hash])
      for (; net_obj; net_obj = (struct NET_OBJECT *) net_obj->nxt) 
         if (net_obj->ip_addr->addr == ip_addr) {
            found = 1; break;
         }

   if (!found) {
      if (net_obj = new_net_object_hash (server, hash)) {
         total_clients++;
         if (ip = (struct IP_ENTRY *) calloc (1, sizeof (struct IP_ENTRY))) {
            ip->addr = ip_addr;
            net_obj->ip_addr = ip;
         } else {
            fprintf (stderr, "major problem\n");
         }

         add_to_queue (&net_objects, net_obj);
      }
   }

   return (net_obj);
}


struct SRV_OBJECT *server_object_hash (ip_addr)
u_long ip_addr;
{
   register struct SRV_OBJECT *srv_obj = NULL;
   register struct IP_ENTRY *ip = NULL;
   register int i, n, found = 0;
   register u_long hash = 0;
   register u_char *ptr;

   for (i = 0, ptr = (u_char *) &ip_addr; i < sizeof (u_long); i++)
      hash += *ptr++;

   if (srv_obj = server_obj_hash_seq [hash])
      for (; srv_obj; srv_obj = (struct SRV_OBJECT *) srv_obj->nxt) 
         if (srv_obj->ip_addr->addr == ip_addr) {
            found = 1; break;
         }

   if (!found) {
      if (srv_obj = new_srv_object_hash (hash)) {
         total_servers++;
         if (ip = (struct IP_ENTRY *) calloc (1, sizeof (struct IP_ENTRY))) {
            ip->addr = ip_addr;
            srv_obj->ip_addr = ip;
         } else {
            fprintf (stderr, "major problem\n");
         }

         add_to_queue (&srv_objects, srv_obj);
      }
   }

   return (srv_obj);
}



struct NET_OBJECT *new_net_object_hash (server, hash)
struct SRV_OBJECT *server;
u_long hash;
{
   register struct NET_OBJECT *net_obj = NULL;

   if (net_obj = (struct NET_OBJECT *) calloc (1, sizeof (struct NET_OBJECT))) {
      net_obj->nxt = server->obj_hash_seq [hash];
      server->obj_hash_seq [hash] = net_obj;
      net_obj->type = MACHINE;
      net_obj->index = total_hosts++;
   }

   return (net_obj);
}

struct SRV_OBJECT *new_srv_object_hash (hash)
u_long hash;
{
   register struct SRV_OBJECT *srv_obj = NULL;

   if (srv_obj = (struct SRV_OBJECT *) calloc (1, sizeof (struct SRV_OBJECT))) {
      srv_obj->nxt = server_obj_hash_seq [hash];
      server_obj_hash_seq [hash] = srv_obj;
      srv_obj->type = SERVER;
      srv_obj->index = total_hosts++;
   }

   return (srv_obj);
}



struct NETWORK *network_hash (ip_addr)
u_long ip_addr;
{
   register struct NETWORK *net = NULL;
   register struct IP_ENTRY *ip = NULL;
   register int found = 0;
   register u_long hash;
   register u_long mask, netnumber;

   mask = ipaddrtonetmask (ip_addr);
   netnumber = getnetnumber (ip_addr & mask);
   hash = netnumber % TSEQ_HASHSIZE;

   if (net = net_hash_seq [hash])
      for (; net; net = net->nxt)
         if (net->ip_net == netnumber) {
            found = 1; break;
         }

   if (!found) {
      if (net = new_network_hash_entry (hash)) {
         net->ip_mask = mask;
         net->ip_net = netnumber;
         net->net.net   = net;
         net->net.type = NET;
         add_to_queue (&networks, net);
      }
   }

   return (net);
}



struct NETWORK *new_network_hash_entry (hash)
u_long hash;
{
   register struct NETWORK *net = NULL;

   if (net = (struct NETWORK *) calloc (1, sizeof (struct NETWORK))) {
      net->nxt = net_hash_seq [hash];
      net->net.type = NET;
      net_hash_seq [hash] = net;
      total_nets++;
   }

   return (net);
}

add_obj_to_net (obj, net)
struct NET_OBJECT *obj;
struct NETWORK *net;
{
   struct NET_OBJECT_PTR *objPtr, *netObj, *startObj;

   if (obj && net) {
      if (startObj = netObj = net->ent) {
         do {
            if (netObj->net_obj->ip_addr->addr >= obj->ip_addr->addr)
               break;
            netObj = netObj->nxt;
         } while (netObj != startObj);
      }

      if (!netObj || 
          (netObj && netObj->net_obj->ip_addr->addr != obj->ip_addr->addr)) {
         if (objPtr = (struct NET_OBJECT_PTR *)
                   calloc (1, sizeof (struct NET_OBJECT_PTR))) {
            objPtr->net_obj = obj;
            objPtr->prv = objPtr; objPtr->nxt = objPtr;
         }
         if (netObj) {
            objPtr->nxt = netObj;
            objPtr->prv = netObj->prv;
            netObj->prv = objPtr;
            objPtr->prv->nxt = objPtr;
            if (startObj->net_obj->ip_addr->addr > obj->ip_addr->addr)
               net->ent = objPtr;
         } else
            net->ent = objPtr;
      }
   }
}


print_client_datum (queue)
struct QUEUE *queue;
{
   struct NET_OBJECT **array;
   struct NET_OBJECT *obj;
   u_long ip_addr;
   int i, count = queue->count;

   if (array = (struct NET_OBJECT **)
                         calloc (1, sizeof (struct NET_OBJECT *) * count)) {

      obj = (struct NET_OBJECT *) queue->start;
      for (i = 0; i < count; i++, obj = (struct NET_OBJECT *) obj->queue.nxt)
         array[i] = obj;
   }

   qsort ((char *) array, count, sizeof (struct SRV_OBJECT *), net_obj_compare);

   printf ("\n\n clients");
   for (i = 0; i < count; i++) {
      if (obj = array [i])
         if (obj->dst_stats.cons || obj->src_stats.cons) {
            ip_addr = obj->ip_addr->addr;
            printf ("\n%15.15s ", ipaddr_string (&ip_addr));
            if (obj->src_stats.cons)
               print_stats_data ("  ", &obj->src_stats);
         }
   }
}


print_server_datum (srv)
struct SRV_OBJECT *srv;
{
   u_long netnum, mask;

   printf ("\nServer");
   if (srv->dst_stats.cons || srv->src_stats.cons) {
      printf ("\n%15.15s ", ipaddr_string (&srv->ip_addr->addr));
      if (srv->dst_stats.cons) {
         print_stats_data ("  ", &srv->dst_stats);

         if (srv->src_stats.cons)
            printf ("\n%15.15s ", " ");
      }
   }
}


print_stats_data (string, stats)
char *string;
struct STATISTICS *stats;
{
   int cons = stats->cons, i;
   struct STAT *stat;
   double bytes, secs, secs_sqrd, bytes_sqrd, bytesvar, secsvar;

   printf ("%3s %5d  ", string, cons);

   for (i = 0; i < 2; i++) {
      stat = (i) ? &stats->dst : &stats->src;
      bytes = stat->bytes;
      bytes_sqrd = stat->bytes_sqrd;
      bytesvar = (double) bytes_sqrd/(double) cons
                                   - pow (bytes/(double) cons, 2.0);
      bytesvar = (bytesvar > 0) ? bytesvar : 0;
      printf ("%9.2f %7.2f ", bytes / (double) cons, sqrt (bytesvar));
   }

   secs = stats->secs;
   secs_sqrd = stats->secs_sqrd;
   secsvar = (double) secs_sqrd/(double) cons - pow (secs/(double) cons, 2.0);
   secsvar = (secsvar > 0) ? secsvar : 0;

   printf ("%8.2f %6.2f", stats->secs / (double) cons, sqrt (secsvar));
}

int
server_compare (ptr1, ptr2)
struct SRV_OBJECT **ptr1, **ptr2;
{
   register int retn = 0;
   
   if (!dflag) {
      if (!(retn = ((*ptr2)->src_stats.cons - (*ptr1)->src_stats.cons)))
         if (!(retn = ((*ptr2)->src_stats.src.pkts -
                                  (*ptr1)->src_stats.src.pkts)))
            retn = ((*ptr2)->src_stats.secs - (*ptr1)->src_stats.secs);

   } else
   if (!sflag) {
      if (!(retn = ((*ptr2)->dst_stats.cons - (*ptr1)->dst_stats.cons)))
         if (!(retn = ((*ptr2)->dst_stats.dst.pkts -
                                  (*ptr1)->dst_stats.dst.pkts)))
            retn = ((*ptr2)->dst_stats.secs - (*ptr1)->dst_stats.secs);

   } else 
      if (!(retn = (((*ptr2)->src_stats.cons +
                     (*ptr2)->dst_stats.cons) -
                    ((*ptr1)->src_stats.cons +
                     (*ptr1)->dst_stats.cons))))
         if (!(retn = (((*ptr2)->src_stats.src.pkts +
                        (*ptr2)->dst_stats.dst.pkts) -
                       ((*ptr1)->src_stats.src.pkts +
                        (*ptr1)->dst_stats.dst.pkts))))
            retn = (((*ptr2)->src_stats.secs +
                     (*ptr2)->dst_stats.secs) -
                    ((*ptr1)->src_stats.secs +
                     (*ptr1)->dst_stats.secs));

   return (retn);
}


int
net_obj_compare (ptr1, ptr2)
struct NET_OBJECT **ptr2, **ptr1;
{
   register int retn = 0;
   
   if (!sflag) {
      if (!(retn = ((*ptr2)->src_stats.cons - (*ptr1)->src_stats.cons)))
         if (!(retn = ((*ptr2)->src_stats.src.pkts -
                                           (*ptr1)->src_stats.src.pkts)))
            retn = ((*ptr2)->src_stats.secs - (*ptr1)->src_stats.secs);

   } else 
   if (!dflag) {
      if (!(retn = ((*ptr2)->dst_stats.cons - (*ptr1)->dst_stats.cons)))
        if (!(retn = ((*ptr2)->dst_stats.dst.pkts -
                                           (*ptr1)->dst_stats.dst.pkts)))
          retn = ((*ptr2)->dst_stats.secs - (*ptr1)->dst_stats.secs);

   } else
      if (!(retn = (((*ptr2)->src_stats.cons + (*ptr2)->dst_stats.cons) -
                   ((*ptr1)->src_stats.cons + (*ptr1)->dst_stats.cons))))
         if (!(retn = (((*ptr2)->src_stats.src.pkts +
                        (*ptr2)->dst_stats.dst.pkts) -
                       ((*ptr1)->src_stats.src.pkts +
                        (*ptr1)->dst_stats.dst.pkts))))
            retn = (((*ptr2)->src_stats.secs + (*ptr2)->dst_stats.secs) -
                    ((*ptr1)->src_stats.secs + (*ptr1)->dst_stats.secs));

   return (retn);
}


sort_servers (progname, queue)
char *progname;
struct QUEUE *queue;
{
   struct QUEUE *ptr;
   struct SRV_OBJECT *srvs;
   struct SRV_OBJECT **array;
   int i, count = queue->count;

   if (array = (struct SRV_OBJECT **)
                         calloc (1, sizeof (struct SRV_OBJECT *) * count)) {

      srvs = (struct SRV_OBJECT *) queue->start;
      for (i = 0; i < count; i++, srvs = (struct SRV_OBJECT *) srvs->queue.nxt)
         array[i] = srvs;
   }

   qsort ((char *) array, count, sizeof (struct SRV_OBJECT *), server_compare);

   printf ("\n\n  entity             cons     src bytes (sd)");
   printf ("    dst bytes (sd)      secs (sd)\n");

   for (i = 0; i < count; i++) {
      srvs = array [i];
      print_server_datum (srvs);

      clearClientQueue ();
      createClientQueue (srvs);
      print_client_datum (&net_objects);

      printf ("\n");
   }

   printf ("\n\n");
}



#define HASHNAMESIZE 4096
#define STAYOPEN     1

struct nnamemem {
   long addr;
   char *name;
   struct nnamemem *nxt;
};

extern struct nnamemem *nnametable [HASHNAMESIZE];

network_db_init ()
{
   register struct nnamemem *ptr;
   register struct netent *n;

   setnetent (STAYOPEN);
   while (n = getnetent ()) {
      if (ptr = (struct nnamemem *) calloc (1, sizeof (struct nnamemem))) {
         ptr->addr = n->n_net;
         ptr->name = strdup (n->n_name);
         ptr->nxt = nnametable[ptr->addr & (HASHNAMESIZE - 1)];
         nnametable[n->n_net & (HASHNAMESIZE - 1)] = ptr;
      }
   }
      
   endnetent ();
}


createClientQueue (srv)
struct SRV_OBJECT *srv;
{
   int i;
   struct NET_OBJECT *net_obj, *nxt_obj;

   for (i = 0; i < TSEQ_HASHSIZE; i++)
      if (net_obj = srv->obj_hash_seq [i]) {
         for (; net_obj; net_obj = (struct NET_OBJECT *) net_obj->nxt) {
            nxt_obj = net_obj->nxt;
            add_to_queue (&net_objects, net_obj);
         }
      }
}

clearClientQueue ()
{
   net_objects.start = NULL;
   net_objects.count = 0;
}
