

/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 *
 * services - list services encountered in connection log.
 *
 */

#define ARGUS_CLIENT

#include <unistd.h>

#include <compat.h>

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#ifndef SOLARIS
#include <strings.h>
#endif
#include <string.h>
#include <time.h>
 
#include <netdb.h>

#include <math.h>
#include <signal.h>

#include <policy.h>

#include <netinet/in.h>

#include <argus_util.h>
#include <argus_parse.h>
#include <argus_client.h>
#include <services.h>

#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>
#include <nametoaddr.h>
#include <cons_icmp.h>

char *appOptstring = NULL;
extern int major_version, minor_version;



#define SRC  1
#define DST  2

struct TCP_seq_hash *connections = NULL;
struct QUEUE  net_objects;
struct QUEUE  networks;

struct NETWORK        *net_hash_seq [TSEQ_HASHSIZE];
struct NET_OBJECT *net_obj_hash_seq [TSEQ_HASHSIZE];

extern arg_uint32 ipaddrtonetmask ();
extern arg_uint32 thisnet, localnet, localnetnumber, netmask;

struct NET_OBJECT *net_object_hash ();
struct NET_OBJECT *new_net_object_hash ();

struct NETWORK *network_hash ();
struct NETWORK *new_network_hash_entry ();
struct NETWORK  this_phys_net;

int network_compare ();
int net_obj_compare ();

extern char *progname;

struct NET_OBJECT total_net_obj;

void argus_parse_complete ();
void print_port_stats_data ();
void print_net_datum ();
void print_host_datum ();
void print_stats_data ();
void do_particular_stats (); 
void do_stats ();
void do_port_stats ();

void
init ()
{
   bzero ((char *) &net_objects, sizeof (struct QUEUE));
   bzero ((char *) &networks, sizeof (struct QUEUE));
   bzero ((char *) &this_phys_net, sizeof (struct NETWORK));
   bzero ((char *) &total_net_obj, sizeof (struct NET_OBJECT));
   (void) signal (SIGPIPE, SIG_IGN);
   (void) signal (SIGHUP,  (void (*)()) argus_parse_complete);
   (void) signal (SIGINT,  (void (*)()) argus_parse_complete);
   (void) signal (SIGTERM, (void (*)()) argus_parse_complete);
   (void) signal (SIGQUIT, (void (*)()) argus_parse_complete);

   dflag = sflag = 1;
}

void
argus_parse_complete ()
{
   struct PORT *ptr, *startPtr;
   putchar ('\n');

   if (total_stats.cons) {
      printf ("%s\n", cmdline);
      print_stats_data ("Total Cons        ", &total_stats);
      printf ("\n        port         cons      src b/pkt       dst b/pkt");
      printf ("         secs\n");
      printf ("                               mean(sd)         mean(sd)");
      printf ("       mean(sd)\n");

      print_port_stats_data ("     ", total_net_obj.ports);
      printf ("\n");
   } else
      printf ("No connections seen\n");
   exit (0);
}


void clientTimeout () {}
void parse_arg (argc, argv)
int argc;
char **argv;
{}
 
void
usage (ptr)
char *ptr;
{
   fprintf (stderr, "usage: %s [-bcfghlmnux] ", ptr);
   fprintf (stderr, "[-C access-file] [-d debug-interval] [-F file] [-P port]\n");
   fprintf (stderr, "          [-r input_file] [-s Services_file] [-S argus_server]\n");
   fprintf (stderr, "          [-t time_range] [-w output_file] expression.\n");
   fprintf (stderr, "options: b - dump packet-matching code.\n");
   fprintf (stderr, "         c - print packet and byte counts.\n");
   fprintf (stderr, "         C - specify Cisco access-file.\n");
   fprintf (stderr, "         d - specify detail mode, interval in secs.\n");
   fprintf (stderr, "         f - print local hostnames only\n");
   fprintf (stderr, "         F - use file to define filter expression.\n");
   fprintf (stderr, "         g - print difference in start and last time values.\n");
   fprintf (stderr, "         h - print help.\n");
   fprintf (stderr, "         l - print last time values [default is start time].\n");
   fprintf (stderr, "         m - print MAC addresses.\n");
   fprintf (stderr, "         n - don't convert numbers to names.\n");
   fprintf (stderr, "         P - specify remote argus port.\n");
   fprintf (stderr, "         r - read Argus input file.\n");
   fprintf (stderr, "         S - specify remote argus host.\n");
   fprintf (stderr, "         s - specify remote argus host.\n");
   fprintf (stderr, "         t - specify time_range for search.\n         ");
   fprintf (stderr, "      format:    ");
   fprintf (stderr, "timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                          timeSpecification: ");
   fprintf (stderr, "[mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             ");
   fprintf (stderr, " mm/dd[/yy]\n");
   fprintf (stderr, "         u - print msecs timestamps (*)\n");
   fprintf (stderr, "         w - write Argus output file.\n");
   fprintf (stderr, "         x - flip ports on input (*)\n");
   exit(1);
}

void
process (ptr, seconds)
struct WriteStruct *ptr;
double seconds;
{
   do_stats (ptr, NULL, 0, seconds);
}

void
process_man (ptr)
struct WriteStruct *ptr;
{
}

void
process_icmp (ptr)
struct WriteStruct *ptr;
{
}

void
process_tcp (ptr)
struct WriteStruct *ptr;
{
   double seconds;

   if ((!(ptr->status & DETAIL)) && (ptr->status & CON_ESTABLISHED)
               && (ptr->status & (SAW_SYN_SENT | SAW_SYN ))) {
      seconds = (double)(((double)(ptr->ws_ip.lasttime.tv_sec -
                                   ptr->ws_ip.startime.tv_sec)) +
                                 ((ptr->ws_ip.lasttime.tv_usec -
                                   ptr->ws_ip.startime.tv_usec)/1000000.0));
      process (ptr, seconds);
   }
}

void
process_udp (ptr)
struct WriteStruct *ptr;
{
   double seconds;

   if ((ptr->ws_ip_udp.src_count == 1) && (ptr->ws_ip_udp.dst_count == 1)) {
      seconds = (double)(((double)(ptr->ws_ip.lasttime.tv_sec -
                                   ptr->ws_ip.startime.tv_sec)) +
                                 ((ptr->ws_ip.lasttime.tv_usec -
                                   ptr->ws_ip.startime.tv_usec)/1000000.0));
      if (seconds < 120.0)
         process (ptr, seconds);
   }
}

void
process_ip (ptr)
struct WriteStruct *ptr;
{
}


void
do_stats (ptr, net_obj, index, seconds)
struct WriteStruct *ptr;
struct NET_OBJECT *net_obj;
int index;
double seconds;
{
   int i; 
   struct STATISTICS *stats;

   if (net_obj) {
      if (net_obj->group)
         do_stats (ptr, net_obj->group, index, seconds);

      switch (index) {
         case SRC:  stats = &net_obj->src_stats; break;
         case DST:  stats = &net_obj->dst_stats; break;
         default:   break;
      }
   } else
      stats = &total_stats;

   if (ptr->ws_ip_udp.src_bytes || ptr->ws_ip_udp.dst_bytes) {
      do_particular_stats (ptr, stats, seconds);

      if ((index == DST) || !(net_obj))
         do_port_stats (ptr, net_obj, seconds);
   }
}


void
do_port_stats (ptr, net_obj, seconds)
struct WriteStruct *ptr;
struct NET_OBJECT *net_obj;
double seconds;
{
   struct PORT *ports, *portStart, *port;
   struct STATISTICS *stats = NULL;
   unsigned short dstPort, srcPort, proto;
   struct NET_OBJECT *object;

   srcPort = ((unsigned short *) &ptr->ws_ip.port)[0];
   dstPort = ((unsigned short *) &ptr->ws_ip.port)[1];

   proto = (ptr->status & TCPPROTO) ? TCP :
           (ptr->status & UDPPROTO) ? UDP : 0;

   if ((srcPort == 20) || (srcPort >= 6000 && (srcPort <= 6010))) {
      srcPort = ((unsigned short *) &ptr->ws_ip.port)[1];
      dstPort = ((unsigned short *) &ptr->ws_ip.port)[0];
   }

   object = (net_obj) ? net_obj : &total_net_obj;

   ports = portStart = object->ports;
   if (ports) {
      do  {
         if ((ports->port > dstPort) ||
                ((ports->port == dstPort) && (ports->proto == proto)))
               break;
         ports = ports->nxt;
      } while (ports != portStart);

      if ((ports->port == dstPort) && (ports->proto == proto))
         stats = &ports->stats;
   }

   if (!stats)
      if (port = (struct PORT *) calloc (1, sizeof (struct PORT))) {
         if (!ports) {
            port->prv = port;
            port->nxt = port;
            object->ports = port;
         } else {
            port->nxt = ports;
            port->prv = ports->prv;
            ports->prv = port;
            port->prv->nxt = port;
 
         }
         stats = &port->stats;
         port->port = dstPort;
         port->proto = proto;
         if (object->ports->port > dstPort) object->ports = port;
      }

   if (stats)
      do_particular_stats (ptr, stats, seconds);
}


void
do_particular_stats (ptr, stats, seconds) 
struct WriteStruct *ptr;
struct STATISTICS *stats;
double seconds;
{
   double bppkt = 0.0;
   int src_count, dst_count, src_bytes, dst_bytes;
   
   src_count = (ptr->ws_ip_udp.src_count > 0) ? ptr->ws_ip_udp.src_count : 0;
   dst_count = (ptr->ws_ip_udp.dst_count > 0) ? ptr->ws_ip_udp.dst_count : 0;
   src_bytes = (ptr->ws_ip_udp.src_bytes > 0) ? ptr->ws_ip_udp.src_bytes : 0;
   dst_bytes = (ptr->ws_ip_udp.dst_bytes > 0) ? ptr->ws_ip_udp.dst_bytes : 0;

   stats->cons++;
   stats->secs += seconds;
   stats->secs_sqrd += pow (seconds, 2.0);

   stats->src.pkts += src_count; stats->dst.pkts += dst_count;
   stats->src.bytes += src_bytes; stats->dst.bytes += dst_bytes;
   
   stats->src.pkts_sqrd += pow (src_count, 2.0);
   stats->dst.pkts_sqrd += pow (dst_count, 2.0);
   
   stats->src.bytes_sqrd += pow (src_bytes, 2.0);
   stats->dst.bytes_sqrd += pow (dst_bytes, 2.0);
   
   if (src_count) {
      double bppkt = (double) src_bytes/src_count;
      stats->src.bytes_per_pkt += bppkt;
      stats->src.bytes_per_pkt_sqrd += pow (bppkt, 2.0);
   }
   if (dst_count) {
      double bppkt = (double) dst_bytes/dst_count;
      stats->dst.bytes_per_pkt += bppkt;
      stats->dst.bytes_per_pkt_sqrd += pow (bppkt, 2.0);
   }
}



#include <nametoaddr.h>

#include <ctype.h>

char output_string [256];


struct NET_OBJECT *net_object_hash (ip_addr)
arg_uint32 ip_addr;
{
   register struct NET_OBJECT *net_obj = NULL;
   register struct IP_ENTRY *ip = NULL;
   register int i, n, found = 0;
   register arg_uint32 hash = 0;
   register u_char *ptr;

   for (i = 0, ptr = (u_char *) &ip_addr; i < sizeof (arg_uint32); i++)
      hash += *ptr++;

   if (net_obj = net_obj_hash_seq [hash])
      for (; net_obj; net_obj = (struct NET_OBJECT *) net_obj->nxt) 
         if (net_obj->ip_addr->addr == ip_addr) {
            found = 1; break;
         }

   if (!found) {
      if (net_obj = new_net_object_hash (hash)) {
         if (ip = (struct IP_ENTRY *) calloc (1, sizeof (struct IP_ENTRY))) {
            ip->addr = ip_addr;
            net_obj->ip_addr = ip;
         }

         add_to_queue (&net_objects, net_obj);
      }
   }

   return (net_obj);
}



struct NET_OBJECT *new_net_object_hash (hash)
arg_uint32 hash;
{
   register struct NET_OBJECT *net_obj = NULL;

   if (net_obj = (struct NET_OBJECT *) calloc (1, sizeof (struct NET_OBJECT))) {
      net_obj->nxt = net_obj_hash_seq [hash];
      net_obj_hash_seq [hash] = net_obj;
      net_obj->type = MACHINE;
      net_obj->index = total_hosts++;
   }

   return (net_obj);
}



struct NETWORK *network_hash (ip_addr)
arg_uint32 ip_addr;
{
   register struct NETWORK *net = NULL;
   register int found = 0;
   register arg_uint32 hash;
   register arg_uint32 mask, netnumber;

   mask = ipaddrtonetmask (ip_addr);
   netnumber = getnetnumber (ip_addr & mask);
   hash = netnumber % TSEQ_HASHSIZE;

   if (net = net_hash_seq [hash])
      for (; net; net = net->nxt)
         if (net->ip_net == netnumber) {
            found = 1; break;
         }

   if (!found) {
      if (net = new_network_hash_entry (hash)) {
         net->ip_mask = mask;
         net->ip_net = netnumber;
         net->net.net   = net;
         net->net.type = NET;
         add_to_queue (&networks, net);
      }
   }

   return (net);
}



struct NETWORK *new_network_hash_entry (hash)
arg_uint32 hash;
{
   register struct NETWORK *net = NULL;

   if (net = (struct NETWORK *) calloc (1, sizeof (struct NETWORK))) {
      net->nxt = net_hash_seq [hash];
      net->net.type = NET;
      net_hash_seq [hash] = net;
      total_nets++;
   }

   return (net);
}


void
add_obj_to_net (obj, net)
struct NET_OBJECT *obj;
struct NETWORK *net;
{
   struct NET_OBJECT_PTR *objPtr, *netObj, *startObj;

   if (obj && net) {
      if (startObj = netObj = net->ent) {
         do {
            if (netObj->net_obj->ip_addr->addr >= obj->ip_addr->addr)
               break;
            netObj = netObj->nxt;
         } while (netObj != startObj);
      }

      if (!netObj || 
          (netObj && netObj->net_obj->ip_addr->addr != obj->ip_addr->addr)) {
         if (objPtr = (struct NET_OBJECT_PTR *)
                   calloc (1, sizeof (struct NET_OBJECT_PTR))) {
            objPtr->net_obj = obj;
            objPtr->prv = objPtr; objPtr->nxt = objPtr;
         }
         if (netObj) {
            objPtr->nxt = netObj;
            objPtr->prv = netObj->prv;
            netObj->prv = objPtr;
            objPtr->prv->nxt = objPtr;
            if (startObj->net_obj->ip_addr->addr > obj->ip_addr->addr)
               net->ent = objPtr;
         } else
            net->ent = objPtr;
      }
   }
}

void
print_host_datum (net)
struct NETWORK *net;
{
   struct NET_OBJECT_PTR *ent;

   ent = net->ent;

   if (ent) {
      do {
         printf ("\n%14.14s ", ipaddr_string (&ent->net_obj->ip_addr->addr));
         if (ent->net_obj->dst_stats.cons) {
            print_stats_data ("dst", &ent->net_obj->dst_stats);
            print_port_stats_data ("     ", ent->net_obj->ports);
            if (ent->net_obj->src_stats.cons)
               printf ("\n%14.14s ", " ");
         }
         if (ent->net_obj->src_stats.cons)
            print_stats_data ("src", &ent->net_obj->src_stats);

         ent = ent->nxt;
      } while (ent != net->ent);
   }
}

void
print_net_datum (net)
struct NETWORK *net;
{
   struct NET_OBJECT *ptr;

   ptr = &net->net;

   if (ptr->dst_stats.cons || ptr->src_stats.cons) {
      printf ("\n%14.14s ", ipaddr_string (net->ip_net));
      if (ptr->dst_stats.cons) {
         print_stats_data ("dst", &ptr->dst_stats);
         print_port_stats_data ("         ", ptr->ports);

         if (ptr->src_stats.cons)
            printf ("\n%14.14s ", " ");
      }

      if (ptr->src_stats.cons)
         print_stats_data ("src", &ptr->src_stats);
   }
}

void
print_port_stats_data (string, port)
char *string;
struct PORT *port;
{
   struct PORT *start;
   char str [256], *servstr, *protostr;

   if (start = port) {
      do {
         switch (port->proto) {
            case TCP: servstr = tcpport_string(port->port);
                      protostr = "tcp"; break;
            case UDP: servstr = udpport_string(port->port);
                      protostr = "udp"; break;
             default: servstr = " "; 
                      protostr = "unk"; break;
         }
         sprintf (str, "\n%s %s %6.6s  ", string, protostr, servstr);
         print_stats_data (str, &port->stats);
         port = port->nxt;
      } while (port != start);
   }
}


void
print_stats_data (string, stats)
char *string;
struct STATISTICS *stats;
{
   int cons = stats->cons, i;
   struct STAT *stat;
   double bppk, secs, secs_sqrd, bppk_sqrd, pktsvar, secsvar;
   char buf[256], *ptr, tmp[256];

   printf ("%3s %6d  ", string, cons);

   for (i = 0; i < 2; i++) { 
      stat = (i) ? &stats->dst : &stats->src;

      bppk = stat->bytes_per_pkt;
      bppk_sqrd = stat->bytes_per_pkt_sqrd, pktsvar;

      pktsvar = (double) bppk_sqrd/(double) cons
                                   - pow (bppk/(double) cons, 2.0);
      pktsvar = (pktsvar > 0) ? pktsvar : 0;
      sprintf (buf, "%7.2f", bppk / (double) cons);
      sprintf (tmp, "(%2.2f)", sqrt (pktsvar));
      ptr = buf + strlen (buf);
      sprintf (ptr, "%-9s", tmp);
      
      printf ("%s ", buf);
   }

   secs = stats->secs;
   secs_sqrd = stats->secs_sqrd;
   secsvar = (double) secs_sqrd/(double) cons - pow (secs/(double) cons, 2.0);
   secsvar = (secsvar > 0) ? secsvar : 0;

   sprintf (buf, "%7.2f", stats->secs / (double) cons);
   sprintf (tmp, "(%2.2f)", sqrt (secsvar));
   ptr = buf + strlen (buf);
   sprintf (ptr, "%-9s", tmp);
   printf ("%s ", buf);
}


int
network_compare (ptr1, ptr2)
struct NETWORK **ptr1, **ptr2;
{
   register int retn = 0;
   
   if (!dflag) {
      if (!(retn = ((*ptr2)->net.src_stats.cons - (*ptr1)->net.src_stats.cons)))
         if (!(retn = ((*ptr2)->net.src_stats.src.pkts -
                                  (*ptr1)->net.src_stats.src.pkts)))
            retn = ((*ptr2)->net.src_stats.secs - (*ptr1)->net.src_stats.secs);

   } else
   if (!sflag) {
      if (!(retn = ((*ptr2)->net.dst_stats.cons - (*ptr1)->net.dst_stats.cons)))
         if (!(retn = ((*ptr2)->net.dst_stats.dst.pkts -
                                  (*ptr1)->net.dst_stats.dst.pkts)))
            retn = ((*ptr2)->net.dst_stats.secs - (*ptr1)->net.dst_stats.secs);

   } else 
      if (!(retn = (((*ptr2)->net.src_stats.cons +
                     (*ptr2)->net.dst_stats.cons) -
                    ((*ptr1)->net.src_stats.cons +
                     (*ptr1)->net.dst_stats.cons))))
         if (!(retn = (((*ptr2)->net.src_stats.src.pkts +
                        (*ptr2)->net.dst_stats.dst.pkts) -
                       ((*ptr1)->net.src_stats.src.pkts +
                        (*ptr1)->net.dst_stats.dst.pkts))))
            retn = (((*ptr2)->net.src_stats.secs +
                     (*ptr2)->net.dst_stats.secs) -
                    ((*ptr1)->net.src_stats.secs +
                     (*ptr1)->net.dst_stats.secs));

   return (retn);
}


int
net_obj_compare (ptr1, ptr2)
struct NET_OBJECT **ptr2, **ptr1;
{
   register int retn = 0;
   
   if (!sflag) {
      if (!(retn = ((*ptr2)->src_stats.cons - (*ptr1)->src_stats.cons)))
         if (!(retn = ((*ptr2)->src_stats.src.pkts -
                                           (*ptr1)->src_stats.src.pkts)))
            retn = ((*ptr2)->src_stats.secs - (*ptr1)->src_stats.secs);

   } else 
   if (!dflag) {
      if (!(retn = ((*ptr2)->dst_stats.cons - (*ptr1)->dst_stats.cons)))
        if (!(retn = ((*ptr2)->dst_stats.dst.pkts -
                                           (*ptr1)->dst_stats.dst.pkts)))
          retn = ((*ptr2)->dst_stats.secs - (*ptr1)->dst_stats.secs);

   } else
      if (!(retn = (((*ptr2)->src_stats.cons + (*ptr2)->dst_stats.cons) -
                   ((*ptr1)->src_stats.cons + (*ptr1)->dst_stats.cons))))
         if (!(retn = (((*ptr2)->src_stats.src.pkts +
                        (*ptr2)->dst_stats.dst.pkts) -
                       ((*ptr1)->src_stats.src.pkts +
                        (*ptr1)->dst_stats.dst.pkts))))
            retn = (((*ptr2)->src_stats.secs + (*ptr2)->dst_stats.secs) -
                    ((*ptr1)->src_stats.secs + (*ptr1)->dst_stats.secs));

   return (retn);
}


void
sort_networks (argv, queue)
char **argv;
struct QUEUE *queue;
{

   register struct NETWORK *ptr;
   register struct NETWORK **array;
   register int i, count = queue->count;
   char startdate [32], enddate [32];
   double total_pktsvar, total_secsvar;


   if (array = (struct NETWORK **)calloc(1,sizeof(struct NETWORK *)*count)) {
      ptr = (struct NETWORK *) queue->start;
      for (i = 0; i < count; i++, ptr = (struct NETWORK *) ptr->queue.nxt)
         array[i] = ptr;
   }

   qsort ((char *) array, count, sizeof (struct NETWORK *), network_compare);

   for (i = 0; i < count; i++)
      print_host_datum (array [i]);

   printf ("\n\n");
}


#define HASHNAMESIZE 4096
#define STAYOPEN     1

struct nnamemem {
   arg_uint32 addr;
   char *name;
   struct nnamemem *nxt;
};

extern struct nnamemem *nnametable [HASHNAMESIZE];

void
network_db_init ()
{
   register struct nnamemem *ptr;
   register struct netent *n;

   setnetent (STAYOPEN);
   while (n = getnetent ()) {
      if (ptr = (struct nnamemem *) calloc (1, sizeof (struct nnamemem))) {
         ptr->addr = n->n_net;
         ptr->name = savestr (n->n_name);
         ptr->nxt = nnametable[ptr->addr & (HASHNAMESIZE - 1)];
         nnametable[n->n_net & (HASHNAMESIZE - 1)] = ptr;
      }
   }
      
   endnetent ();
}
