

/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * argus_parse - parse argus output.
 *       this module performs all the argus(1) related connection parsing,
 *       selects datum from a set of criteria, and then calls specific
 *       protocol dependant routines, depending on the selected datum.
 *       at the end of processing, argus_parse calls an application
 *       specific finish routine, argus_parse_complete(), and when
 *       connected to a remote data source, it supplies a periodic
 *       timeout routine;
 *
 *       this module defines all things, except:
 *
 *   (void) usage ((char *) argv[0]);
 *                    this routine should print the standard usage message
 *                    for the specific application.
 *
 *          init ();  this is the application specific init
 *                    routine, which is called after all parsing
 *                    initialization is done, prior to reading the
 *                    first monitor(1) datum.
 *
 *   (void) clientTimeout ();
 *                    this routine is called every second, when
 *                    argus_parse is connected to a remote data source.
 *
 *          process_man ((struct WriteStruct *) ptr);
 *                    this routine should process management control events;
 *
 *          process_tcp ((struct WriteStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_udp ((struct WriteStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_icmp ((struct WriteStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *          process_ip ((struct WriteStruct *) ptr);
 *                    this routine should process tcp events;
 *
 *   (void) argus_parse_complete ();
 *                    this routine will be called after all the
 *                    monitor data has been read.
 *
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */

#define MONITOR_PARSE

#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#endif

#include <stdio.h>

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#ifndef HAVE_SOLARIS
#include <strings.h>
#endif

#include <string.h>
#include <time.h>
#include <errno.h>
 
#include <netinet/in.h>
#include <arpa/inet.h>

#include <compat.h>
#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>

#include <argus_util.h>
#include <policy.h>
#include <argus_client.h>
#include <argus_parse.h>

#include <netinet/ip_icmp.h>

#include <cons_def.h>
#include <cons_icmp.h>
#include <cons_ip.h>
#include <cons_udp.h>

extern init ();
extern void usage ();
extern parse_arg ();
extern process_man ();
extern process_tcp ();
extern process_icmp ();
extern process_udp ();
extern process_ip ();
extern char *copy_argv ();
extern void argus_parse_complete ();
extern char *appOptstring;


#include <nametoaddr.h>

void remove_from_queue ();
void add_to_queue ();
void version_process ();
void read_remote ();
void adjust_group_port ();
void handle_datum ();
void wsntoh ();
void reformat_datum ();
void read_udp_services ();

struct policyStruct *policy = NULL;
char *policyfile =  NULL;
char *wfile =  NULL;

struct FILE_ENTRY *input_file_list = NULL;
struct IP_ENTRY *remote_host_list = NULL;

char *tag_string = "Argus Version ";
int major_version = VERSION_MAJOR;
int minor_version = VERSION_MINOR;
int detail = 0;
struct WriteStruct *initCon = NULL;
int read_mode = 0;

#define MAXSTR  256
#define MAXTIME 100000
#define READ_REMOTE_CON 0x40000000
#define READ_LOCAL_CON  0x20000000

arg_uint32 localnet, netmask;

struct tm starTimeFilter;
struct tm lastTimeFilter;

char *progname = NULL;
char *timearg = NULL;
char *servicesfile = NULL;
char cmdline[256]; 

extern char *argus_lookupdev();
int dowriteout = 1;


void
argus_parse_init (fd)
int fd;
{
   char errbuf[256];
   char *device = NULL;
   struct tm *tm;
   struct argtimeval tvpbuf, *tvp = &tvpbuf;
   arg_uint32 net, mask;

   if (initCon) {
      if (initCon->status & DETAIL) detail++;
      localnet = htonl(initCon->ws_init.localnet);
      netmask = htonl(initCon->ws_init.netmask);
      if (tflag && timearg) {
         tvp->tv_sec  = ntohl(initCon->ws_init.now.tv_sec);
         tm = localtime((time_t *) &tvp->tv_sec);
         check_time_format (tm, timearg);
      }
   } else {
      if (device = argus_lookupdev (errbuf)) {
         argus_lookupnet(device, &net, &mask, errbuf);
         localnet = net;
         netmask = mask;
      }
   }
      
   init_addrtoname (fflag, localnet, netmask);
}


#include <pcap-int.h>

int portnum = 0;
char *getoptStr = "bcC:d:F:fghlnmP:r:S:s:t:uw:x";

main (argc, argv)
int argc;
char **argv;
{
   int i, op, retn = 0, fd = 0;
   struct bpf_program bpfcode;
   struct FILE_ENTRY *ptr;
   char *cmdbuf = NULL, *infile = NULL, *read_infile ();
   struct timeval now;
   struct timezone tz;
   struct tm *tm;
   pcap_t p;
   extern char *optarg;
   extern int optind, opterr;

   opterr = 0;
   for (i = 0, *cmdline = '\0'; i < argc; i++) {
      strcat (cmdline, argv[i]);
      strcat (cmdline, " ");
   }
   if (strchr (argv[0], '/'))
      argv[0] = strrchr(argv[0], '/') + 1;

   if (gettimeofday(&now, &tz) < 0)
      error("gettimeofday");

   thiszone = tz.tz_minuteswest * -60;
   if ((tm = localtime ((time_t *)&now.tv_sec))->tm_isdst)
      thiszone += 3600;

   progname = argv[0];
   while ((op = getopt (argc, argv, getoptStr)) != EOF) {
      switch (op) {
         case 'b': ++bflag; break;
         case 'c': ++cflag; break;
         case 'C': policyfile = optarg; break;
         case 'd':  debugflag = atoi (optarg); break;
         case 'f': ++fflag; break;
         case 'F': infile = optarg; break;
         case 'g': ++gflag; break;
         case 'l': ++lflag; break;
         case 'm': ++mflag; break;
         case 'n': ++nflag; break;
         case 'P':  portnum = atoi (optarg); break;
         case 'r': ++rflag; 
            if (optarg == NULL) optarg = "-";
            if (!(new_file_arg (optarg))) {
               fprintf (stderr, "%s: error: file arg %s \n", *argv, optarg);
               exit (1);
            }
            break;
         case 's': servicesfile = optarg; break;
         case 'S': ++Sflag;
            if (!(remote_host_arg (optarg))) {
               fprintf (stderr, "%s: host %s unknown\n", *argv, optarg);
               exit (1);
            }
            break;

         case 't': ++tflag; 
            if (timearg = optarg)
               if ((retn = parseTimeArg (&timearg, argv, optind, tm)) < 0)
                  usage (argv[0]);
               else {
                  optind += retn;
               }
            break;
         case 'u': ++uflag; break;
         case 'w':  
            if ((wfile = optarg) == NULL)
               if (!strcmp (argv[optind], "-")) {
                  wfile = "-";
               }
            break;
	 case 'x': ++xflag; break;
         case 'h':
            default:  
               usage (argv[0]);
            /* NOTREACHED */
      }
   }
 
   if (infile)
      cmdbuf = read_infile (infile);
   else
      cmdbuf = copy_argv (&argv[optind]);

   if (servicesfile) read_udp_services (servicesfile);
   if (readPolicy (&policy, policyfile)) {
      bzero (&bpfcode, sizeof (bpfcode));
      if (policy_compile (&p, &bpfcode, cmdbuf, 1, netmask) < 0) {
         fprintf (stderr, "%s: expression: %s\n", progname, p.errbuf);
         exit (-1);
      }
      if (bflag) {
         bpf_dump(&bpfcode, bflag);
         exit (0);
      }
      if (Sflag) {
         if ((fd = read_remote_connection (get_server_socket (remote_host_list),
                        &bpfcode)) < 0)
            fprintf (stderr, "%s: connection failed\n", *argv);
      } else {
         if (ptr = input_file_list) {
            fd = read_connection (ptr->str, &bpfcode);
         } else
            fd = read_connection (NULL, &bpfcode);
      }
      if (fd >= 0) {
         if (Sflag)
            read_remote (fd, &bpfcode);
         else
            read_file (fd, &bpfcode);
         argus_parse_complete ();
      } else
         retn = 1;
   }
   exit (retn);
}

readPolicy (policy, file)
struct policyStruct **policy;
char *file;
{
   register int retn = 1, linenum = 0, parse;
   register struct policyStruct *pol, *policyLast = NULL;
   register FILE *fd;
   char buffer [1024], *str;

   if (file) {
      if (fd = fopen (file, "r")) {
         while (fgets (buffer, 1024, fd)) {
            linenum++;
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!')) {
               if (pol = (struct policyStruct *) calloc (1, 
                                sizeof (struct policyStruct))) {
                  pol->str = savestr (buffer);
                  if (!(parsePolicy (pol, buffer))) {
                     free (pol->str);
                     free (pol);
                     str = parseerrorstr[parseerror];
                     fprintf (stderr, "%s: parsing error: %s\n",
                                           progname, file);
                     fprintf (stderr, "\tline number %d: %s\n", linenum, str);
                     retn = 0;
                  }
                  if (policyLast)  {
                     policyLast->nxt = pol;
                     pol->prv = policyLast;
                     policyLast = pol;
                  } else
                     *policy = policyLast = pol;
               }
            }
         }
         fclose (fd);
      } else {
         retn = 0;
         fprintf (stderr, "%s: error couldn't open %s\n", progname, file);
      }
   }

   return (retn);
}

arg_uint16 udpPorts [128];
arg_uint16 *udpServicePorts = NULL;

#define MAXSTRLEN  1024

void
read_udp_services (file)
char *file;
{
   FILE *fd;
   int port, i = 0;
   char *ptr = NULL, buffer[MAXSTRLEN];

   if (file) {   
      if ((fd = fopen (file, "r")) != NULL) {
         bzero ((char *) udpPorts, sizeof (udpPorts));
         while ((i < 128) && fgets (buffer, MAXSTRLEN, fd))
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!'))
               if ((ptr = strtok (buffer, " \t")) && (*ptr != '\n'))
                  if ((ptr = strtok (NULL, " \t")) && (*ptr != '\n'))
                     if (strstr (ptr, "udp"))
                        if ((sscanf (ptr, "%d", &port)) == 1)
                           udpPorts[i++] = (arg_uint16) port;
         if (i)
            udpServicePorts = udpPorts;
              
         fclose (fd);
      }
   }
}     


#include <string.h>

parsePolicy (policy, buf)
struct policyStruct *policy;
char *buf;
{
   register int retn = 1, error = 0, i;
   register char *ptr;

   if (strncmp (buf, "access", 6)) {
      if (strstr (buf, "ip"))
         if (strstr (buf, "source-route"))
            policy->type = SSRCROUTE;
   } else {
      policy->type = ACCESSLIST;
      for (i = 0; ((i < POLICYFIELDNUM) && !(error)); i++) {
         if ((ptr = strtok (buf, " ")) && (*ptr != '\n')) {
            switch (i) {
               case POLICYSTRING: 
                  if ((strcmp (ptr, POLICY_STRING))) error++; 
                  break;
   
               case POLICYID: 
                  if (!(policy->policyID = atoi (ptr))) error++;
                  break;
   
               case POLICYACTION: 
                  if (!(strcmp (ptr, "permit")))
                     policy->flags |= PERMIT;
                  else if (!(strcmp (ptr, "deny")))
                     policy->flags |= DENY;
                  else error++;
                  break;
   
               case POLICYPROTO: 
                  if (!(strcmp (ptr, "icmp")))
                     policy->flags |= ICMP;
                  else if (!(strcmp (ptr, "tcp")))
                     policy->flags |= TCP;
                  else if (!(strcmp (ptr, "udp")))
                     policy->flags |= UDP;
                  else if (!(strcmp (ptr, "ip")))
                     policy->flags |= IP;
                  else error++;
                  break;
   
               case POLICYDSTADDR: 
                  policy->dst.addr = inet_addr (ptr); break;
               case POLICYDSTMASK: 
                  policy->dst.mask = inet_addr (ptr); break;
               case POLICYSRCADDR: 
                  policy->src.addr = inet_addr (ptr); break;
               case POLICYSRCMASK: 
                  policy->src.mask = inet_addr (ptr); break;
   
               case POLICYPORTACT: 
                  if (!(strcmp (ptr, "eq")))
                     policy->action |= EQ;
                  else if (!(strcmp (ptr, "lt")))
                     policy->action |= LT;
                  else if (!(strcmp (ptr, "gt")))
                     policy->action |= GT;
                  else if (!(strcmp (ptr, "neq")))
                     policy->action |= NEQ;
                  else if (!(strncmp (ptr, "established",11)))
                     policy->action |= EST;
                  else error++;
                  break;
   
               case POLICYPORT: 
                  if (!(policy->port = (arg_uint16) atoi (ptr))) error++;
                  break;
   
               case POLICYNOTIFICATION: 
                  break;
            }
   
            buf = NULL;
         } else
            break;
      }
   }

   if (error) {
      parseerror = i - 1; retn = 0;
   }

   return (retn);
}


check_policy (ptr, policy)
struct WriteStruct *ptr;
struct policyStruct *policy;
{
   int retn = 1, i, policymatch = 0;

   if (policy) {
      while (policy) {
         if (retn = meets_policy_criteria (ptr, policy)) {
            retn = do_notification (ptr, policy);
            policymatch = 1;
            break;
         }
         policy = policy->nxt;
      }
   }

   return (retn);
}


meets_policy_criteria (ptr, policy)
struct WriteStruct *ptr;
struct policyStruct *policy;
{
   register int retn = 0, i = 0;
   arg_uint32 saddr = 0, daddr = 0, addr = 0;
   arg_uint16 sport = 0, dport = 0, port = 0;

   switch (policy->type) {
   case SSRCROUTE:
      if ((ptr->status & SSRCROUTE) || (ptr->status & LSRCROUTE))
         retn++;
      break;

   case ACCESSLIST:
      if ((ptr->status & ICMPPROTO) || (ptr->status & UDPPROTO) ||
         ((ptr->status & TCPPROTO) && !(ptr->status & REVERSE))) {
         saddr = ptr->ws_ip.src.s_addr;
         daddr = ptr->ws_ip.dst.s_addr;
         sport = ((arg_uint16 *)&ptr->ws_ip.port)[0];
         dport = ((arg_uint16 *)&ptr->ws_ip.port)[1];
         port = dport;
      } else {
         saddr = ptr->ws_ip.dst.s_addr;
         daddr = ptr->ws_ip.src.s_addr;
         sport = ((arg_uint16 *)&ptr->ws_ip.port)[1];
         dport = ((arg_uint16 *)&ptr->ws_ip.port)[0];
         port = dport;
      }
   
      for (i = 0, retn = 1; ((i < POLICYTESTCRITERIA) && retn); i++) {
         retn = 0;
         switch (i) {
            case POLICYTESTPROTO:
               switch (policy->flags & (UDP | TCP | ICMP | IP)) {
                  case  UDP: if (ptr->status &  UDPPROTO) retn++; break;
                  case  TCP: if (ptr->status &  TCPPROTO) retn++; break;
                  case ICMP: if (ptr->status & ICMPPROTO) retn++; break;
                  case   IP: retn++; break;
               }
               break;
   
            case POLICYTESTSRC:
               if ((saddr & ~policy->src.mask) == policy->src.addr)
                  retn++;
               break;
            case POLICYTESTDST:
               if ((daddr & ~policy->dst.mask) == policy->dst.addr)
                  retn++;
               break;
            case POLICYTESTPORT:
               switch (policy->action) {
                  case  EQ:
                     if (port == policy->port)
                        retn++; break;
                  case  LT:
                     if (port < policy->port)
                        retn++; break;
                  case  GT:
                     if (port > policy->port)
                        retn++; break;
                  case NEQ:
                     if (port != policy->port)
                        retn++; break;
                  case  EST:
                     if (!(ptr->status & SAW_SYN))
                        retn++; break;
                  default:
                     retn++; break;
               }
               break;
         }
      }
   }

   return (retn);
}


do_notification (ptr, policy)
struct WriteStruct *ptr;
struct policyStruct *policy;
{
   register int retn = 1;

   if (policy)
      if (policy->flags & PERMIT) {
         if (debugflag > 1)
            printf ("%s %s", "policy: permitted", policy->str);
         else
            retn = 0;
      } else
         if (debugflag)
            printf ("%s %s", "policy: denyed", policy->str);

   return (retn);
}

 

struct naddrmem *naddrtable [HASHNAMESIZE];

struct naddrmem {
   struct naddrmem *nxt;
   arg_uint32 addr;
   arg_uint16 port;
};


struct WriteStruct wsOriginal;
int totalrecords = 0;
int argusrecords = 0;


void
handle_datum (ptr, bpfcode)
struct WriteStruct *ptr;
struct bpf_program *bpfcode;
{
   int retn;
   unsigned int status;
   struct WriteStruct buf;
   struct bpf_insn *fcode = bpfcode->bf_insns;

   totalrecords++;
   bcopy ((char *)ptr, (char *)&wsOriginal, sizeof (*ptr));
   if (!(ntohl(ptr->status) & ARGUSCONTROL)) {
      argusrecords++;
      reformat_datum (ptr, &buf);
   } else
      bcopy ((char *)ptr, (char *)&buf, sizeof (*ptr));

   if (retn = argus_filter (fcode, &buf)) {
	 wsntoh (&buf);	 
      if (retn = check_time (&buf)) {
         if (retn = check_policy (&buf, policy)) {
            if (wfile) {
               if (dowriteout) {
                  if (writeNewLogfile (&wsOriginal)) {
                     fprintf (stderr, "writeNewLogfile: error\n");
                     exit (1);
                  }
               } else
                  version_process (&buf);
            } else
               version_process (&buf);
         }
      }
   }
}


void
wsntoh (ws)
struct WriteStruct *ws;
{
   arg_uint32 status;

   ws->status = ntohl(ws->status);
   if ((status = ws->status) & ARGUSCONTROL) {
      switch (status & (INIT | STATUS)) {
         case INIT:
            ws->ws_init.now.tv_sec  = ntohl(ws->ws_init.now.tv_sec);
            ws->ws_init.now.tv_usec = ntohl(ws->ws_init.now.tv_usec);
            ws->ws_init.startime.tv_sec  = ntohl(ws->ws_init.startime.tv_sec);
            ws->ws_init.startime.tv_usec = ntohl(ws->ws_init.startime.tv_usec);
            ws->ws_init.localnet = ntohl(ws->ws_init.localnet);
            ws->ws_init.netmask = ntohl(ws->ws_init.netmask);
            break;

         case STATUS:
            ws->ws_stat.startime.tv_sec  = ntohl(ws->ws_stat.startime.tv_sec);
            ws->ws_stat.startime.tv_usec = ntohl(ws->ws_stat.startime.tv_usec);
            ws->ws_stat.now.tv_sec  = ntohl(ws->ws_stat.now.tv_sec);
            ws->ws_stat.now.tv_usec = ntohl(ws->ws_stat.now.tv_usec);
            ws->ws_stat.pktsRcvd = ntohl(ws->ws_stat.pktsRcvd);
            ws->ws_stat.bytesRcvd = ntohl(ws->ws_stat.bytesRcvd);
            ws->ws_stat.pktsDrop = ntohl(ws->ws_stat.pktsDrop);
            ws->ws_stat.actTCPcons = ntohs(ws->ws_stat.actTCPcons);
            ws->ws_stat.cloTCPcons = ntohs(ws->ws_stat.cloTCPcons);
            ws->ws_stat.actUDPcons = ntohs(ws->ws_stat.actUDPcons);
            ws->ws_stat.cloUDPcons = ntohs(ws->ws_stat.cloUDPcons);
            ws->ws_stat.cloIPcons = ntohs(ws->ws_stat.cloIPcons);
            ws->ws_stat.actIPcons = ntohs(ws->ws_stat.actIPcons);
            ws->ws_stat.cloICMPcons = ntohs(ws->ws_stat.cloICMPcons);
            ws->ws_stat.actICMPcons = ntohs(ws->ws_stat.actICMPcons);
            ws->ws_stat.cloFRAGcons = ntohs(ws->ws_stat.cloFRAGcons);
            ws->ws_stat.actFRAGcons = ntohs(ws->ws_stat.actFRAGcons);
            break;
      }
   } else {
      arg_uint32 newvalue;
      arg_uint16 *ptr = (unsigned short *)&newvalue;

      ws->ws_ip.startime.tv_sec  = ntohl(ws->ws_ip.startime.tv_sec);
      ws->ws_ip.startime.tv_usec = ntohl(ws->ws_ip.startime.tv_usec);

      if ((major_version > 1) || (minor_version > 6)) {
         ptr[0] = ntohs(((unsigned short *)&ws->ws_ip.lasttime.tv_sec)[0]);
         ptr[1] = ntohs(((unsigned short *)&ws->ws_ip.lasttime.tv_sec)[1]);
         ws->ws_ip.lasttime.tv_sec = newvalue;
      } else {
         ws->ws_ip.lasttime.tv_sec  = ntohl(ws->ws_ip.lasttime.tv_sec);
      }

      ws->ws_ip.lasttime.tv_usec  = ntohl(ws->ws_ip.lasttime.tv_usec);

      ws->ws_ip.src.s_addr = ntohl(ws->ws_ip.src.s_addr);
      ws->ws_ip.dst.s_addr = ntohl(ws->ws_ip.dst.s_addr);

      ptr[0] = ntohs(((unsigned short *)&ws->ws_ip.port)[0]);
      ptr[1] = ntohs(((unsigned short *)&ws->ws_ip.port)[1]);

      switch (status & (ICMPPROTO | TCPPROTO | UDPPROTO)) {
         case TCPPROTO:
            ws->ws_ip.port = newvalue;
            ws->ws_ip_tcp.src_count = ntohl(ws->ws_ip_tcp.src_count);
            ws->ws_ip_tcp.dst_count = ntohl(ws->ws_ip_tcp.dst_count);
            ws->ws_ip_tcp.src_bytes = ntohl(ws->ws_ip_tcp.src_bytes);
            ws->ws_ip_tcp.dst_bytes = ntohl(ws->ws_ip_tcp.dst_bytes);
            break;

         case ICMPPROTO:
            if (!(status & (CON_ESTABLISHED | TIMED_OUT))) {
               ws->ws_ip_icmp.data = ntohs(ws->ws_ip_icmp.data);
               ws->ws_ip_icmp.srcaddr.s_addr=ntohl(ws->ws_ip_icmp.srcaddr.s_addr);
               ws->ws_ip_icmp.dstaddr.s_addr=ntohl(ws->ws_ip_icmp.dstaddr.s_addr);
               ws->ws_ip_icmp.gwaddr.s_addr =ntohl(ws->ws_ip_icmp.gwaddr.s_addr);
            } else {
               ws->ws_ip_udp.src_count = ntohl(ws->ws_ip_udp.src_count);
               ws->ws_ip_udp.dst_count = ntohl(ws->ws_ip_udp.dst_count);
               ((unsigned short *)&ws->ws_ip_udp.src_bytes)[1] = 
                                        ntohs(((unsigned short *)&ws->ws_ip_udp.src_bytes)[1]);
               ws->ws_ip_udp.dst_bytes = ntohl(ws->ws_ip_udp.dst_bytes);
            }
            break;

         case UDPPROTO:
            ws->ws_ip.port = newvalue;
         default:
            ws->ws_ip_udp.src_count = ntohl(ws->ws_ip_udp.src_count);
            ws->ws_ip_udp.dst_count = ntohl(ws->ws_ip_udp.dst_count);
            ws->ws_ip_udp.src_bytes = ntohl(ws->ws_ip_udp.src_bytes);
            ws->ws_ip_udp.dst_bytes = ntohl(ws->ws_ip_udp.dst_bytes);
            break;
      }
   }
}

void
wshton (ws)
struct WriteStruct *ws;
{
   arg_uint32 status = ws->status;

   ws->status = htonl(ws->status);
   if (status & ARGUSCONTROL) {
      switch (status & (INIT | STATUS)) {
         case INIT:
            ws->ws_init.now.tv_sec  = htonl(ws->ws_init.now.tv_sec);
            ws->ws_init.now.tv_usec = htonl(ws->ws_init.now.tv_usec);
            ws->ws_init.startime.tv_sec  = htonl(ws->ws_init.startime.tv_sec);
            ws->ws_init.startime.tv_usec = htonl(ws->ws_init.startime.tv_usec);
            ws->ws_init.localnet = htonl(ws->ws_init.localnet);
            ws->ws_init.netmask = htonl(ws->ws_init.netmask);
            break;

         case STATUS:
            ws->ws_stat.startime.tv_sec  = htonl(ws->ws_stat.startime.tv_sec);
            ws->ws_stat.startime.tv_usec = htonl(ws->ws_stat.startime.tv_usec);
            ws->ws_stat.now.tv_sec  = htonl(ws->ws_stat.now.tv_sec);
            ws->ws_stat.now.tv_usec = htonl(ws->ws_stat.now.tv_usec);
            ws->ws_stat.pktsRcvd = htonl(ws->ws_stat.pktsRcvd);
            ws->ws_stat.bytesRcvd = htonl(ws->ws_stat.bytesRcvd);
            ws->ws_stat.pktsDrop = htonl(ws->ws_stat.pktsDrop);
            ws->ws_stat.actTCPcons = htons(ws->ws_stat.actTCPcons);
            ws->ws_stat.cloTCPcons = htons(ws->ws_stat.cloTCPcons);
            ws->ws_stat.actUDPcons = htons(ws->ws_stat.actUDPcons);
            ws->ws_stat.cloUDPcons = htons(ws->ws_stat.cloUDPcons);
            ws->ws_stat.cloIPcons = htons(ws->ws_stat.cloIPcons);
            ws->ws_stat.actIPcons = htons(ws->ws_stat.actIPcons);
            ws->ws_stat.cloICMPcons = htons(ws->ws_stat.cloICMPcons);
            ws->ws_stat.actICMPcons = htons(ws->ws_stat.actICMPcons);
            ws->ws_stat.cloFRAGcons = htons(ws->ws_stat.cloFRAGcons);
            ws->ws_stat.actFRAGcons = htons(ws->ws_stat.actFRAGcons);
            break;
      }
   } else {
      arg_uint32 newvalue;
      arg_uint16 *ptr = (arg_uint16 *)&newvalue;

      ws->ws_ip.startime.tv_sec  = htonl(ws->ws_ip.startime.tv_sec);
      ws->ws_ip.startime.tv_usec = htonl(ws->ws_ip.startime.tv_usec);

      if ((major_version > 1) || (minor_version > 6)) {
         ptr[0] = htons(((unsigned short *)&ws->ws_ip.lasttime.tv_sec)[0]);
         ptr[1] = htons(((unsigned short *)&ws->ws_ip.lasttime.tv_sec)[1]);
         ws->ws_ip.lasttime.tv_sec = newvalue;
      } else {
         ws->ws_ip.lasttime.tv_sec  = htonl(ws->ws_ip.lasttime.tv_sec);
      }

      ws->ws_ip.lasttime.tv_usec  = htonl(ws->ws_ip.lasttime.tv_usec);

      ws->ws_ip.src.s_addr = htonl(ws->ws_ip.src.s_addr);
      ws->ws_ip.dst.s_addr = htonl(ws->ws_ip.dst.s_addr);

      ptr[0] = htons(((arg_uint16 *)&ws->ws_ip.port)[0]);
      ptr[1] = htons(((arg_uint16 *)&ws->ws_ip.port)[1]);
 
      switch (status & (ICMPPROTO | TCPPROTO | UDPPROTO)) {
         case TCPPROTO:
            ws->ws_ip.port = newvalue;
            ws->ws_ip_tcp.src_count = htonl(ws->ws_ip_tcp.src_count);
            ws->ws_ip_tcp.dst_count = htonl(ws->ws_ip_tcp.dst_count);
            ws->ws_ip_tcp.src_bytes = htonl(ws->ws_ip_tcp.src_bytes);
            ws->ws_ip_tcp.dst_bytes = htonl(ws->ws_ip_tcp.dst_bytes);
            break;

         case ICMPPROTO:
            if (!(status & (CON_ESTABLISHED | TIMED_OUT))) {
               ws->ws_ip_icmp.data = htons(ws->ws_ip_icmp.data);
               ws->ws_ip_icmp.srcaddr.s_addr=htonl(ws->ws_ip_icmp.srcaddr.s_addr);
               ws->ws_ip_icmp.dstaddr.s_addr=htonl(ws->ws_ip_icmp.dstaddr.s_addr);
               ws->ws_ip_icmp.gwaddr.s_addr =htonl(ws->ws_ip_icmp.gwaddr.s_addr);
            } else {
               ws->ws_ip_udp.src_count = htonl(ws->ws_ip_udp.src_count);
               ws->ws_ip_udp.dst_count = htonl(ws->ws_ip_udp.dst_count);
               ((unsigned short *)&ws->ws_ip_udp.src_bytes)[1] =
                                        htons(((unsigned short *)&ws->ws_ip_udp.src_bytes)[1]);
               ws->ws_ip_udp.dst_bytes = htonl(ws->ws_ip_udp.dst_bytes);
            }
            break;

         case UDPPROTO:
            ws->ws_ip.port = newvalue;
         default:
            ws->ws_ip_udp.src_count = htonl(ws->ws_ip_udp.src_count);
            ws->ws_ip_udp.dst_count = htonl(ws->ws_ip_udp.dst_count);
            ws->ws_ip_udp.src_bytes = htonl(ws->ws_ip_udp.src_bytes);
            ws->ws_ip_udp.dst_bytes = htonl(ws->ws_ip_udp.dst_bytes);
            break;
      }
   }
}


#define IPPROTOMASK   (TCPPROTO | ICMPPROTO | UDPPROTO)
#define THEPROTOMASK  (PROTOMASK & ~IPPROTOMASK)

unsigned char src_ttl = 0;
unsigned char dst_ttl = 0;

void
reformat_datum (ptr, buf)
struct WriteStruct *ptr, *buf;
{
   int rev = 0;
   unsigned int lasttime, difftime;
   arg_uint32 saddr, daddr;
   arg_uint16 sport, dport, ttl;
   struct WriteStruct tmpbuf, *tmp = &tmpbuf;

   if (ptr && buf) {
      wsntoh(ptr);

      if ((major_version > 1) || (minor_version > 6)) {
         ttl = ((u_short *)&ptr->ws_ip.lasttime.tv_sec)[0];
         difftime = (u_int) (((u_short *)&ptr->ws_ip.lasttime.tv_sec)[1]);
         lasttime = ptr->ws_ip.startime.tv_sec + difftime;
         ptr->ws_ip.lasttime.tv_sec = lasttime;
      }

      bcopy ((char *) ptr, (char *) tmp, sizeof (struct WriteStruct));

      src_ttl = ((u_char *)&ttl)[1];
      dst_ttl = ((u_char *)&ttl)[0];

      if (tmp->status & REVERSE) {
         rev++;
         bcopy ((char *) &ptr->ws_ip_phys.etherdst,
                (char *) &tmp->ws_ip_phys.ethersrc, 6);
         bcopy ((char *) &ptr->ws_ip_phys.ethersrc,
                (char *) &tmp->ws_ip_phys.etherdst, 6);
         bcopy ((char *) &ptr->ws_ip.dst, (char *) &tmp->ws_ip.src, 4);
         bcopy ((char *) &ptr->ws_ip.src, (char *) &tmp->ws_ip.dst, 4);

         tmp->status &= ~REVERSE;

         if (tmp->status & (TCPPROTO | UDPPROTO)) {
            sport = ((unsigned short *) &ptr->ws_ip.port)[0];
            dport = ((unsigned short *) &ptr->ws_ip.port)[1];
/*
            src_ttl = ((u_char *)&ttl)[0];
            dst_ttl = ((u_char *)&ttl)[1];
*/
            bcopy ((char *) &sport,
                (char *) &((unsigned short *)&tmp->ws_ip.port)[1], 2);
            bcopy ((char *) &dport,
                (char *) &((unsigned short *)&tmp->ws_ip.port)[0], 2);
         }
/*
      } else {
         src_ttl = ((u_char *)&ttl)[1];
         dst_ttl = ((u_char *)&ttl)[0];
*/
      }

      bcopy ((char *) tmp, (char *) buf, sizeof (struct WriteStruct));
      sport = ((unsigned short *) &tmp->ws_ip.port)[0];
      dport = ((unsigned short *) &tmp->ws_ip.port)[1];

      if (tmp->status & IPPROTOMASK) {
         tmp->status |= IPPROTO;
         if (!((tmp->status & TCPPROTO) && ((tmp->status & SAW_SYN) ||
                (tmp->status & SAW_SYN_SENT))))
               
         if (tmp->status & (TCPPROTO | UDPPROTO))
            if ((dport > sport) && (sport != 0)) {
               tmp->status |= REVERSE;
               sport = ((unsigned short *) &tmp->ws_ip.port)[1];
               dport = ((unsigned short *) &tmp->ws_ip.port)[0];
            }

         switch (tmp->status & IPPROTOMASK) {
            case TCPPROTO:
               if (dport == 20)
                  tmp->status ^= REVERSE;
               break;

            case UDPPROTO:
               if (((sport == 53) && (dport != 53)) ||
                   ((sport == 2049) && (dport != 2049)) ||
                   ((sport >= 7000) && (sport < 7010)) ||
                   ((sport >= 6000) && (sport < 6010)))
                     tmp->status ^= REVERSE;
               break;
         }
      }
      
      if (tmp->status & REVERSE) {
         bcopy ((char *) &tmp->ws_ip_phys.etherdst,
                (char *) &buf->ws_ip_phys.ethersrc, 6);
         bcopy ((char *) &tmp->ws_ip_phys.ethersrc,
                (char *) &buf->ws_ip_phys.etherdst, 6);
         bcopy ((char *) &tmp->ws_ip.dst, (char *) &buf->ws_ip.src, 4);
         bcopy ((char *) &tmp->ws_ip.src, (char *) &buf->ws_ip.dst, 4);
         buf->status = tmp->status & ~REVERSE;

         if (tmp->status & (TCPPROTO | UDPPROTO)) {
            sport = ((unsigned short *) &tmp->ws_ip.port)[0];
            dport = ((unsigned short *) &tmp->ws_ip.port)[1];
            bcopy ((char *) &sport, 
                   (char *) &((unsigned short *) &buf->ws_ip.port)[1], 2);
            bcopy ((char *) &dport, 
                   (char *) &((unsigned short *) &buf->ws_ip.port)[0], 2);
            buf->ws_ip_udp.src_count = tmp->ws_ip_udp.dst_count;
            buf->ws_ip_udp.dst_count = tmp->ws_ip_udp.src_count;
            buf->ws_ip_udp.src_bytes = tmp->ws_ip_udp.dst_bytes;
            buf->ws_ip_udp.dst_bytes = tmp->ws_ip_udp.src_bytes;
         }
      }
      if (rev) buf->status |= REVERSE;

      if ((buf->ws_ip_udp.src_count + buf->ws_ip_udp.dst_count) == 1) {
         if (buf->status & MULTIADDR) buf->status &= ~MULTIADDR;
         buf->ws_ip.lasttime = buf->ws_ip.startime;
      }

      wshton(ptr); wshton(buf);
   }
}


void
adjust_group_port (sport, dport)
arg_uint16 *sport, *dport;
{
   int is_src_server = 0, is_dst_server = 0;
   extern arg_uint16 *udpServicePorts;
   arg_uint16 *ptr = udpServicePorts, *end = NULL;
 
   if (ptr && *ptr) {
      do {
         if (*sport == *ptr) is_src_server = 1;
         if (*dport == *ptr) is_dst_server = 1;
         ptr++;
      } while (*ptr);
 
      if ((is_src_server || is_dst_server) &&
                              !(is_src_server && is_dst_server)) {
         if (is_src_server) *dport = 0;
         if (is_dst_server) *sport = 0;
      }  
   }
}

read_remote_connection (fd, bpfcode)
int fd;
struct bpf_program *bpfcode;
{
   int retn = 0, cnt = 0;
   char buf[256], *str = buf, c;
   struct WriteStruct ws;

   bzero (buf, 256);
   if (fd) {
      if (read (fd, str, 1) == 1)
         if (*str == 'A') {
            while (read (fd, ++str, 1)) {
               cnt++;
               if (*str == '\n') break;
            }   
         } else {
            read (fd, &buf[1], sizeof(ws) - 1);
            bzero ((char *) &ws, sizeof(ws));
            bcopy ((char *) buf, (char *) &ws, sizeof(ws));
            bzero ((char *) buf, sizeof(buf));
            strcpy (buf, ws.ws_init.initString);
            cnt = strlen (buf);
            if (initCon == NULL) {
               if (initCon = (struct WriteStruct *) calloc (1, sizeof (ws))) {
                  bcopy ((char *) &ws, (char *) initCon, sizeof (ws));
               } 
            }
         }

      if (cnt) {
         if (strstr (buf, tag_string)) {
            str = &buf[strlen (tag_string)];
            if (sscanf (str, "%d.%d", &major_version, &minor_version)) {
               fprintf (stderr, "\nArgus Version %d.%d\n", major_version,
                      minor_version);
               read_mode = major_version;
               argus_parse_init (fd);
               init ();
            } else {
               fprintf (stderr, "not argus server: disconnecting.\n");
               retn = 1;
            }
         } else {
            fprintf (stderr, "not argus server: disconnecting.\n");
            retn = 1;
         }
      } else {
         fprintf (stderr, "refused.\n");
         retn = 1;
      }
   } else
      fd = -1;

   if (retn) {
      close (fd);
      fd = -1;
   }
   return (fd);
}


read_connection (filename, bpfcode)
char *filename;
struct bpf_program *bpfcode;
{
   int cnt, fd = 0;
   char buf[256], *str = buf, c;
   struct WriteStruct ws;

   if (filename) {
      if ((fd = open(filename, O_RDONLY)) < 0) {
         sprintf (buf, "open: %s", filename);
         perror (buf);
      }
   }

   if (fd >= 0) {
      if ((cnt = read (fd, str, 1)) == 1) {
         if (*str == 'A') {
            while (cnt = read (fd, ++str, 1)) 
               if (*str == '\n')
                  break;
         } else
            cnt = read (fd, &buf[1], sizeof(ws) - 1);

         bzero ((char *) &ws, sizeof(ws));
         bcopy ((char *) buf, (char *) &ws, sizeof(ws));
         bzero ((char *) buf, sizeof(buf));
         strcpy (buf, ws.ws_init.initString);

         if (initCon == NULL) {
            if (initCon = (struct WriteStruct *) calloc (1, sizeof (ws)))
               bcopy ((char *) &ws, (char *) initCon, sizeof (ws));
         }

         if (strstr (buf, tag_string)) {
            str = &buf[strlen (tag_string)];
            if (sscanf (str, "%d.%d", &major_version, &minor_version)) {
               if (strstr (str, "detail")) detail++;
               read_mode = major_version;
               argus_parse_init (fd);
               init ();
               if (initCon) {
                  if (wfile) {
                     if (writeNewLogfile (NULL)) {
                        fprintf (stderr, "writeNewLogfile: error\n");
                        exit (1);
                     }
                  } else
                     handle_datum (initCon, bpfcode);
               }
            } else {
               fprintf (stderr, "Argus file format error: %s", buf);
               close (fd);
               fd = -1;
            }
         } else {
            fprintf (stderr, "Not Argus file: %s", buf);
            close (fd);
            fd = -1;
         }

      } else
         if (filename)
            fprintf (stdout, "read: %s empty?\n", filename);
   }

   return (fd);
}


#define CLIENTTIMEOUT  1

void
read_remote (fd, bpfcode)
int fd;
struct bpf_program *bpfcode;
{
   struct WriteStruct ptr;
   register int cnt, size, retn;
   extern void clientTimeout ();
   struct timeval wait, now, timeoutValue;
   fd_set readmask;
   int width = fd + 1;

   switch (read_mode) {
      default: size = sizeof (struct WriteStruct) ; break;
   }

   bzero (&ptr, sizeof (struct WriteStruct));

   if (gettimeofday (&now, NULL) == 0) {
      timeoutValue = now;
      timeoutValue.tv_sec += CLIENTTIMEOUT;
      wait.tv_sec = 0; wait.tv_usec = 250000;
   
      FD_ZERO (&readmask);
      FD_SET (fd, &readmask);

      for (;;) {
         if ((retn = select (width, &readmask, NULL, NULL, &wait)) >= 0) {
            if (FD_ISSET (fd, &readmask))
               if ((cnt = read (fd, &ptr, size)) >= 0)
                  if (cnt == size)
                     handle_datum (&ptr, bpfcode);
   
            gettimeofday (&now, NULL);
   
            if (now.tv_sec > timeoutValue.tv_sec) {
               (void) clientTimeout ();
               now = timeoutValue;
               timeoutValue.tv_sec += CLIENTTIMEOUT;
            }
            FD_SET (fd, &readmask);
         } else
            if (errno != EINTR)
               break;
      }

      if (retn < 0)
         perror ("select");
   }
}


read_file (fd, bpfcode)
int fd;
struct bpf_program *bpfcode;
{
   struct WriteStruct ptr;
   register int cnt, size, retn = 1;

   switch (read_mode) {
      default: size = sizeof (struct WriteStruct) ; break;
   }

   while ((cnt = read (fd, &ptr, size)) > 0) {
      retn = 0;
      handle_datum (&ptr, bpfcode);
   }

   return (retn);
}


struct tm tm_lasttime;
struct tm tm_startime;

void
version_process (ptr)
struct WriteStruct *ptr;
{
   unsigned int status;
   struct tm *tm;

   if (tm = localtime ((time_t *)&ptr->ws_ip.startime.tv_sec))
      bcopy ((char *) tm, (char *)&tm_startime, sizeof (struct tm));
   else
      bzero ((char *)&tm_startime, sizeof (struct tm));

   if (tm = localtime ((time_t *)&ptr->ws_ip.lasttime.tv_sec))
      bcopy ((char *) tm, (char *)&tm_lasttime, sizeof (struct tm));
   else
      bzero ((char *)&tm_startime, sizeof (struct tm));
   
   if (ptr->status & ARGUSCONTROL)
      process_man (ptr);

   else {
      if (status = (ptr->status & THEPROTOMASK)) {
         switch (status) {
            case   IPPROTO:
               switch (ptr->status & IPPROTOMASK) {
                  case  TCPPROTO: process_tcp (ptr); break;
                  case  UDPPROTO: process_udp (ptr); break;
                  case ICMPPROTO: process_icmp (ptr); break;
                         default: process_ip (ptr); break;
               }
               break;
         }
      }
   }
}


in_queue (queue, object)
struct QUEUE *queue;
struct NET_OBJECT *object;
{
   register int retn = 0;
   register struct NET_OBJECT *ptr;

   if (ptr = (struct NET_OBJECT *) queue->start)
      for ( ; ptr; ptr = (struct NET_OBJECT *) ptr->queue.nxt)
         if (ptr = object) {
            retn = 1; break;
         }

   return (retn);
}


void
add_to_queue (queue, object)
struct QUEUE *queue;
struct NET_OBJECT *object;
{
   register int i;
   register struct QUEUE_HEADER *head = queue->start;

   object->queue.queue = queue;
   queue->count++;

   if (head) {
      object->queue.prv = head->prv;
      if (object->queue.prv)
         (object->queue.prv)->nxt = &object->queue;

      object->queue.nxt = head;
   }

   queue->start = &object->queue;
}


void
remove_from_queue (object)
struct NET_OBJECT *object;
{
   register struct QUEUE *queue;
   register struct QUEUE_HEADER *prv, *nxt;

   if (object) {
      if (queue = object->queue.queue) {
         queue->count--;
         object->queue.queue = NULL;
   
         nxt = object->queue.nxt;
   
         if (prv = object->queue.prv)
            prv->nxt = nxt;
   
         if (nxt)
            nxt->prv = prv;
   
         if (queue->start == &object->queue)
            queue->start = nxt;
      }

      object->queue.prv = NULL;
      object->queue.nxt = NULL;
   }
}




#include <netdb.h>

get_server_socket (ip)
struct IP_ENTRY *ip;
{
   register int retn = 0;
   struct sockaddr_in server;
   struct servent *sp;
   struct hostent *hp;
   int s;

   if (!portnum) {
      if (sp = getservbyname ("monitor", "tcp")) portnum = sp->s_port;
      else portnum = 561;
   } else
      portnum = htons(portnum);

   if (hp = gethostbyaddr ((char *)&ip->addr, sizeof (ip->addr), AF_INET)) {
      bzero ((char *) &server, sizeof (server));
      bcopy ((char *) hp->h_addr, (char *)&server.sin_addr, hp->h_length);
      server.sin_family = hp->h_addrtype;
      server.sin_port = portnum;
      fprintf (stderr, "%s: Trying %s port %d\n", progname, (hp->h_name) ?
                  (hp->h_name) : intoa (ip->addr), ntohs(portnum)); 
      if ((s = socket (AF_INET, SOCK_STREAM, 0)) >= 0) {
         if ((connect (s, (struct sockaddr *)&server, sizeof(server))) >= 0) {
            retn = s;
            fprintf (stderr, "%s: connected ", progname);
         } else
            perror ("get_server_socket: connect() failed.");
      } else
         perror ("get_server_socket: socket() failed.");
   } else
      perror ("get_server_socket: gethostbyaddr() failed.");

   return (retn);
}


new_file_arg (ptr)
char *ptr;
{
   register int retn = 0;
   register struct FILE_ENTRY *file, *list, *tmp;

   if (ptr) {
      if (file = (struct FILE_ENTRY *) calloc (1, sizeof(struct FILE_ENTRY))) {
         if (list = input_file_list) {
            while (list->nxt) list = list->nxt;
            list->nxt = file;
         } else
            input_file_list = file;

         file->str = savestr (ptr);
         retn = 1;
      }
   }
   return (retn);
}


remote_host_arg (ptr)
char *ptr;
{
   register int retn = 0;
   register struct IP_ENTRY *addr;
   arg_uint32 alist;
   arg_uint32 **name;

   if ((alist = (arg_uint32) inet_addr (ptr)) == (unsigned int) -1) {
      if (name = (arg_uint32 **) argus_nametoaddr (ptr)) {
         while (*name) {
            if (addr = (struct IP_ENTRY *) calloc (1, 
                                            sizeof (struct IP_ENTRY))) {
               addr->nxt = remote_host_list;
               remote_host_list = addr;
               addr->addr = ntohl(**name);
               *name++;
               retn = 1;
            }
         }
      }
   } else
      if (addr = (struct IP_ENTRY *) calloc (1, sizeof (struct IP_ENTRY))) {
         addr->nxt = remote_host_list;
         remote_host_list = addr;
         addr->addr = alist;
         retn = 1;
      }

   return (retn);
}


static int firstWrite = 1;

writeNewLogfile (ws)
struct WriteStruct *ws;
{
   int i, retn = 0;
   struct stat buf;
   FILE *fd;

   if (wfile) {
      if (strcmp (wfile, "-")) {
         if (stat (wfile, &buf) < 0)
            if (errno == ENOENT)
               if (fd = fopen (wfile, "a")) {
                  if (initCon) {
                     if (!(fwrite ((char *) initCon, sizeof (*initCon), 1, fd)))
                        perror ("fwrite");
                  } else
                     if (!(fprintf (fd, "Argus Version %d.%d\n",
                                           major_version, minor_version)))
                        perror ("fprintf");
                  fclose (fd);
               } else
                  perror ("fopen");

         if (ws && (fd = fopen (wfile, "a"))) {
            if (!(fwrite (ws, sizeof (struct WriteStruct), 1, fd)))
               perror ("fwrite");
            fclose (fd);
         }
      } else {
         if (firstWrite) {
            if (initCon) {
               if (!(fwrite ((char *) initCon, sizeof (*initCon), 1, stdout)))
                  perror ("fwrite");
            } else
               if (!(fprintf (stdout, "Argus Version %d.%d\n",
                                           major_version, minor_version)))
                  perror ("fprintf");

            fflush (stdout);
            firstWrite = 0;
         }
         if (ws) {
            if (!(fwrite (ws, sizeof (struct WriteStruct), 1, stdout)))
               retn++;
            fflush (stdout);
         }
      }
   }
   return (retn);
}


int explicit_date = 0;
time_t lasttime_t = 0;
time_t startime_t = 0;

check_time (ptr)
struct WriteStruct *ptr;
{
   struct tm tmbuf, *tm;
   time_t *sec;
   int retn = 0;

   if (tflag) {
      if (!explicit_date) {
         if (ptr->status & ARGUSCONTROL) {
            sec = (time_t *)&ptr->ws_init.now.tv_sec;
         } else {
            sec = (time_t *)&ptr->ws_ip.startime.tv_sec;
         }

         tm = localtime(sec);
         if (tm->tm_yday != starTimeFilter.tm_yday) {
            bcopy ((char *) tm, (char *) &tmbuf, sizeof (struct tm));
            check_time_format (&tmbuf, timearg);
         }
         explicit_date++;
      }

      if (ptr->status & ARGUSCONTROL) {
         if (ptr->status & INIT) {
            if ((ptr->ws_init.now.tv_sec >= startime_t) &&
                (ptr->ws_init.now.tv_sec <= lasttime_t))
               retn++;
         } else {
            if ((ptr->ws_stat.now.tv_sec >= startime_t) &&
                (ptr->ws_stat.now.tv_sec <= lasttime_t))
               retn++;
         }
      } else {
         if (((ptr->ws_ip.startime.tv_sec >= startime_t) &&
              (ptr->ws_ip.startime.tv_sec <= lasttime_t)) ||
             ((ptr->ws_ip.lasttime.tv_sec >= startime_t) &&
              (ptr->ws_ip.lasttime.tv_sec <= lasttime_t)) ||
             ((ptr->ws_ip.startime.tv_sec <  startime_t) &&
              (ptr->ws_ip.lasttime.tv_sec >  lasttime_t)))
            retn++;
      }
   } else
      retn++;

   return (retn);
}

parseTimeArg (arg, args, ind, tm)
char **arg, *args[];
int ind;
struct tm *tm;
{
   int retn = -1;
   char buf[64], *ptr = buf;

   bzero (buf, 64);
   strcpy (buf, *arg);
   ptr += strlen (buf);
   if ((ptr = strchr(*arg, '-')) && (*(ptr + 1) != '\0')) {
      retn = 0;
   } else
      if (args[ind] && (*args[ind] == '-')) {
         if (strlen (args[ind]) == 1) {
            strcat (buf, "-");
            strcat (buf, args[ind + 1]);
            retn = 2;
         } else {
            ptr = args[ind];
            if (isdigit(*(ptr + 1))) {
               strcat (buf, args[ind]);
               retn = 1;
            } else
               retn = 0;
         }
      } else
         retn = 0;

   if (check_time_format (tm, buf))
      usage (args[0]);

   *arg = savestr(buf);
   return (retn);
}


check_time_format (tm, str)
struct tm *tm;
char *str;
{
   int retn = 0;
   char *ptr, buf[64];

   /*[mm/dd[/yy].]hh[:mm[:ss]]-[mm/dd[/yy].]hh[:mm[:ss]]*/

   strcpy (buf, str);

   if (ptr = strchr(buf, '-')) {
      *ptr = '\0';
      if (parseTime (&starTimeFilter, tm, buf)) retn++;
      if (parseTime (&lastTimeFilter, &starTimeFilter, ptr + 1)) retn++;
   } else {
      if (parseTime (&starTimeFilter, tm, buf)) retn++;
      bcopy ((char *)&starTimeFilter, (char *)&lastTimeFilter,
                 sizeof(struct tm));
   }
   startime_t = timelocal (&starTimeFilter);
   lasttime_t = timelocal (&lastTimeFilter);

   if (!(lasttime_t >= startime_t)) {
      fprintf (stderr, "error: invalid time range\n");
      retn++;
   }
      
   return (retn);
}

parseTime (tm, now, str)
struct tm *tm, *now;
char *str;
{
   char *ptr, *md, *yr;
   int retn = 0, day = 0, mon = -1, year = 0;
   int hour = 0, mins = 0, sec = 0;
   time_t thistime;

   /*[mm/dd[/yy].]hh[:mm[:ss]]*/

   bcopy ((u_char *) now, (u_char *) tm, sizeof (struct tm));

   if ((ptr = strchr (str, '.')) || !strchr (str, '/')) {
      if (ptr) *ptr++ = '\0';
      else ptr = str;
      hour = atoi(ptr);
      if (ptr = strchr (ptr, ':')) {
         mins = atoi(++ptr);
         if (ptr = strchr (ptr, ':'))
            sec = atoi(++ptr);
      }
   }

   if (md = strchr (str, '/')) {
      mon = atoi(str) - 1;
      day = atoi(md + 1);
      explicit_date++;
      if (yr = strchr (md + 1, '/')) 
         year = atoi(yr + 1);
   }

   if (day) tm->tm_mday  = day;
   if (mon >= 0)  tm->tm_mon  = mon;
   if (year) tm->tm_year = year;
   tm->tm_hour = hour; tm->tm_wday = 0;
   tm->tm_min = mins; tm->tm_sec = sec;
   tm->tm_yday = 0; tm->tm_isdst = 0;

#if !defined(HAVE_SOLARIS) && !defined(__sgi) && !defined(linux) && !defined(AIX)
   tm->tm_zone = NULL;
   tm->tm_gmtoff = 0;
#endif

   thistime = timelocal (tm);
   tm = localtime ((time_t *)&thistime);
   return (retn);
}
