
/*
Hey Carter,

Remember, the goal here is to "explain"... and to have so many
comments that it makes you sick!  This will help folks come up
to speed.

Rich
*/


/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 *
 * fullra - Print out an ungodly amount of information about an 
 *          argus record.
 *
 * Rich Verjinski
 * FORE Systems, Inc.
 *
 * Carter Bullard
 * Network Flow Sciences
 *
 */

/* 
 * Comments added by R. Verjinski.  Hopefully this  
 * will allow new folks to understand this code.  :)
 */

#define ARGUS_CLIENT

#if !defined(SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>

#ifndef SOLARIS
#include <strings.h>
#endif

#include <string.h>
#include <time.h>

#include <sys/socket.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>

#include <arpa/inet.h>
#include <netdb.h>

#include <policy.h>

#include <cons_out.h>
#include <argus_parse.h>

#include <pcap.h>
#include <interface.h>
#include <addrtoname.h>
#include <nametoaddr.h>

/* 
 * Standard IP Protocol Numbers, used as a lookup table to print
 * out the protocol type in a standard argus output string.
 */
#define IPPROTOSTR 99
char *ip_proto_string [IPPROTOSTR] = {"ip", "icmp", "igmp", "ggp",
   "ipnip", "st", "tcp", "ucl", "egp", "igp", "bbn-rcc-mon", "nvp-ii",
   "pup", "argus", "emcon", "xnet", "chaos", "udp", "mux", "dcn-meas",
   "hmp", "prm", "xns-idp", "trunk-1", "trunk-2", "leaf-1", "leaf-2",
   "rdp", "irtp", "iso-tp4", "netblt", "mfe-nsp", "merit-inp", "sep",
   "3pc", "idpr", "xtp", "ddp", "idpr-cmtp", "tp++", "il", "unas",
   "unas", "unas", "unas", "unas", "unas", "unas", "unas", "unas",
   "unas", "unas", "unas", "unas", "unas", "unas", "unas", "unas",
   "unas", "unas", "unas", "any", "cftp", "any", "sat-expak", "kryptolan",
   "rvd", "ippc", "any", "sat-mon", "visa", "ipcv", "cpnx", "cphb", "wsn",
   "pvp", "br-sat-mon", "sun-nd", "wb-mon", "wb-expak", "iso-ip", "vmtp",
   "secure-vmtp", "vines", "ttp", "nsfnet-igp", "dgp", "tcf", "igrp",
   "ospfigp", "sprite-rpc", "larp", "mtp", "ax.25", "ipip", "micp",
   "aes-sp3-d", "etherip", "encap",
};

                         /*
                          *  This table is the ICMP type lookup table.
                          */

char *icmptypestr[ICMP_MAXTYPE + 1] = {
   "ECR", "   ", "   ", "UR", "SRC", "RED",
   "   ", "   ", "ECO", "   ", "   ", "TIM",
   "PAR", "TST", "TSR", "IRQ", "IRR", "MAS",
   "MSR",
};    

			/*
			 * Global variable, used in get_xxx_string() 
			 */
char buf[1024];

			/*
			 * Tell the compiler that these procedures 
			 * return a pointer to a character. Actually,
			 * it returns a pointer to a string. And it
                         * actually returns buf[].
			 */
char *get_tcp_string ();
char *get_icmp_string ();
char *get_ip_string ();
char *get_udp_string ();
char *get_man_init_string ();
char *get_man_stat_string ();

                        /* 
                         * Common code will use this string if you want
                         * to add more command line option flags.  Used
                         * but not completely tested.  Add unique flags
                         * here, as if it was going to be a string for
                         * getopt() and add the parsing in parse_arg()
                         * routine, which is called from the common
                         * routines if appOptstring is not NULL.
                         * See ./common/argus_parse.c for hints as to
                         * how to use this feature.
                         */

char *appOptstring = NULL;

			/*
			 * Externals are references to variables that
                         * are declared and set in the common directory.
			 */

extern int major_version, minor_version;

			/*
			 * The start and lastimte structs are used
			 * for the timestamping the data flows...
			 * Starttime is the start of the flow.
			 * Lasttime is the time of THIS record.
                         */

extern struct tm tm_startime, tm_lasttime;

			/*
			 * The src_ttl and dst_ttl are the
			 * ttl values that were seen on the
			 * unidirectional flow.  These are
			 * the actual ttl values from the
			 * packet stream and so you need to
			 * do a little to figure out how far
			 * the two entities are.
                         */

extern u_char src_ttl, dst_ttl;


/*  REQUIRED ARGUS CLIENT ROUTINE
 *          init ();
 *		This is the application specific init                    
 *              routine, which is called after all parsing
 *              initialization is done, prior to reading the
 *              first argus(1) datum.                      
 */

init ()
{
   printf("\nARGUS INIT: Common code calls client specific initialization.\n");
}

/*  REQUIRED ARGUS CLIENT ROUTINE
 *	argus_parse_complete();
 *		This call will be called after all monitor data
 *		has been parsed.
 */

void
argus_parse_complete ()
{
   printf("\nARGUS PARSE COMPLETE: Common code calls argus_parse_complete as final step.\n");
   printf("                    : Great place to print statistics of all data seen so far.\n");
   printf("                    : Also great place to perform cleanups, in your own clients.\n");
}

/*  REQUIRED ARGUS CLIENT ROUTINE
 *      clientTimeout ();
 *		This routine is called every second, when
 *		the argus client is connected to a remote
 *		data source using the -S flag.
 */

void
clientTimeout ()
{
   printf("ARGUS clientTimeout(): called 1/sec if connected to live server.\n");
}

/*
 *      parse_arg (argc, argv)
 *		This allows THIS client to parse it's own options.  Called
 *		from the common code when appOptstring is not NULL.
 *		See getopt and ./common/argus_parse.c to see how it works.
 */

parse_arg (argc, argv)
int argc;
char**argv;
{}


/*  REQUIRED ARGUS CLIENT ROUTINE
 *  	usage (ptr); 
 *		Prints out the usage of this client program.
 *		If you add your own options, or change the meaning
 *		of the standard ones, make the changes here.
 *		The standard options are all handled by the
 *		./common code and you get all these functions
 *		when you link the common libs.
 */

void
usage (ptr)
char *ptr;
{
printf("ARGUS USAGE: Common code calls usage if UNKNOWN parameters\n");
printf("           : are passed. Here are the 'common' options:\n");
   fprintf (stderr, "usage: %s [-bchmnEIMNORTWX] ", ptr);
   fprintf (stderr, "[-C access-file] ");
   fprintf (stderr, "[-d debug-level]\n");
   fprintf (stderr, "              [-r input_file] ");
   fprintf (stderr, "[-w output_file] ");
   fprintf (stderr, "[-F file] [-P port]\n              ");
   fprintf (stderr, "[-t time_range] [-S argus_server] expression.\n");
   fprintf (stderr, "options: b - dump packet-matching code.\n");
   fprintf (stderr, "         c - print packet and byte counts.\n");
   fprintf (stderr, "         C - specify Cisco access-file.\n");
   fprintf (stderr, "         h - print help.\n");
   fprintf (stderr, "         m - print MAC addresses.\n");
   fprintf (stderr, "         n - don't convert numbers to names.\n");
   fprintf (stderr, "         r - read Argus input file.\n");
   fprintf (stderr, "         t - specify time_range for search.\n         ");
   fprintf (stderr, "      format:    ");
   fprintf (stderr, "timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                          timeSpecification: ");
   fprintf (stderr, "[mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             ");
   fprintf (stderr, " mm/dd[/yy]\n");
   fprintf (stderr, "         u - print msecs timestamps (*)\n");
   fprintf (stderr, "         w - write Argus output file.\n");
   fprintf (stderr, "         x - flip ports on input (*)\n");
   fprintf (stderr, "         P - specify remote argus port.\n");
   fprintf (stderr, "         S - specify remote argus host.\n");
   fprintf (stderr, "         E - print ESTABLISHED transactions.\n");
   fprintf (stderr, "         I - print extended ICMP status.\n");
   fprintf (stderr, "         F - use file to define filter expression.\n");
   fprintf (stderr, "         M - print MULTIROUTE transactions.\n");
   fprintf (stderr, "         N - print NORMAL_CLOSED transactions.\n");
   fprintf (stderr, "         O - print transactions with ip OPTIONS.\n");
   fprintf (stderr, "         R - print RESET transactions.\n");
   fprintf (stderr, "         T - print TIMEDOUT transactions.\n");
   fprintf (stderr, "         W - print WINDOW_SHUT transactions.\n");
   fprintf (stderr, "         X - print PKTS_RETRANS transactions.\n");
   exit(1);
}

#define FLAGSTRLEN   5

/*  OPTIONAL ARGUS CLIENT ROUTINE
 *	process_argus_record ((struct writeStruct *) ptr);
 *		This routine is called from each protocol specific
 *              routine if you require a generic record processing
 *              routine.  This is so, because the common code, has
 *              specific argus record type processing code, so it
 *              knows what the argus record type is.  No sense in
 *              loosing this knowledge, so the common routine
 *              calls out to specific argus record type routines.
 *              The strategy is if you want to have common processing
 *              for each record in your client code, put it here,
 *              rather than changing the ./common code.
 */

process_argus_record (ptr)
struct WriteStruct *ptr;
{
  printf ("\nARGUS RECORD: Each protocol specific routine calls process_argus_record ().\n");

  if (ptr->status & ARGUSCONTROL) {
     printf (" ARGUS CONTROL RECORD:\n");

  } else {
     printf (" ARGUS DATA RECORD:\n");
     printf ("    Protocol Field:\n");
     if (ptr->status &   IPPROTO) printf ("      -   IP protocol\n");
     if (ptr->status &  UDPPROTO) printf ("      -  UDP protocol\n");
     if (ptr->status & ICMPPROTO) printf ("      - ICMP protocol\n");
     if (ptr->status &  TCPPROTO) printf ("      -  TCP protocol\n");
     if (ptr->status &   EPPROTO) printf ("      -   EP protocol\n");
     if (ptr->status &  ARPPROTO) printf ("      -  ARP protocol\n");

     printf ("    TTL Values:\n");
     printf ("      -   Src TTL %d\n", src_ttl);
     printf ("      -   Dst TTL %d\n", dst_ttl);

     if (ptr->status & (FRAGMENTS | MULTIADDR | FRAG_ONLY)) {
        printf ("    Link Field:\n");
        if (ptr->status &  FRAGMENTS) printf ("      -  Fragments seen on this flow.\n");
        if (ptr->status &  MULTIADDR) printf ("      -  Multiple Link Addresses Encountered\n");
  
        if ((ptr->status & FRAG_ONLY) && !(ptr->status & TCPPROTO)) {
           printf ("      -  Initial Fragment Not Seen\n");

           if (ptr->status & FRAG_INIT) printf ("         -  Fragment Initiated\n");
           if (ptr->status & FRAG_OUT_OF_ORDER) printf ("         -  Fragments Out of Order\n");
           if (ptr->status & TCP_FRAG_OFFSET_PROBLEM) printf ("         -  TCP Fragment Attack\n");
        }
     }

     if (ptr->status & IPOPTIONMASK) {
        printf ("    IP Options Field:\n");
        if (ptr->status &   TIMESTAMP) printf ("      -  Timestamp\n");
        if (ptr->status &    SECURITY) printf ("      -  Security\n");
        if (ptr->status &   LSRCROUTE) printf ("      -  Loose Source Route\n");
        if (ptr->status &   SSRCROUTE) printf ("      -  Strict Source Route\n");
        if (ptr->status & RECORDROUTE) printf ("      -  Record Route\n");
        if (ptr->status &    SATNETID) printf ("      -  SAT Net Identification\n");
     }

     printf ("    Reporting Status Bits:\n");
     if (ptr->status &   DETAIL) printf ("      -  Detail Mode\n");
     if (ptr->status &  REVERSE) printf ("      -  Reversed Format\n");
     if (ptr->status & MODIFIED) printf ("      -  Modified Since Last Report\n");
     if (ptr->status &   LOGGED) printf ("      -  Previously Reported Flow\n");

     printf ("    Protocol State Report:\n");
     if (ptr->status & IP_INIT) printf ("      -  Generic Flow Initiated\n");
     if (ptr->status & CON_ESTABLISHED) printf ("      -  Generic Flow Established\n");

     if (ptr->status & TIMED_OUT) printf ("      -  Flow Timed Out\n");
  }
}


/*  REQUIRED ARGUS CLIENT ROUTINE
 *	process_man ((struct writeStruct *) ptr);
 *		This routine is called from the common code
 *              when a management control record is encountered.
 *              Put your specific management handling code
 *              here.
 */

process_man (ptr)
struct WriteStruct *ptr;
{
struct manInitStruct *init = &ptr->ws_init;
struct manStatStruct *stat = &ptr->ws_stat;
extern struct timeval  *mytime; 
char date[128];
char *str;

  process_argus_record (ptr);
		/*
		 * Check the status to see what type of record it is,
		 * then print out the contents of its structure.
		 */
  if (ptr->status & (INIT)) {

    printf("  ARGUS MGMT INIT: Common code calls process_man when a management\n");
    printf("                   record is encountered.\n"); 
    printf("       ARGUS INIT: Management Record Recieved\n"); 

    strftime ((char *)date, 128, "%a %m/%d %T",
       localtime((time_t *)&ptr->ws_stat.startime.tv_sec));
    printf("        - startime: %s\n",date);
    strftime ((char *)date, 128, "%a %m/%d %T",
       localtime((time_t *)&ptr->ws_stat.now.tv_sec));
    printf("        - now: %s\n",date);
    printf("        - InitString: %s",init->initString);
    printf("        - localnet: %s\n",ipaddr_string(&init->localnet));
    printf("        - netmask: %s\n",ipaddr_string(&init->netmask));
    printf("        - reportInterval: %d\n",init->reportInterval);
    printf("        - interfaceType: %d\n",init->interfaceType);
    printf("        - interfaceStatus: %d\n",init->interfaceStatus);
  }

  if ( (ptr->status & (STATUS)) || (ptr->status & (CLOSE )) ) {

    if (ptr->status & (STATUS)) {
      printf("\nARGUS MGMT STATUS: Common code calls process_man when a management\n");
      printf("      : Record is encountered.\n");
      printf("      * ARGUS STATUS Management Record Recieved:\n");
    }
    if (ptr->status & (CLOSE)) {
      printf("\nARGUS MGMT CLOSE: Common code calls process_man when a management\n");
      printf("      : Record is encountered.\n");
      printf("      * ARGUS CLOSE Management Record Recieved:\n");
    }

    strftime ((char *)date, 128, "%a %m/%d %T",
             localtime((time_t *)&ptr->ws_stat.startime.tv_sec));
    printf("        - startime: %s\n",date);
    strftime ((char *)date, 128, "%a %m/%d %T",
             localtime((time_t *)&ptr->ws_stat.now.tv_sec));
    printf("        - now: %s\n",date);

    printf("        - reportInterval: %d\n",stat->reportInterval); 
    printf("        - interfaceType: %d\n",stat->interfaceType); 
    printf("        - interfaceStatus: %d\n",stat->interfaceStatus); 
    printf("        - PktsRcvd: %d\n",stat->pktsRcvd);
    printf("        - bytesRcvd: %d\n",stat->bytesRcvd);
    printf("        - PktsDrop: %d\n",stat->pktsDrop);
    printf("        - actTCPcons: %d\n",stat->actTCPcons);
    printf("        - cloTCPcons: %d\n",stat->cloTCPcons);
    printf("        - actUDPcons: %d\n",stat->actUDPcons);
    printf("        - cloUDPcons: %d\n",stat->cloUDPcons);
    printf("        - actIPcons: %d\n",stat->actIPcons);
    printf("        - cloIPcons: %d\n",stat->cloIPcons);
    printf("        - actICMPcons: %d\n",stat->actICMPcons);
    printf("        - cloICMPcons: %d\n",stat->cloICMPcons);
    printf("        - actFRAGcons: %d\n",stat->actFRAGcons);
    printf("        - cloFRAGcons: %d\n",stat->cloFRAGcons);
    printf("\n");
  }

   switch (ptr->status & (INIT|STATUS|CLOSE)) {
      case   INIT:  str = get_man_init_string (ptr); break;
      case STATUS:  str = get_man_stat_string (ptr); break;
      case  CLOSE:  str = get_man_stat_string (ptr); break;
   }
 
   if (str) {
      switch (ptr->status & (INIT|STATUS|CLOSE)) {
         case   INIT: strcat (str, "INT"); ; break;
         case STATUS: strcat (str, "STA"); ; break;
         case  CLOSE: strcat (str, "CLO"); ; break;
      }
   }
    
   (void) printf ("\n%s\n", str);
}

/*
 * 	process_frag (prt);
 *		Process an IP fragment...
 * 		This routine is called by process_ip.
 */
process_frag (ptr)
struct WriteStruct *ptr;
{
   int vc = 0;
   char *str;
   char strbuf[256];
   struct fragWriteStruct *frag = &ptr->ws_ip_frag;

			/*
			 * Check to see what type of fragment
			 * it is, then call the routinge to print
			 * out the description.
			 */
   switch (ptr->status & (UDPPROTO|ICMPPROTO|TCPPROTO)) {
      case UDPPROTO:  str = get_udp_string (ptr); break;
      case ICMPPROTO: str = get_icmp_string (ptr); break;
      case TCPPROTO:  str = get_tcp_string (ptr); break;
      default:        str = get_ip_string (ptr); break;
   }

			/*
			 * Copy the word fragment into the output.
			 */
   bcopy ((char *) "frag", (char *)&str[19], 4);

   if (mflag) vc += 17;
   str[vc + 50] = ' ';
   sprintf (&str[vc + 69], " %-5d ", frag->frag_id);

   if (cflag) {
      sprintf (strbuf, "pk %2d  ex %4d  ob %4d  max %4d ",
          frag->fragnum, frag->totlen, frag->currlen,
          frag->maxfraglen);
      strcat (str, strbuf);
   }

   strcat (str, "TIM");
   (void) printf ("%s\n", str);
   fflush (stdout);
}




/*  REQUIRED ARGUS CLIENT ROUTINE
 *	process_tcp ((struct writeStruct *) ptr);
 *		This routine should process tcp events;
 */

process_tcp (ptr)
struct WriteStruct *ptr;
{
   process_argus_record (ptr);

   printf ("\n  ARGUS TCP: Common code calls process_tcp when TCP record encountered.\n");

   printf ("%s\n", get_tcp_string (ptr));
   fflush (stdout);
}


/*  REQUIRED ARGUS CLIENT ROUTINE
 *	process_icmp ((struct writeStruct *) ptr);
 *		This routine should process icmp events.
 */

process_icmp (ptr)
struct WriteStruct *ptr;
{
   process_argus_record (ptr);

   printf ("\nARGUS ICMP: Common code calls process_icmp when ICMP record encountered.\n");
   printf ("%s\n", get_icmp_string (ptr));
   fflush (stdout);
}


/*  REQUIRED ARGUS CLIENT ROUTINE
 * 	process_udp (ptr);
 * 		This routinte should process UDP events.
 */

process_udp (ptr)
struct WriteStruct *ptr;
{
   process_argus_record (ptr);

   printf ("\nARGUS UDP: Common code calls process_udp when UDP record encountered.\n");
   printf ("%s\n", get_udp_string (ptr));
   fflush (stdout);
}


/*  REQUIRED ARGUS CLIENT ROUTINE
 * 	process_ip(ptr);
 * 		This routine will process IP events
 * 		that are NOT ICMP,UDP, or TCP.
 */

process_ip (ptr)
struct WriteStruct *ptr;
{
   process_argus_record (ptr);

   printf ("\nARGUS  IP: Common code calls process_ip when the record is NOT \n");
   printf ("\n         : a TCP, UDP or ICMP record.\n");

   if (ptr->status & FRAG_ONLY)
      process_frag (ptr);
   else
      printf ("%s\n", get_ip_string (ptr));
   fflush (stdout);
}


/*
 *  The remaining procedures acuatlly do the work... they
 *  analyze the records and generate output format.
 */

/*
 * print_date (ptr, date)
 * struct WriteStruct *ptr;
 * char *date;
 * 
 * Print_date writes the date as supplied in an argus record
 * into the buffer pointed to by *date,  based on the -u
 * and -l flags.
 *
 * print_date can return one of two formats, a month day year
 * style string for the startime (default) or the lasttime (-l),
 * by default, or a second.microsec format if the -u flag is used.
 * 
 */

void
print_date(ptr, date)
struct WriteStruct *ptr;
char *date;
{
   struct timeval *time;
   struct tm *tm;
   
   if (lflag) {
      time = &ptr->ws_ip.lasttime;
      tm = &tm_lasttime;
   } else {
      time = &ptr->ws_ip.startime;
      tm = &tm_startime;
   }

  if (uflag)
    sprintf(date, "%9ld.%-8ld", (unsigned long) time->tv_sec,
                                (unsigned long) time->tv_usec);
  else
    strftime ((char *) date, 128, "%a %m/%d %T", tm);
}


/*
 *   get_tcp_string (ptr)
 *
 *  This routine is passed a pointer to an argus record, and 
 *  returns an output string for the record. 
 *
 */

char *
get_tcp_string (ptr)
struct WriteStruct *ptr;
{
   int i, src_bytes, dst_bytes, src_count, dst_count, ind, vc = 0;
   unsigned int state;
   struct in_addr *srcAddr, *dstAddr;
   struct ether_addr *esrcAddr, *edstAddr;

   char date[128], fmtstr[256];

   char *dstString, *srcString, *protoStr;
   char *edstString, *esrcString;
   char *processStr = NULL;
   unsigned short srcPort, dstPort;
   u_long port, srcSeq, dstSeq, srcDst, dstDst;

   /* Zero out the memory allocated above, and the global   */
   /* variable buf.			                    */      	
   bzero (buf, sizeof (buf));

   edstAddr = &ptr->ws_ip_phys.etherdst;
   esrcAddr = &ptr->ws_ip_phys.ethersrc;
   esrcString = etheraddr_string ((u_char *) esrcAddr);
   edstString = etheraddr_string ((u_char *) edstAddr);

   printf ("    PHY Structure: \n");
   printf ("      - ethersrc: %s\n", esrcString);  
   printf ("      - etherdst: %s\n", edstString);  
   printf ("    IP Structure:\n");

   if (uflag)
      sprintf(date, "%9ld.%-6ld", (unsigned long) ptr->ws_ip.startime.tv_sec,
	                          (unsigned long) ptr->ws_ip.startime.tv_usec);
   else
       strftime ((char *) date, 128, "%a %m/%d %T", &tm_startime);

   printf("      - startime: %s\n", date);

   if (uflag)
      sprintf(date, "%9ld.%-6ld", (unsigned long) ptr->ws_ip.lasttime.tv_sec,
	                          (unsigned long) ptr->ws_ip.lasttime.tv_usec);
   else
       strftime ((char *) date, 128, "%a %m/%d %T", &tm_lasttime);

   printf("      - lasttime: %s\n", date);

   printf ("      - src addr: %s\n", ipaddr_string(&ptr->ws_ip.src));
   printf ("      - dst addr: %s\n", ipaddr_string(&ptr->ws_ip.dst));

   printf ("    TCP Structure:\n");

   port = ptr->ws_ip.port;  /* Extract the IP port number.  */
   srcPort = ((unsigned short *) &port)[0];
   dstPort = ((unsigned short *) &port)[1];
   printf ("      -     src port num: %d\n", srcPort);  
   printf ("      -     dst port num: %d\n", dstPort);  

   if ((ptr->status & SAW_SYN) &&
            !(ptr->status & (SAW_SYN_SENT|CON_ESTABLISHED|RESET))) {
      printf ("      - src base seq num: 0x%x\n",ptr->ws_ip_inittcp.seq);
   } else {
      if ((ptr->status & (SAW_SYN | SAW_SYN_SENT)) &&
                         !(ptr->status & (CON_ESTABLISHED))) {
         printf ("      - dst base seq num: 0x%x\n",ptr->ws_ip_inittcp.seq);
         src_bytes = 0; dst_bytes = 0;
      } else {
         printf ("      -   src byte count: %d\n",ptr->ws_ip_tcp.src_bytes);
         printf ("      -   dst byte count: %d\n",ptr->ws_ip_tcp.dst_bytes);
      }
   }

   printf ("      -    src pkt count: %d\n",ptr->ws_ip_tcp.src_count);
   printf ("      -    dst pkt count: %d\n\n",ptr->ws_ip_tcp.dst_count);

   print_date(ptr,date);    /* Date is filled upon return.  */

   state = ptr->status &    /* Check current state of flow. */
         (SAW_SYN | NORMAL_CLOSE | SAW_SYN_SENT | CON_ESTABLISHED);

   printf ("    TCP Protocol State Report:\n");
   if (!(ptr->status & (SAW_SYN | SAW_SYN_SENT | NORMAL_CLOSE)))
      printf ("      -  TCP Direction Unknown\n");
   if (ptr->status & SAW_SYN) printf ("      -  TCP SYN\n");
   if (ptr->status & SAW_SYN_SENT) printf ("      -  TCP SYN ACK\n");
   if (ptr->status & CON_ESTABLISHED) printf ("      -  ESTABLISHED\n");
   if (ptr->status & CLOSE_WAITING) printf ("      -  TCP FIN_WAIT\n");
   if (ptr->status & PKTS_RETRANS) printf ("      -  TCP Packet Retransmissions\n");
   if (ptr->status & SRC_WINDOW_SHUT) printf ("      -  TCP Source Flow Control\n");
   if (ptr->status & DST_WINDOW_SHUT) printf ("      -  TCP Destination Flow Control\n");
   if (ptr->status & NORMAL_CLOSE) printf ("      -  TCP Closed\n");
   if (ptr->status & RESET) printf ("      -  TCP Reset\n");
   printf ("\n");

   protoStr = (nflag > 1) ? "6  " : "tcp";

   if (mflag) {            /* Mac address to be printed?    */
      vc = 17;
      strcpy (fmtstr, 
         "%s %17.17s %17.17s       %3s %15.15s.%-5.5s  -  %15.15s.%-5.5s ");
   } else
      strcpy (fmtstr, "%s      %3s %15.15s.%-5.5s  -  %15.15s.%-5.5s ");
  
   if (cflag)              /* Packet and byte counts to be  */
                           /* printed? Append to fmt string */ 
      strcat (fmtstr, "%-6d %-6d  %-9d %-9d");
               
                           /* Determine which way the arrow */  
   if (state) {            /* will go...                    */    
      if (state & NORMAL_CLOSE) {
         fmtstr[29 + vc] = '>';
         processStr = process_state_strings[3];
      } else {
         if (ptr->status & DETAIL) {
            if (state & CLOSE_WAITING) {
               fmtstr[27 + vc] = '<'; fmtstr[29 + vc] = '>';
               processStr = process_state_strings[6];
            } else
            if (state & CON_ESTABLISHED) {
               fmtstr[27 + vc] = '<'; fmtstr[29 + vc] = '>';
               processStr = process_state_strings[2];
            } else
            if (state & SAW_SYN_SENT) {
               fmtstr[27 + vc] = '<';
               processStr = process_state_strings[1];
            } else
            if (state & SAW_SYN) {
               fmtstr[29 + vc] = '>';
               processStr = process_state_strings[0];
            }
         } else {
            fmtstr[29 + vc] = '>';
            if (!(state & (SAW_SYN | SAW_SYN_SENT)))
               fmtstr[27 + vc] = '<';
            processStr = process_state_strings[3];
         }
      }
   }

   if (!(ptr->status & (SAW_SYN | SAW_SYN_SENT)) &&
      (!(ptr->ws_ip_tcp.src_count) || !(ptr->ws_ip_tcp.dst_count))) {
      fmtstr[28 + vc] = '?';
   }
                           /* Check for a reset.            */  
   if (ptr->status & RESET) {
      fmtstr[28 + vc] = '|';
      processStr = process_state_strings[5];
   } 

                           /* Check for a timeout.          */  
   if (ptr->status & TIMED_OUT) {
      fmtstr[28 + vc] = 'o';
      processStr = process_state_strings[4];
   }
                           /* Append the processStr to the  */
                           /* end of the format statement.  */      
   strcat (fmtstr, processStr);

                           /* Mark output with * if retrans */ 
   if (ptr->status & PKTS_RETRANS) fmtstr[vc + 3] = '*';
 
                           /* Look at other options...      */
                           /* modify fmtstr appropriately.  */
   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 2] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 2] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 2] = 'T'; break;
         case SECURITY:     fmtstr[vc + 2] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 2] = 'R'; break;
         case SATNETID:     fmtstr[vc + 2] = 'N'; break;
         default:           fmtstr[vc + 2] = '?'; break;
      }
   }
                           /* Mark these too...             */      
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';
   if (ptr->status & IPROUTED)  fmtstr[vc + 5] = 'R';

                           /* Get the ethernet addresses... */
                           /* Why not only when mflag set?  */     
   edstAddr = &ptr->ws_ip_phys.etherdst;
   esrcAddr = &ptr->ws_ip_phys.ethersrc;

                           /* Get the IP src & dest addr... */
   srcAddr = &ptr->ws_ip.src; dstAddr = &ptr->ws_ip.dst;
                           /* And the port numbers...       */
   srcPort = ((unsigned short *) &port)[0];
   dstPort = ((unsigned short *) &port)[1];
                           /* And the number of bytes...    */
   src_count = ptr->ws_ip_tcp.src_count;
   dst_count = ptr->ws_ip_tcp.dst_count;

   if ((ptr->status & SAW_SYN) &&
            !(ptr->status & (SAW_SYN_SENT|CON_ESTABLISHED|RESET))) {
      srcSeq = ptr->ws_ip_inittcp.seq;
      srcDst = ptr->ws_ip_inittcp.addr;
      src_bytes = 0; dst_bytes = 0;
   } else {
      if ((ptr->status & (SAW_SYN | SAW_SYN_SENT)) &&
                         !(ptr->status & (CON_ESTABLISHED|RESET))) {
         dstSeq = ptr->ws_ip_inittcp.seq;
         dstDst = ptr->ws_ip_inittcp.addr;
         src_bytes = 0; dst_bytes = 0;
      } else {
         src_bytes = ptr->ws_ip_tcp.src_bytes;
         dst_bytes = ptr->ws_ip_tcp.dst_bytes;
      }
   }

                           /* Convert addresses to strings. */
   srcString = ipaddr_string (srcAddr);
   dstString = ipaddr_string (dstAddr);

                           /* Convert addresses to strings. */
   esrcString = etheraddr_string ((u_char *) esrcAddr);
   edstString = etheraddr_string ((u_char *) edstAddr);

                           /* Check for FTP or X windows... */
                           /* These flows are in the        */
                           /* opposite direction.           */

   if ((srcPort == 20) || ((srcPort >= 6000) && (srcPort < 6010))) {
      int swap27 = 0, swap29 = 0;
      if (fmtstr[27 + vc] == '<') swap27 = 1;
      if (fmtstr[29 + vc] == '>') swap29 = 1;
      if (!(swap27 && swap29)) {
         if (swap27) fmtstr[29 + vc] = '>', fmtstr[27 + vc] = ' ';
         if (swap29) fmtstr[27 + vc] = '<', fmtstr[29 + vc] = ' ';
      }
 
                           /* Check for Mac address flag... */
      if (mflag)
         sprintf (buf, fmtstr, date,
            edstString, esrcString, protoStr,
            dstString, tcpport_string(dstPort),
            srcString, tcpport_string(srcPort),
            dst_count, src_count, dst_bytes, src_bytes);
      else
         sprintf (buf, fmtstr, date, protoStr,
            dstString, tcpport_string(dstPort),
            srcString, tcpport_string(srcPort),
            dst_count, src_count, dst_bytes, src_bytes);
   } else
      if (mflag)
         sprintf (buf, fmtstr, date,
            esrcString, edstString, protoStr,
            srcString, tcpport_string(srcPort),
            dstString, tcpport_string(dstPort),
            src_count, dst_count, src_bytes, dst_bytes);
      else
         sprintf (buf, fmtstr, date, protoStr,
            srcString, tcpport_string(srcPort),
            dstString, tcpport_string(dstPort),
            src_count, dst_count, src_bytes, dst_bytes);
 
   return (buf);
}


char *
get_icmp_string (ptr)
struct WriteStruct *ptr;
{
   int i, vc = 0, rev = 0, hlen, srccnt = 0, dstcnt = 0;
   struct ip *oip;
   struct udphdr *ouh;
   struct icmpWriteStruct *icmp;
   char fmtstr[64], *blankstring = "     ", icmptype[32];
   char extendedstring[32];
   char *dstString, *srcString, *protoStr;
   char *edstString, *esrcString;
   char startdate[128];
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;
   extern u_long getnetnumber (), ipaddrtonetmask ();

     			/*
			 * Check for the existance of a pointer first...
			 * Error checking?  Why would the common argus
			 * code call this routine without a ptr?
 			 */ 
   if (ptr) {
      unsigned char  ra_icmp_type, ra_icmp_code;
      unsigned short ra_icmp_data;
      unsigned long  ra_src_addr = 0, ra_dst_addr = 0, ra_gw_addr = 0;

			/*
			 * Look into the struct for the ICMP specifics.
			 */
      icmp = &ptr->ws_ip_icmp;

      bzero (icmptype, sizeof (icmptype));
      bzero (buf, sizeof (buf));
      bzero (extendedstring, sizeof (extendedstring));
      bzero (icmptype, sizeof (icmptype));

      print_date(ptr,startdate);
      /* strftime ((char *) &startdate, 128, "%a %m/%d %T", &tm_startime); */

      if (!(ptr->status & (CON_ESTABLISHED | TIMED_OUT))) {
         ra_icmp_type = icmp->type;
         ra_icmp_code = icmp->code;
         ra_icmp_data = icmp->data;
         ra_src_addr =  icmp->srcaddr.s_addr;
         ra_dst_addr =  icmp->dstaddr.s_addr;
         ra_gw_addr  =  icmp->gwaddr.s_addr;

			/*
			 * Not really sure what's up here 
      } else {		 */
         ra_icmp_type = ((unsigned char *) &ptr->ws_ip_udp.src_bytes) [0];
         ra_icmp_code = ((unsigned char *) &ptr->ws_ip_udp.src_bytes) [1];
         ra_icmp_data = ((unsigned short *)&ptr->ws_ip_udp.src_bytes) [1];
         ra_src_addr  =  (unsigned long) ptr->ws_ip_udp.dst_bytes;
         ra_dst_addr  =  (unsigned long) ptr->ws_ip_udp.dst_bytes;
         ra_gw_addr  =  (unsigned long) ptr->ws_ip_udp.dst_bytes;
      }
			/*
			 * Fill in the ICMP message type.  See the
			 * icmptypestr def near top of this file.
`			 */
      if (ra_icmp_type < (ICMP_MAXTYPE + 1))
         strcpy (icmptype, icmptypestr[ra_icmp_type]);
      else
         strcpy (icmptype, "UNK");

			/*
			 * Now switch on the type... report info
			 */
      switch (ra_icmp_type) {
			/* Unreachable, Net, host, proto, port, etc.  */
         case ICMP_UNREACH:
            switch (ra_icmp_code) {
			/* Unreachable, Net.         */
               case ICMP_UNREACH_NET:
                  strcat (icmptype, "N");
                  if (icmp->dstaddr.s_addr) {
                     u_long addr = ra_dst_addr;
                     addr &= ipaddrtonetmask (addr);
                     sprintf (extendedstring, "%s", 
                              ipaddr_string (&addr));
                  }
                  break;
			/* Unreachable, Host.        */
               case ICMP_UNREACH_HOST:
                  strcat (icmptype, "H");
                  if (ra_dst_addr)
                     sprintf (extendedstring, "%s", 
                           ipaddr_string (&ra_dst_addr));
                  break;

               case ICMP_UNREACH_PROTOCOL:
                  strcat (icmptype, "O");
                  if (ra_icmp_data && (ra_icmp_data < IPPROTOSTR))
                     sprintf (extendedstring,"%s",
                        ip_proto_string[ra_icmp_data]);
                  break;

               case ICMP_UNREACH_PORT: {
                  char index = *((char *)&ra_src_addr);
                  strcat (icmptype, "P"); 

                  if ((ra_icmp_data & ((index < IPPROTOSTR)) && (index > 0))
                          && (minor_version > 5)) {
                     sprintf (extendedstring, "%s port %d",
                        ip_proto_string[*((unsigned char *)&ra_src_addr)],
                           ra_icmp_data);

                  } else if (ra_icmp_data)
                     sprintf (extendedstring, "%d", ra_icmp_data);
                  break;
               }
               case ICMP_UNREACH_NEEDFRAG:
                  strcat (icmptype, "F"); break;
               case ICMP_UNREACH_SRCFAIL:
                  strcat (icmptype, "S"); break;
            }
            break;

			/*
			 * Mask reply... who uses this?
			 */
         case ICMP_MASKREPLY:
               if (ra_src_addr)
                  sprintf (extendedstring, "0x%08x", ra_src_addr);
            break;

			/*
			 * The 'ol ICMP redirect.
			 */
         case ICMP_REDIRECT:
            switch (ra_icmp_code) {
            case ICMP_REDIRECT_NET:
               (void) sprintf (extendedstring, "net %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_HOST:
               (void) sprintf (extendedstring, "host %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSNET:
               (void) sprintf (extendedstring, "tosN %s",
                         ipaddr_string (&ra_gw_addr));
               break;

            case ICMP_REDIRECT_TOSHOST:
               (void) sprintf (extendedstring, "tosH %s",
                         ipaddr_string (&ra_gw_addr));
               break;
            }
            break;

         case ICMP_ECHOREPLY:
            rev = 1;
            break;

         case ICMP_TIMXCEED:
               (void) sprintf (extendedstring, "%s",
                         icmp->code ? "reassembly" : "in-transit");
               break;

         case ICMP_PARAMPROB:
         case ICMP_SOURCEQUENCH:
         case ICMP_ECHO:
         case ICMP_TSTAMP:
         case ICMP_TSTAMPREPLY:
         case ICMP_IREQ:
         case ICMP_IREQREPLY:
         case ICMP_MASKREQ:
         default:
            break;
      }
   
      protoStr = (nflag > 1) ? "1   " : "icmp";
      if (rev) {
         edstString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.ethersrc);
         esrcString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.etherdst);
         dstString = ipaddr_string (&ptr->ws_ip.src);
         srcString = ipaddr_string (&ptr->ws_ip.dst);
      } else {
         esrcString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.ethersrc);
         edstString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.etherdst);
         srcString = ipaddr_string (&ptr->ws_ip.src);
         dstString = ipaddr_string (&ptr->ws_ip.dst);
      }
   
      if (mflag) {
         vc = 15;
         sprintf (fmtstr, "%s",
              "%s %17.17s %17.17s      icmp %15.15s %-5.5s  -  %15.15s %-5.5s");
      } else {
         sprintf (fmtstr, "%s",
              "%s     %-4.4s %15.15s %-5.5s  -  %15.15s %-5.5s");
      }
  
      if (rev) {fmtstr[29 + vc] = '<'; fmtstr[31 + vc] = ' ';}
      else     {fmtstr[29 + vc] = ' '; fmtstr[31 + vc] = '>';}
   
      if (cflag) strcat (fmtstr, " %-6d %-6d  %-17.17s ");
   
      if (ptr->status & CON_ESTABLISHED) {
         srccnt = ptr->ws_ip_udp.src_count;
         dstcnt = ptr->ws_ip_udp.dst_count;
         fmtstr[29 + vc] = ' '; fmtstr[31 + vc] = ' ';
         if (srccnt)
            fmtstr[31 + vc] = '>';
         if (dstcnt)
            fmtstr[29 + vc] = '<';
      } else {
         if (rev) dstcnt = 1;
         else     srccnt = 1;
      }

      strcat (fmtstr, " %3.3s");
   
      if (ptr->status & IPOPTIONMASK) {
         switch (ptr->status & IPOPTIONMASK) {
            case SSRCROUTE:    fmtstr[vc + 2] = 'S'; break;
            case LSRCROUTE:    fmtstr[vc + 2] = 'L'; break;
            case TIMESTAMP:    fmtstr[vc + 2] = 'T'; break;
            case SECURITY:     fmtstr[vc + 2] = '+'; break;
            case RECORDROUTE:  fmtstr[vc + 2] = 'R'; break;
            case SATNETID:     fmtstr[vc + 2] = 'N'; break;
            default:           fmtstr[vc + 2] = '?'; break;
         }
      }
      if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
      if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';
      if (ptr->status & IPROUTED)  fmtstr[vc + 5] = 'R';
  
      if (mflag) {
         if (cflag) {
            sprintf (buf, fmtstr, startdate, esrcString, edstString,
               srcString, blankstring, dstString, blankstring, 
            srccnt, dstcnt, extendedstring, icmptype);
         } else
            sprintf (buf, fmtstr, startdate, esrcString, edstString,
               srcString, blankstring, dstString, icmptype,
               icmptype);
      } else {
         if (cflag) {
            sprintf (buf, fmtstr, startdate, protoStr,
               srcString, blankstring, dstString, blankstring,
               srccnt, dstcnt, extendedstring, icmptype);
         } else
            sprintf (buf, fmtstr, startdate, protoStr,
               srcString, blankstring, dstString, blankstring,
               icmptype);
      }
   }
   return (buf);
}


char *
get_udp_string (ptr)
struct WriteStruct *ptr;
{
   int i, vc = 0;
   char *blankstring = "*    ";
   char *dstString, *srcString, *udpString;
   char *edstString, *esrcString, *protoStr;
   char date[128], fmtstr[256];
   u_short srcport, dstport;
   u_long port;
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;

   bzero (buf, sizeof (buf));
   esrcString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.ethersrc);
   edstString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.etherdst);

   printf ("    PHY Structure: \n");
   printf ("      - ethersrc: %s\n", etheraddr_string((u_char *) &ptr->ws_ip_phys.ethersrc));  
   printf ("      - etherdst: %s\n", etheraddr_string((u_char *) &ptr->ws_ip_phys.etherdst));  
   printf ("    IP Structure:\n");


   if (uflag)
      sprintf(date, "%9ld.%-6ld", (unsigned long) ptr->ws_ip.startime.tv_sec,
	                          (unsigned long) ptr->ws_ip.startime.tv_usec);
   else
       strftime ((char *) date, 128, "%a %m/%d %T", &tm_startime);

   printf("      - startime: %s\n", date);

   if (uflag)
      sprintf(date, "%9ld.%-6ld", (unsigned long) ptr->ws_ip.lasttime.tv_sec,
	                          (unsigned long) ptr->ws_ip.lasttime.tv_usec);
   else
       strftime ((char *) date, 128, "%a %m/%d %T", &tm_lasttime);

   printf("      - lasttime: %s\n", date);

   printf ("      - src addr: %s\n", ipaddr_string(&ptr->ws_ip.src));
   printf ("      - dst addr: %s\n", ipaddr_string(&ptr->ws_ip.dst));

   printf ("    UDP Structure:\n");
   printf ("      -  src pkt count: %d\n",ptr->ws_ip_udp.src_count);
   printf ("      -  dst pkt count: %d\n",ptr->ws_ip_udp.dst_count);
   printf ("      - src byte count: %d\n",ptr->ws_ip_udp.src_bytes);
   printf ("      - dst byte count: %d\n",ptr->ws_ip_udp.dst_bytes);

   port = ptr->ws_ip.port;  /* Extract the IP port number.  */
   srcport = ((unsigned short *) &port)[0];
   dstport = ((unsigned short *) &port)[1];
   printf ("      -   src port num: %d\n", srcport);  
   printf ("      -   dst port num: %d\n\n", dstport);  


   srcString = ipaddr_string (&ptr->ws_ip.src.s_addr);
   dstString = ipaddr_string (&ptr->ws_ip.dst.s_addr);
   srcport = ((u_short *)&ptr->ws_ip.port)[0];
   dstport = ((u_short *)&ptr->ws_ip.port)[1];

   print_date(ptr,date);

   protoStr = (nflag > 1) ? "17 " : "udp";

   if (mflag) {
      vc = 17;
      strcpy (fmtstr, 
         "%s %17.17s %17.17s       %3s %15.15s.%-5.5s  -  %15.15s.%-5.5s ");
   } else
      strcpy (fmtstr, "%s      %3s %15.15s.%-5.5s  -  %15.15s.%-5.5s ");

   if (cflag)
      strcat (fmtstr, "%-6d %-6d  %-9d %-9d");

   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 2] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 2] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 2] = 'T'; break;
         case SECURITY:     fmtstr[vc + 2] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 2] = 'R'; break;
         case SATNETID:     fmtstr[vc + 2] = 'N'; break;
         default:           fmtstr[vc + 2] = '?'; break;
      }
   }
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';
   if (ptr->status & IPROUTED)  fmtstr[vc + 5] = 'R';

   fmtstr[27 + vc] = (ptr->ws_ip_udp.dst_count) ? '<' : ' '; 
   fmtstr[29 + vc] = (ptr->ws_ip_udp.src_count) ? '>' : ' '; 
   if ((ptr->status & TIMED_OUT)) strcat (fmtstr, "TIM");
   else if (ptr->status & CON_ESTABLISHED) {
      if ((ptr->ws_ip_udp.src_count == 1) && (ptr->ws_ip_udp.dst_count == 1))
         strcat (fmtstr, "ACC");
      else strcat (fmtstr, "CON");
   } else if (ptr->status & UDP_INIT) strcat (fmtstr, "INT");

   if (mflag)
      sprintf (buf, fmtstr, date,
         esrcString, edstString,
         protoStr,
         srcString, udpport_string (srcport), 
         dstString, udpport_string (dstport), 
         ptr->ws_ip_udp.src_count, ptr->ws_ip_udp.dst_count,
         ptr->ws_ip_udp.src_bytes, ptr->ws_ip_udp.dst_bytes);
   else
      sprintf (buf, fmtstr, date, protoStr,
         srcString, udpport_string (srcport),
         dstString, udpport_string (dstport), 
         ptr->ws_ip_udp.src_count, ptr->ws_ip_udp.dst_count,
         ptr->ws_ip_udp.src_bytes, ptr->ws_ip_udp.dst_bytes);

   return (buf);
}


char *
get_ip_string (ptr)
struct WriteStruct *ptr;
{
   int i, vc = 0;
   char *blankstring = "*    ";
   char *dstString, *srcString, *protoStr ;
   char *edstString, *esrcString;
   char date[128], fmtstr[256], protoStrbuf[16];
   extern char *timestatmp_fmt;
   extern long timestamp_scale, thiszone;
   int src_count, dst_count, src_bytes, dst_bytes;
   u_char proto;
   
   bzero (buf, sizeof (buf));
   srcString = ipaddr_string (&ptr->ws_ip.src.s_addr);
   dstString = ipaddr_string (&ptr->ws_ip.dst.s_addr);
   esrcString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.ethersrc);
   edstString = etheraddr_string ((u_char *) &ptr->ws_ip_phys.etherdst);
   src_count = ptr->ws_ip_udp.src_count; dst_count = ptr->ws_ip_udp.dst_count;
   src_bytes = ptr->ws_ip_udp.src_bytes; dst_bytes = ptr->ws_ip_udp.dst_bytes;

   print_date(ptr,date);

   if (mflag) {
      vc = 17;
      strcpy (fmtstr, 
            "%s %17.17s %17.17s     %-5.5s%20.20s  -  %20.20s  ");
   } else
      strcpy (fmtstr, "%s      %3s %15.15s        -  %15.15s       ");

   fmtstr[26 + vc] = (dst_count) ? '<' : ' '; 
   fmtstr[28 + vc] = (src_count) ? '>' : ' '; 

   if (cflag)
      strcat (fmtstr, "%-6d %-6d  %-9d %-9d");

   if ((ptr->status & TIMED_OUT)) strcat (fmtstr, "TIM");
   else if (ptr->status & CON_ESTABLISHED) {
      if ((ptr->ws_ip_udp.src_count == 1) && (ptr->ws_ip_udp.dst_count == 1))
         strcat (fmtstr, "ACC");
      else strcat (fmtstr, "CON");
   } else if (ptr->status & UDP_INIT) strcat (fmtstr, "INT");

   if (ptr->status & IPOPTIONMASK) {
      switch (ptr->status & IPOPTIONMASK) {
         case SSRCROUTE:    fmtstr[vc + 2] = 'S'; break;
         case LSRCROUTE:    fmtstr[vc + 2] = 'L'; break;
         case TIMESTAMP:    fmtstr[vc + 2] = 'T'; break;
         case SECURITY:     fmtstr[vc + 2] = '+'; break;
         case RECORDROUTE:  fmtstr[vc + 2] = 'R'; break;
         case SATNETID:     fmtstr[vc + 2] = 'N'; break;
         default:           fmtstr[vc + 2] = '?'; break;
      }
   }
   if (ptr->status & MULTIADDR) fmtstr[vc + 4] = 'M';
   if (ptr->status & FRAGMENTS) fmtstr[vc + 4] = 'F';
   if (ptr->status & IPROUTED)  fmtstr[vc + 5] = 'R';

   proto = ((unsigned char *)&ptr->ws_ip.port)[3];
   sprintf (protoStrbuf, "%u", proto);
   protoStr = (nflag > 1) ? protoStrbuf :
             proto >= IPPROTOSTR ? "unas" : ip_proto_string[proto];

   if (mflag && cflag)
      sprintf (buf, fmtstr, date, 
            esrcString, edstString, protoStr,
            srcString, dstString, 
            src_count, dst_count,
            src_bytes, dst_bytes);
   else
   if (cflag)
      sprintf (buf, fmtstr, date, protoStr,
            srcString, dstString, 
            src_count, dst_count,
            src_bytes, dst_bytes);
   else
   if (mflag)
      sprintf (buf, fmtstr, date, 
            esrcString, edstString,  protoStr,
            srcString, dstString);
   else
      sprintf (buf, fmtstr, date, protoStr,
            srcString, dstString);

   return (buf);
}


char *
get_man_init_string (ptr)
struct WriteStruct *ptr;
{
   int i, ind, vc = 0;
   unsigned int state;
   struct tm *tm;
   unsigned int *netAddr, *netMask;
   char *dstString, *srcString, *protoStr;
   char date [128], fmtstr[256], *blankstring = "     ";
   char *processStr = NULL;

   bzero (fmtstr, 256);
   bzero (buf, sizeof (buf));

   if (uflag)
      sprintf(date, "%9ld.%-8ld", (unsigned long) ptr->ws_init.now.tv_sec,
	                          (unsigned long) ptr->ws_init.now.tv_usec);
   else
      if (tm = localtime ((time_t *)&ptr->ws_init.now))
         strftime ((char *) date, 128, "%a %m/%d %T", tm);

   protoStr =  "man";

   if (mflag) {
      vc = 17;
      strcpy (fmtstr, 
         "%s %17.17s %17.17s       %3s %15.15s %-5.5s     %15.15s %-5.5s ");
   } else
      strcpy (fmtstr, "%s      %3s %15.15s %-5.5s     %15.15s %-5.5s ");

   if (cflag)
      strcat (fmtstr, "%-6d %-6d  %-9d %-9d");

   netAddr  = (unsigned int *) &ptr->ws_init.localnet;
   if (*netAddr)
      while ((*netAddr & 0xff000000) == 0)
         *netAddr <<= 8;

   netMask = (unsigned int *) &ptr->ws_init.netmask;
   if (*netMask)
      while ((*netMask & 0xff000000) == 0)
         *netMask <<= 8;

   srcString = ipaddr_string (netAddr);
   dstString = ipaddr_string (netMask);

   if (mflag)
      sprintf (buf, fmtstr, date,
         blankstring, blankstring, protoStr,
         srcString, blankstring,
         dstString, blankstring,
         0, 0, 0, 0);
   else
      sprintf (buf, fmtstr, date, protoStr,
         srcString, blankstring, dstString, blankstring,
         0, 0, 0, 0);

   return (buf);
}


char *
get_man_stat_string (ptr)
struct WriteStruct *ptr;
{
   int i, ind, vc = 0;
   unsigned int state;
   struct tm *tm;
   long *netAddr, *netMask;
   char *dstString, *srcString, *protoStr;
   char date [128], fmtstr[256], *blankstring = "     ";
   char *processStr = NULL;

   bzero (fmtstr, 256);
   bzero (buf, sizeof (buf));

   if (uflag)
      sprintf(date, "%9ld.%-8ld", (unsigned long) ptr->ws_init.now.tv_sec,
	                          (unsigned long) ptr->ws_init.now.tv_usec);
   else
      if (tm = localtime ((time_t *)&ptr->ws_stat.now.tv_sec))
         strftime ((char *) date, 128, "%a %m/%d %T", tm);

   protoStr =  "man";

   if (mflag) {
      vc = 17;
      strcpy (fmtstr, 
         "%s %17.17s %17.17s       %3s  pkts %5d  drops %5d  ");
   } else
      strcpy (fmtstr, "%s      %3s  pkts %5d  drops %5d  ");

   if (cflag) {
      strcat (fmtstr, "ip %3hd %3hd  t %3hd %3hd  u %3hd %3hd  i %3hd %3hd  f %3hd %3hd  ");

      if (mflag)
         sprintf (buf, fmtstr, date,
            blankstring, blankstring, protoStr,
            ptr->ws_stat.pktsRcvd, ptr->ws_stat.pktsDrop, 
            ptr->ws_stat.actIPcons, ptr->ws_stat.cloIPcons,
            ptr->ws_stat.actTCPcons, ptr->ws_stat.cloTCPcons,
            ptr->ws_stat.actUDPcons, ptr->ws_stat.cloUDPcons,
            ptr->ws_stat.actICMPcons, ptr->ws_stat.cloICMPcons,
            ptr->ws_stat.actFRAGcons, ptr->ws_stat.cloFRAGcons);
      else
         sprintf (buf, fmtstr, date, protoStr,
            ptr->ws_stat.pktsRcvd, ptr->ws_stat.pktsDrop, 
            ptr->ws_stat.actIPcons, ptr->ws_stat.cloIPcons,
            ptr->ws_stat.actTCPcons, ptr->ws_stat.cloTCPcons,
            ptr->ws_stat.actUDPcons, ptr->ws_stat.cloUDPcons,
            ptr->ws_stat.actICMPcons, ptr->ws_stat.cloICMPcons,
            ptr->ws_stat.actFRAGcons, ptr->ws_stat.cloFRAGcons);
   } else {
      strcat (fmtstr, " flows %8d       ");

      if (mflag)
         sprintf (buf, fmtstr, date,
            blankstring, blankstring, protoStr,
            ptr->ws_stat.pktsRcvd, ptr->ws_stat.pktsDrop,
            ptr->ws_stat.actIPcons + ptr->ws_stat.cloIPcons +
            ptr->ws_stat.actTCPcons + ptr->ws_stat.cloTCPcons +
            ptr->ws_stat.actUDPcons + ptr->ws_stat.cloUDPcons +
            ptr->ws_stat.actICMPcons + ptr->ws_stat.cloICMPcons +
            ptr->ws_stat.actFRAGcons + ptr->ws_stat.cloFRAGcons);
      else
         sprintf (buf, fmtstr, date, protoStr,
            ptr->ws_stat.pktsRcvd, ptr->ws_stat.pktsDrop, 
            ptr->ws_stat.actIPcons + ptr->ws_stat.cloIPcons +
            ptr->ws_stat.actTCPcons + ptr->ws_stat.cloTCPcons +
            ptr->ws_stat.actUDPcons + ptr->ws_stat.cloUDPcons +
            ptr->ws_stat.actICMPcons + ptr->ws_stat.cloICMPcons +
            ptr->ws_stat.actFRAGcons + ptr->ws_stat.cloFRAGcons);
   }

   return (buf);
}

