
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * argus_util.c - supports connection tracking 
 *
 * Useful routines to support argus package.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 */


#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>

#include <compat.h>
#include <pcap.h>
#include <pcap-int.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <addrtoname.h>

#include <cons_tcp.h>

#include <malloc.h>
#include <string.h>

extern fd_set readmask, writemask, exceptmask;

struct callback {
   pcap_handler function;
   int type;
};

static struct callback callbacks[] = {
   { cons_ether_packet,  DLT_EN10MB },
   { cons_fddi_packet,   DLT_FDDI },
   { cons_raw_packet,    DLT_RAW },
   { NULL,               DLT_SLIP },
   { NULL,               DLT_PPP },
   { NULL,               DLT_NULL },
   { NULL,               0 },
};


extern int lfd;
extern pcap_t *pd;

void check_all_timeouts ();
static void wrapup ();

int readingoffline = 0;

void
argus_loop (p, callback)
pcap_t *p;
pcap_handler callback;
{
   int pfd = 0;
   int width = 1;
   struct timeval wait, tvpbuf, *tvp = &tvpbuf;
   struct argtimeval argtvpbuf, *argtvp = &argtvpbuf;
   extern char *rfile;
   
   wait.tv_sec = 1; wait.tv_usec = 0;

   if (p && callback) {
      if ((pfd = p->fd) >= 0) {
         if (pfd >= 0) FD_SET(pfd, &readmask);
         if (lfd >= 0) FD_SET(lfd, &readmask);
         width += (pfd > lfd) ? pfd : lfd;

         while (select (width, &readmask, NULL, NULL, &wait) >= 0) {
            gettimeofday (tvp, NULL);

            if ((lfd >= 0) && FD_ISSET(lfd, &readmask)) {
               argtvp->tv_sec  = (arg_uint32) tvp->tv_sec;
               argtvp->tv_usec = (arg_uint32) tvp->tv_usec;
               check_client_status (lfd, argtvp);
            }

            if (FD_ISSET(pfd, &readmask))
               pcap_read (p, -1, callback, (u_char *) NULL);

            if (updatetime (tvp)) check_all_timeouts (tvp);

            if (lfd >= 0) FD_SET(lfd, &readmask);
            if (pfd >= 0) FD_SET(pfd, &readmask);

	    wait.tv_sec = 1; wait.tv_usec = 0;
	  }
      } else {
         readingoffline++;
         pcap_offline_read (p, -1, callback, (u_char *) NULL);
      }
   }
}

int reporttime = 0;
struct pcap_stat laststat;

int lastbytes = 0;
short lasttcpclosed = 0;
short lastudpdeletecount = 0;
short lastipdeletecount = 0;
short lasticmpdeletecount = 0;
short lastfragdeletecount = 0;

void
check_all_timeouts (tvp)
struct timeval *tvp;
{
   extern int debug_interval;

   check_timeouts ( &tcp_timeout_list, tvp,  30);
   check_timeouts ( &tcp_display_list, tvp,  tcptimeout);
   check_timeouts ( &udp_display_list, tvp,  udptimeout);
   check_timeouts (  &ip_display_list, tvp,   iptimeout);
   check_timeouts (&icmp_display_list, tvp, icmptimeout);
   check_timeouts (&frag_display_list, tvp, fragtimeout);

   if (debug_interval) {
      struct WriteStruct thismss;
      extern struct manStatStruct *mss;
      extern char interfaceStatus ();
      extern int totalBytesRcv;

      if (tvp->tv_sec >= reporttime) {
         extern int tcpclosed, udpdeletecount, ipdeletecount;
         extern int icmpdeletecount, fragdeletecount;
         extern pcap_t *pd;
         struct pcap_stat stat;

         reporttime = tvp->tv_sec + debug_interval;

         bcopy ((char *) mss, (char *)&thismss, sizeof (thismss));
         bcopy ((char *) tvp, (char *)&thismss.ws_stat.now, sizeof (*tvp));

         thismss.ws_stat.reportInterval = (unsigned short) debug_interval;
         thismss.ws_stat.interfaceStatus = interfaceStatus ((char *) NULL);
         
         if (pcap_stats (pd, &stat) >= 0) {
            thismss.ws_stat.pktsRcvd  = stat.ps_recv - laststat.ps_recv;
            thismss.ws_stat.pktsDrop  = stat.ps_drop - laststat.ps_drop;
            thismss.ws_stat.bytesRcvd = totalBytesRcv - lastbytes;
            laststat = stat;
            lastbytes = totalBytesRcv;
         }

         thismss.ws_stat.actTCPcons   = (short) tcp_display_list.count;
         thismss.ws_stat.cloTCPcons   = (short)(tcpclosed - lasttcpclosed);
         lasttcpclosed = tcpclosed;
         thismss.ws_stat.actUDPcons   = (short) udp_display_list.count;
         thismss.ws_stat.cloUDPcons   = (short)(udpdeletecount - lastudpdeletecount);
         lastudpdeletecount = udpdeletecount;
         thismss.ws_stat.actIPcons    = (short) ip_display_list.count;
         thismss.ws_stat.cloIPcons    = (short)(ipdeletecount - lastipdeletecount);
         lastipdeletecount = ipdeletecount;
         thismss.ws_stat.actICMPcons  = (short) icmp_display_list.count;
         thismss.ws_stat.cloICMPcons  = (short)(icmpdeletecount - lasticmpdeletecount);
         lasticmpdeletecount = icmpdeletecount;
         thismss.ws_stat.actFRAGcons  = (short) frag_display_list.count;
         thismss.ws_stat.cloFRAGcons  = (short)(fragdeletecount - lastfragdeletecount);
         lastfragdeletecount = fragdeletecount;

         writeOutData (&thismss, tvp);
      }
   }
}

extern int totalPktsRcv;

static void
wrapup (pd)
pcap_t *pd;
{
   struct pcap_stat stat;
   extern char *rfile;

   if (rfile) {
      stat.ps_recv = totalPktsRcv;
      stat.ps_drop = 0;
   } else
      if (pcap_stats (pd, &stat) < 0)
         (void) fprintf (stderr, "pcap_stats: %s\n", pcap_geterr (pd));

   (void) fprintf (stderr, "\n%d packets recv'd by filter\n", stat.ps_recv);
   (void) fprintf (stderr, "%d packets dropped by kernel\n", stat.ps_drop);

   pcap_close (pd);
   close_clients ();
}

pcap_handler
lookup_pcap_callback (int type)
{
   pcap_handler retn = NULL;
   struct callback *callback;

   for (callback = callbacks; callback->function; ++callback)
      if (type == callback->type) {
         retn = callback->function;
         break;
      }

   return (retn);
}


void
cleanup ()
{
   struct timeval tvpbuf, *tvp = &tvpbuf;
   int cleanuptimeout = -1;

   struct WriteStruct thismss;
   extern struct manStatStruct *mss;
   extern char interfaceStatus ();
   extern int totalBytesRcv;
   extern int tcpclosed, udpdeletecount, ipdeletecount;
   extern int icmpdeletecount, fragdeletecount;
   extern pcap_t *pd;
   struct pcap_stat stat;

   gettimeofday (tvp, NULL);

   check_timeouts (&tcp_display_list,  tvp, cleanuptimeout);
   check_timeouts (&tcp_timeout_list,  tvp, cleanuptimeout);
   check_timeouts (&udp_display_list,  tvp, cleanuptimeout);
   check_timeouts (&ip_display_list,   tvp, cleanuptimeout);
   check_timeouts (&icmp_display_list, tvp, cleanuptimeout);
   check_timeouts (&frag_display_list, tvp, cleanuptimeout);

   bcopy ((char *) mss, (char *)&thismss, sizeof (thismss));
   bcopy ((char *) tvp, (char *)&thismss.ws_stat.now, sizeof (*tvp));

   thismss.status &= ~(INIT|STATUS);
   thismss.status |= CLOSE;
   thismss.ws_stat.reportInterval = 0;
   thismss.ws_stat.interfaceStatus = interfaceStatus ((char *) NULL);

   if (readingoffline) {
      stat.ps_recv = totalPktsRcv;
      stat.ps_drop = 0;
   } else
      pcap_stats (pd, &stat);

   thismss.ws_stat.pktsRcvd = stat.ps_recv - laststat.ps_recv;
   thismss.ws_stat.pktsDrop = stat.ps_drop - laststat.ps_drop;
   laststat = stat;
   lastbytes = totalBytesRcv;

   thismss.ws_stat.actTCPcons   = (short) tcp_display_list.count;
   thismss.ws_stat.cloTCPcons   = (short) tcpclosed;
   thismss.ws_stat.actUDPcons   = (short) udp_display_list.count;
   thismss.ws_stat.cloUDPcons   = (short) udpdeletecount;
   thismss.ws_stat.actIPcons    = (short) ip_display_list.count;
   thismss.ws_stat.cloIPcons    = (short) ipdeletecount;
   thismss.ws_stat.actICMPcons  = (short) icmp_display_list.count;
   thismss.ws_stat.cloICMPcons  = (short) icmpdeletecount;
   thismss.ws_stat.actFRAGcons  = (short) frag_display_list.count;
   thismss.ws_stat.cloFRAGcons  = (short) fragdeletecount;

   writeOutData (&thismss, tvp);
   
   if (pd != NULL) wrapup (pd);
   exit (0);
}

void
usr1sig ()
{
   debugflag = (debugflag++ > 30) ? 30 : debugflag;
   fprintf (stderr, "argus: debug enabled level %d\n", debugflag);
}

void
usr2sig ()
{
   debugflag = 0;
   fprintf (stderr, "argus: debug disabled\n");
}

void
usage(progname)
char *progname;
{
   fprintf (stderr, "Version %d.%d\n", VERSION_MAJOR, VERSION_MINOR);
   fprintf (stderr, "usage: %s [-bChOp][-d detail-interval]", progname);
   fprintf (stderr, "[-D debug-level][-r tcpdump-file]\n");
   fprintf (stderr, "           [-w argus-file]");
   fprintf (stderr, "[-U udp-timeout][-T tcp-timeout][-I ip-timeout]\n");
   fprintf (stderr, "           [-P port] [-i interface] expression\n");
   fprintf (stderr, "options: b - print filter definition\n");
   fprintf (stderr, "         d - log detailed events on sec interval\n");
   fprintf (stderr, "         C - report all ICMP packets\n");
   fprintf (stderr, "         D - set debug level\n");
   fprintf (stderr, "         h - print help\n");
   fprintf (stderr, "         i - specify network interface\n");
   fprintf (stderr, "         I - set IP transaction timer\n");
   fprintf (stderr, "         O - turn off filter optimizer\n");
   fprintf (stderr, "         p - don't go into promiscuous mode\n");
   fprintf (stderr, "         P - specify tcp remote access port\n");
   fprintf (stderr, "         r - read from tcpdump(1) packet file\n");
   fprintf (stderr, "         T - set TCP transaction timer\n");
   fprintf (stderr, "         U - set UDP transaction timer\n");
   fprintf (stderr, "         w - write output to logfile, or stdout '-'\n");
   exit (-1);
}

struct HASH_TABLE_HEADER *
find_hash_object (table, hash, size, buffer)
struct HASH_TABLE *table;
unsigned short hash;
int size;
unsigned char *buffer;
{
   struct HASH_TABLE_HEADER *retn = NULL;
   struct HASH_TABLE_HEADER *entry = NULL, *start = NULL;

   if (entry = table->hash_array[hash % table->size]) {
      start = entry;
      do {
         if (entry && entry->tha.buffer) {
            if ((entry->tha.size == size) &&
               (!(bcmp (buffer, entry->tha.buffer, size)))) {
               retn = entry;
               break;
            } else
               entry = entry->nxt;
         }
      } while (entry != start);
   }

   return (retn);
}


struct HASH_TABLE_HEADER *
add_hash_table_entry (table, tha, ptr)
struct HASH_TABLE *table;
struct THA_OBJECT *tha;
struct OBJECT *ptr;
{
   struct HASH_TABLE_HEADER *retn = NULL, *start = NULL;
   unsigned char *buffer;
   unsigned short hash = 0;
   int i, len, size = 0;

   if ((size = tha->size) && (buffer = tha->buffer)) {
      for (i = 0, len = (size / sizeof (hash)); i < len; i++)
         hash += ((unsigned short *) buffer)[i];

      if (!(retn = find_hash_object (table, hash, size, buffer))) {
         if (retn = (struct HASH_TABLE_HEADER *) calloc (1, sizeof (*retn))) {
            retn->object = ptr;
            retn->hash = hash;
            retn->tha.size = tha->size;
            if (buffer = malloc (tha->size)) {
               bcopy (tha->buffer, buffer, tha->size);
               retn->tha.buffer = buffer;
            }
   
            if (start = table->hash_array[hash % table->size]) {
               retn->nxt = start;
               retn->prv = start->prv;
               retn->prv->nxt = retn;
               retn->nxt->prv = retn;
            } else 
               retn->prv = retn->nxt = retn;

            table->hash_array[hash % table->size] = retn;

         } else perror ("add_hash_table_entry: calloc");
      } else aerror ("add_hash_table_entry: already in queue", 0L);
   }

   return (retn);
}


struct OBJECT *
find_hash_entry (table, tha)
struct HASH_TABLE *table;
struct THA_OBJECT *tha;
{
   struct OBJECT *retn = NULL;
   struct HASH_TABLE_HEADER *entry, *start;
   unsigned char *buffer;
   unsigned short hash = 0;
   int i, len, size;

   if ((size = tha->size) && (buffer = tha->buffer)) {
      for (i = 0, len = (size / sizeof (hash)); i < len; i++)
         hash += ((unsigned short *) buffer)[i];
   
      if (entry = find_hash_object (table, hash, size, buffer))
         retn = entry->object;
   }

   return (retn);
}


void
remove_hash_table_entry (table, tha)
struct HASH_TABLE *table;
struct THA_OBJECT *tha;
{
   struct HASH_TABLE_HEADER *retn = NULL;
   struct HASH_TABLE_HEADER *entry;
   unsigned char *buffer;
   unsigned short hash = 0;
   int i, len, size;

   if ((size = tha->size) && (buffer = tha->buffer)) {
      for (i = 0, len = (size / sizeof (hash)); i < len; i++)
         hash += ((unsigned short *) buffer)[i];

      if (retn = find_hash_object (table, hash, size, buffer)) {
         retn->prv->nxt = retn->nxt;
         retn->nxt->prv = retn->prv;

         if (retn == table->hash_array[hash % table->size]) {
            if (retn == retn->nxt)
               table->hash_array[hash % table->size] = NULL;
            else
               table->hash_array[hash % table->size] = retn->nxt;
         }
   
         if (retn->tha.buffer)
            free (retn->tha.buffer);
         free (retn);
      }
   }
}


add_to_queue (queue, ptr)
struct QUEUE *queue;
struct QUEUE_HEADER *ptr;
{
   int retn = 0;
   struct QUEUE_HEADER *head;

   if (queue && ptr) {
      if (!(ptr->queue)) {
         ptr->queue = queue;
         if (head = queue->start) {
            if (head->prv && head->nxt) {
               ptr->nxt = head;
               ptr->prv = head->prv;
               ptr->nxt->prv = ptr;
               ptr->prv->nxt = ptr;
            } else aerror ("add_to_queue: head pointers bad", 0L);
         } else {
            ptr->prv = ptr;
            ptr->nxt = ptr;
         }
         queue->start = ptr;
         queue->count++;
         retn = 1;
      } else
         aerror ("add_to_queue: ptr->queue not NULL", 0L);
   } else
      aerror ("add_to_queue: bad parameters", 0L);

   return (retn);
}

void
remove_from_queue (queue, ptr)
struct QUEUE *queue;
struct QUEUE_HEADER *ptr;
{
   struct QUEUE *thisqueue;

   if (queue && ptr) {
      if (ptr->queue && (ptr->queue == queue)) {
         if (--queue->count > 0) {
            if (ptr->prv)
               ptr->prv->nxt = ptr->nxt; 
            else
               aerror ("remove_from_queue: ptr->prv is null", 0L);

            if (ptr->nxt)
               ptr->nxt->prv = ptr->prv; 
            else
               aerror ("remove_from_queue: ptr->nxt is null", 0L);
   
            if (ptr == queue->start)
                  queue->start = ptr->nxt;
         } else
            queue->start = NULL;
      } else
         if (ptr->queue)
            aerror ("remove_from_queue: remove from wrong queue", 0L);
         else
            aerror ("remove_from_queue: object not in queue", 0L);

      ptr->queue = NULL;
      ptr->prv = NULL;
      ptr->nxt = NULL;

   } else
      aerror ("remove_from_queue: bad parameters", 0L);
}


void
update_queue_status (ptr)
struct QUEUE_HEADER *ptr;
{
   struct QUEUE *queue;

   if (ptr) { 
      queue = ptr->queue;
      if (queue) {
         if (ptr != queue->start) {
            remove_from_queue (queue, ptr);
            if (!(add_to_queue (queue, ptr)))
               aerror ("update_queue_status: add_to_queue error", NULL);
         }
      } else
         aerror ("update_queue_status: queue is NULL", 0L);
   } else
      aerror ("update_queue_status: ptr is NULL", 0L);
}

char *
copy_argv (argv)
char **argv;
{
   char **p;
   int len = 0;
   char *buf = NULL, *src, *dst;

   p = argv;
   if (*p == 0) return 0;

   while (*p) len += (int) strlen (*p++) + 1;

   if (buf = calloc (1, len)) {
      p = argv;
      dst = buf;
      while ((src = *p++) != NULL) {
         while ((*dst++ = *src++) != '\0') ;
         dst[-1] = ' ';
      }

      dst[-1] = '\0';
   }
   return buf;
}

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

char *
read_infile (char *fname)
{
   int fd;
   char *p;
   struct stat buf;
 
   if ((fd = open(fname, O_RDONLY)) >= 0) {
      if (fstat(fd, &buf) < 0)
         aerror ("can't state '%s'", fname);
 
      if (p = calloc (1, (u_int) buf.st_size))
         if (read (fd, p, (unsigned int)buf.st_size) != buf.st_size)
            aerror ("problem reading '%s'", fname);
 
   } else
      aerror ("can't open '%s'", fname);

   return (p);
}

unsigned short udpPorts [128];
unsigned short *udpServicePorts = NULL;

#define MAXSTRLEN  1024

void
read_udp_services (file)
char *file;
{
   FILE *fd;
   int port, i = 0;
   char *ptr = NULL, buffer[MAXSTRLEN];

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         bzero ((char *) udpPorts, sizeof (udpPorts));
         while ((i < 128) && fgets (buffer, MAXSTRLEN, fd))
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!'))
               if ((ptr = strtok (buffer, " \t")) && (*ptr != '\n'))
                  if ((ptr = strtok (NULL, " \t")) && (*ptr != '\n'))
                     if (strstr (ptr, "udp"))
                        if ((sscanf (ptr, "%d", &port)) == 1)
                           udpPorts[i++] = (unsigned short) port;
         if (i)
            udpServicePorts = udpPorts;

         fclose (fd);
      }
   }
}


char *interfaceName = NULL;

#include <sys/ioctl.h>
#ifdef HAVE_SOLARIS
#include <sys/sockio.h>
#endif

 
int
argus_lookupnet(device, netp, maskp, errbuf)
char *device;
arg_uint32 *netp, *maskp;
char *errbuf;
{
   int fd, retn = 0;
   struct sockaddr_in *sin;
   struct ifreq ifr;
 
   if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) >= 0) {
      strncpy (ifr.ifr_name, device, sizeof(ifr.ifr_name));
      if (ioctl(fd, SIOCGIFADDR, (char *)&ifr) >= 0) {
         sin = (struct sockaddr_in *) &ifr.ifr_addr;
         *netp = sin->sin_addr.s_addr;
         if (ioctl(fd, SIOCGIFNETMASK, (char *)&ifr) >= 0) {
            if (!(*maskp = sin->sin_addr.s_addr)) {
               if (IN_CLASSA(*netp)) *maskp = IN_CLASSA_NET;
               else if (IN_CLASSB(*netp)) *maskp = IN_CLASSB_NET;
               else if (IN_CLASSC(*netp)) *maskp = IN_CLASSC_NET;
            }
         } else {
            perror ("ioctl: SIOCGIFNETMASK");
            retn = -1;
         }
      } else {
         perror ("ioctl: SIOCGIFADDR");
         retn = -1;
      }
      (void) close(fd);
   } else {
      perror ("socket");
      retn = -1;
   }
   return (retn);
}


char
interfaceStatus (device)
char *device;
{
   int fd;
   char retn = '\0';
   struct ifconf ifc;
   struct ifreq ibuf[16], ifr;

   if (device) interfaceName = device;

   if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) >= 0) {
      if (interfaceName) {
         strncpy(ifr.ifr_name, interfaceName, sizeof(ifr.ifr_name));
         if (ioctl(fd, SIOCGIFFLAGS, (char *)&ifr) >= 0)
            retn = (ifr.ifr_flags & IFF_UP) ? 1 : 0;
         (void) close (fd);
      }
   }
   return (retn);
}

void
updateInitWs (ws, tvp)
struct WriteStruct *ws;
struct argtimeval *tvp;
{
   ws->ws_init.now.tv_sec  = tvp->tv_sec;
   ws->ws_init.now.tv_usec = tvp->tv_usec;
}


#include <varargs.h>

void
aerror (fmt, va_alist)
char *fmt;
va_dcl
{
   va_list ap;

   (void) fprintf (stderr, "%s: ", program_name);
   va_start (ap);
   (void) vfprintf (stderr, fmt, ap);
   va_end (ap);
   if (*fmt) {
      fmt += (int) strlen (fmt);
      if (fmt[-1] != '\n')
         (void) fputc ('\n', stderr);
   }
   exit (1);
}

