
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */


/*
 * cons_frag.c - handle fragmented packets and define
 *               and routines related to fragment event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */


#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>
#include <pcap.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <cons_frag.h>

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;
extern int ip_error;

extern unsigned char ip_ttl;
extern unsigned char ip_tos;

static struct FRAG_OBJECT *new_frag_hash_entry ();
static void remove_frag_hash_entry ();
void delete_frag ();

void create_frag_tha ();
void adjust_group_port ();
void update_frag_record ();


void
cons_frag_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &frag_display_list, sizeof (frag_display_list));
   bzero ((char *) &frag_hash_table, sizeof (frag_hash_table));

   if (ptr = (struct HASH_TABLE_HEADER **) calloc (USEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      frag_hash_table.hash_array = ptr;
      frag_hash_table.size = USEQ_HASHSIZE;
   } else
      perror ("cons_frag_init: calloc");

   fragtimeout = fragtimeout ? fragtimeout : FRAGTIMEOUT; 
   frag_display_list.timerRoutine = check_frag_timeouts;
   frag_display_list.logRoutine   = log_frag_connection;
}


struct FRAG_OBJECT *
cons_frag (ep, ip, length, tvp)
struct ether_header *ep;
struct ip *ip;
int length;
struct argtimeval *tvp;
{
   struct FRAG_OBJECT *fh;
   struct tha buffer;
   struct THA_OBJECT thabuf, *tha = &thabuf;
   int rev;

   tha->buffer = (u_char *) &buffer;
   create_frag_tha (tha, ip);

   if (fh = (struct FRAG_OBJECT *) find_hash_entry (&frag_hash_table, tha)) {
      update_frag_record (fh, ep, ip, length, tvp, rev);
   } else
      fh = new_frag_hash_entry (ip, ep, tvp, tha, length);

   return (fh);
}


void
create_frag_tha (tha, ip)
struct THA_OBJECT *tha;
struct ip *ip;
{
   struct tha thabuf;

   thabuf.src = ip->ip_src, thabuf.dst = ip->ip_dst;
   ((unsigned short *)&thabuf.port)[0] = ip->ip_p;
   ((unsigned short *)&thabuf.port)[1] = ip->ip_id;

   if (tha->buffer)
      bcopy ((u_char *)&thabuf, tha->buffer, sizeof (struct tha));

   tha->size = sizeof (struct tha);
}

void
update_frag_record (fh, ep, ip, length, tvp)
struct FRAG_OBJECT *fh;
struct ether_header *ep;
struct ip *ip;
int length;
struct argtimeval *tvp;
{
   struct fragcb *fragCb;

   if (fh) {
      fragCb = &fh->frag_cb;
      fragCb->lasttime = *tvp; fh->qhdr.last_time = *tvp;

      if (!(((ip->ip_off & 0x1fff) == 0) && (ip->ip_off & IP_MF))) {
         fragCb->count++;
         fragCb->bytes += length;
      }

      fragCb->ttl = ip_ttl;
      fragCb->tos = ip_tos;
   
      if (ep && !(fragCb->status & MULTIADDR)) {
         if (bcmp ((char *)&ep->ether_shost,
                       (char *)&fh->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_dhost,
                       (char *)&fh->link.phys.etherdst, 6)) {
               fragCb->status |= MULTIADDR;
               bcopy ((char *)&ep->ether_shost,
                              (char *)&fh->link.phys.ethersrc, 6);
               bcopy ((char *)&ep->ether_dhost,
                              (char *)&fh->link.phys.etherdst, 6);
         }
      }

      if (ip_options) { 
         if (ip_rr)   fragCb->status |= RECORDROUTE; 
         if (ip_ts)   fragCb->status |= TIMESTAMP;
         if (ip_sec)  fragCb->status |= SECURITY; 
         if (ip_lsrr) fragCb->status |= LSRCROUTE; 
         if (ip_ssrr) fragCb->status |= SSRCROUTE; 
         if (ip_satid) fragCb->status |= SATNETID; 
         if (ip_error) fragCb->status |= IPOPTIONMASK; 
      }
   
      if ((ip->ip_p == IPPROTO_TCP) && (ip->ip_off == 1))
         fragCb->status |= TCP_FRAG_OFFSET_PROBLEM;
   
      fragCb->currlen += length;
      if (length > fragCb->maxfraglen)
         fragCb->maxfraglen = length;

      if (!(ip->ip_off & IP_MF)) {
         fragCb->totlen = ((ip->ip_off & 0x1fff) << 3) +
                           (ip->ip_len - (ip->ip_hl << 2));
      }
      if (fragCb->totlen)
         if (fragCb->totlen == fragCb->currlen)
            fragCb->status |= NORMAL_CLOSE;
   
      update_queue_status ((struct QUEUE_HEADER *) &fh->qhdr);
      fragCb->status |= MODIFIED;
   
      if (((tvp->tv_sec - fragCb->startime.tv_sec) >= fragtimeout) ||
                             (fragCb->status & TCP_FRAG_OFFSET_PROBLEM))
         log_frag_connection (fh, tvp, CON_ESTABLISHED);
   }
}
 

int fragdeletecount = 0;

void
delete_frag (fh)
struct FRAG_OBJECT *fh;
{
   fragdeletecount++;
   remove_hash_table_entry (&frag_hash_table, &fh->tha);
   free (fh->tha.buffer);
   free (fh);
}


static struct FRAG_OBJECT *
new_frag_hash_entry (ip, ep, tvp, tha, len)
struct ip *ip;
struct ether_header *ep;
struct argtimeval *tvp;
struct THA_OBJECT *tha;
int len;
{
   struct FRAG_OBJECT *ptr = NULL;
   struct fragcb *fragCb;

   if (ptr = (struct FRAG_OBJECT *) calloc (1, sizeof (*ptr))) {
      fragCb = &ptr->frag_cb;
      fragCb->status = FRAGMENTS | FRAG_INIT | MODIFIED;
      if (ip) fragCb->status |= IPPROTO;
      if (dflag) fragCb->status |= DETAIL;
      fragCb->startime = *tvp;
      fragCb->lasttime = *tvp;
      if (ep) {
         bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
         bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
      }

      ptr->qhdr.last_time = *tvp;

      if (ptr->tha.buffer = (u_char *) malloc (tha->size)) {
         bcopy (tha->buffer, (char *) ptr->tha.buffer, tha->size);
         ptr->tha.size = tha->size;
      }

      if (ip_options) { 
         if (ip_rr)   fragCb->status |= RECORDROUTE; 
         if (ip_ts)   fragCb->status |= TIMESTAMP;
         if (ip_sec)  fragCb->status |= SECURITY; 
         if (ip_lsrr) fragCb->status |= LSRCROUTE; 
         if (ip_ssrr) fragCb->status |= SSRCROUTE; 
         if (ip_satid) fragCb->status |= SATNETID; 
         if (ip_error) fragCb->status |= IPOPTIONMASK; 
      }

      if (!(((ip->ip_off & 0x1fff) == 0) && (ip->ip_off & IP_MF))) {
         fragCb->count++;
         fragCb->bytes += len;
      }

      fragCb->ttl = ip_ttl;
      fragCb->ip_id = ip->ip_id;
      fragCb->currlen = fragCb->maxfraglen = len;

      if (add_to_queue (&frag_display_list, (struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&frag_hash_table, tha, ptr);
         if (dflag) {
            log_frag_connection (ptr, tvp, FRAG_INIT);
            fragCb->status &= ~LOGGED;
         }
      } else {
         delete_frag (ptr);
         ptr = NULL;
      }
   }

   return (ptr);
}


#include <sys/stat.h>
#include <sys/errno.h>

void
log_frag_connection (ptr, tvp, state)
struct FRAG_OBJECT *ptr;
struct argtimeval *tvp;
int state;
{
   unsigned int status = 0;
   struct tha *tha;
   struct WriteStruct output;

   if (clienttags || wflag) {
      if (ptr->frag_cb.count && (ptr->frag_cb.status & MODIFIED)) {
         bzero ((char *) &output, sizeof (struct WriteStruct));
         ptr->frag_cb.status |= state | FRAG_ONLY;
         status = ptr->frag_cb.status;
         ptr->frag_cb.status &= ~(MODIFIED | MULTIADDR);
         output.ws_ip.startime =  ptr->frag_cb.startime;


         ((u_short *)&output.ws_ip.lasttime.tv_sec)[1] =
                                (u_short)(ptr->frag_cb.lasttime.tv_sec -
                                          ptr->frag_cb.startime.tv_sec);
         output.ws_ip.lasttime.tv_usec =  ptr->frag_cb.lasttime.tv_usec;
         ((u_char *)&output.ws_ip.lasttime.tv_sec)[0] = (u_char) ptr->frag_cb.ttl;

         output.status   =  status;
         bcopy ((char *)&ptr->link.phys.ethersrc,
                             (char *)&output.ws_ip_phys.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst,
                             (char *)&output.ws_ip_phys.etherdst, 6);

         tha = (struct tha *) ptr->tha.buffer;
         output.ws_ip.src.s_addr = tha->src.s_addr;
         output.ws_ip.dst.s_addr = tha->dst.s_addr;

         output.ws_ip_frag.fragnum    = ptr->frag_cb.count;
         output.ws_ip_frag.frag_id    = ptr->frag_cb.ip_id;
         output.ws_ip_frag.totlen     = ptr->frag_cb.totlen;
         output.ws_ip_frag.currlen    = ptr->frag_cb.currlen;
         output.ws_ip_frag.maxfraglen = ptr->frag_cb.maxfraglen;

         if (ptr->frag_cb.status & state) {
            writeOutData (&output, tvp);
            ptr->frag_cb.status |= LOGGED;
         }

         if (!(dflag && (state & FRAG_INIT)) &&
                               (ptr->frag_cb.status & LOGGED)) {
            ptr->frag_cb.lastcount = ptr->frag_cb.count;
            ptr->frag_cb.lastbytes = ptr->frag_cb.bytes;
            ptr->frag_cb.startime = *tvp;
            ptr->frag_cb.lasttime = *tvp;
         }
      }
   }

   ptr->qhdr.logtime = *tvp;
}
   

void
check_frag_timeouts (ptr, tvp)
struct FRAG_OBJECT *ptr;
struct argtimeval *tvp;
{
   int count;

   if (!(ptr->frag_cb.status & LOGGED)) {
      count = ptr->frag_cb.count;
   } else {
      count = (ptr->frag_cb.count - ptr->frag_cb.lastcount);
   }  
 
   if (count) {
      ptr->frag_cb.status |= TIMED_OUT;
      log_frag_connection (ptr, tvp, TIMED_OUT);
   }
   remove_from_queue (&frag_display_list,
                       (struct QUEUE_HEADER *) &ptr->qhdr);
   delete_frag (ptr);
}


#include <cons_ip.h>

#define PROTOS (UDPPROTO | TCPPROTO | ICMPPROTO)

adjustParentCounts (frag)
struct FRAG_OBJECT *frag;
{
   struct IP_OBJECT *obj = (struct IP_OBJECT *)frag->obj;

   switch (obj->ip_cb.status & PROTOS) {
      case  TCPPROTO: break;
      case ICMPPROTO: break;
      case  UDPPROTO:
      default: {
         struct tha *fragtha = (struct tha *) frag->tha.buffer;
         struct tha *objtha  = (struct tha *) obj->tha.buffer;

         if (objtha->src.s_addr == fragtha->src.s_addr) {
            obj->ip_cb.src.count += frag->frag_cb.count;
            obj->ip_cb.src.bytes += frag->frag_cb.bytes;
         } else {
            obj->ip_cb.dst.count += frag->frag_cb.count;
            obj->ip_cb.dst.bytes += frag->frag_cb.bytes;
         }
         break;
      }
   }
}
