
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_icmp.c - supports ICMP tracking 
 *
 * Used to parse received icmp packets.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 */

#define CONS_ICMP

#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>
#include <pcap.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <netinet/tcp.h>
#include <netinet/ip_icmp.h>
#include <cons_icmp.h>

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;
extern int ip_error;

extern unsigned char ip_ttl;
extern unsigned char ip_tos;

void update_icmp_record ();

void
cons_icmp_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &icmp_display_list, sizeof (icmp_display_list));
   bzero ((char *) &icmp_hash_table, sizeof (icmp_hash_table));
   if (ptr = (struct HASH_TABLE_HEADER **) calloc (ICMPSEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      icmp_hash_table.hash_array = ptr;
      icmp_hash_table.size = ICMPSEQ_HASHSIZE;
   } else
      perror ("cons_icmp_init: calloc");

   icmp_display_list.timerRoutine = check_icmp_timeouts;
   icmp_display_list.logRoutine =   log_icmp_connection;
 
   icmptimeout = icmptimeout ? icmptimeout : ICMPTIMEOUT;
}


create_icmp_tha (tha, ip, icmp, len, rev)
struct THA_OBJECT *tha;
struct ip *ip;
struct icmp *icmp;
int len, *rev;
{
   struct tha thabuf;
   int retn, hlen, length, optionlen;
   arg_uint32 src, dst;
   unsigned char *cp;

   retn = IPPROTO_ICMP;

   src = ip->ip_src.s_addr;
   dst = ip->ip_dst.s_addr;

   if (src < dst) {
      thabuf.src.s_addr = dst, thabuf.dst.s_addr = src;
      *rev = 1;
   } else {
      thabuf.src.s_addr = src, thabuf.dst.s_addr = dst;
      *rev = 0;
   }

   if (len >= sizeof (struct icmp)) {
      switch (icmp->icmp_type) {
         case ICMP_ECHOREPLY:    thabuf.port = ICMP_ECHO;    break;
         case ICMP_TSTAMPREPLY:  thabuf.port = ICMP_TSTAMP;  break;
         case ICMP_IREQREPLY:    thabuf.port = ICMP_IREQ;    break;
         case ICMP_MASKREPLY:    thabuf.port = ICMP_MASKREQ; break;
                     default:    thabuf.port = icmp->icmp_type; break;
      }
   }

   if (tha->buffer)
      bcopy ((u_char *)&thabuf, tha->buffer, sizeof (struct tha));

   tha->size = sizeof (struct tha);

   return (retn);
}


extern struct HASH_TABLE tcp_hash_table;
extern struct HASH_TABLE udp_hash_table;
extern struct HASH_TABLE  ip_hash_table;
static struct ICMP_OBJECT *new_icmp_hash_entry ();
static void delete_icmp ();

struct OBJECT *
cons_icmp (ep, icmp, len, ip, tvp)
struct ether_header *ep;
struct icmp *icmp;
int len;
struct ip *ip;
struct argtimeval *tvp;
{
   struct OBJECT *obj = NULL;
   int rev = 0, proto;
   struct THA_OBJECT thabuf, *tha = &thabuf;
   struct tha buffer;
   
   if (len && icmp) {
      tha->buffer = (u_char *) &buffer;
      create_icmp_tha (tha, ip, icmp, len, &rev);

      if (Cflag) {
         struct ICMP_OBJECT *ih = NULL;

         if (ih = new_icmp_hash_entry (ep, ip, icmp,
                                           rev, tvp, tha, len)) {
            log_icmp_connection (ih, tvp, IP_INIT);
            remove_from_queue (&icmp_display_list,
                              (struct QUEUE_HEADER *) &ih->qhdr);
            delete_icmp (ih);
         }
      } else {
         if (obj = (struct OBJECT *) find_hash_entry(&icmp_hash_table, tha))
            update_icmp_record (obj, ep, icmp, len, tvp, rev);
         else
            obj = (struct OBJECT *) new_icmp_hash_entry (ep, ip, icmp,
                                           rev, tvp, tha, len);
      }
   }

   return (obj);
}

void
update_icmp_record (ih, ep, icmp, len, tvp, rev)
struct ICMP_OBJECT *ih;
struct ether_header *ep;
struct icmp *icmp;
int len;
struct argtimeval *tvp;
int rev;
{
   struct icmpcb *icmpCb;

   icmpCb = &ih->icmp_cb;
   icmpCb->lasttime = *tvp; ih->qhdr.last_time = *tvp;

   if (icmpCb->rev == rev) {
      icmpCb->src.count++;
      icmpCb->src.ttl = ip_ttl;
      icmpCb->src.tos = ip_tos;
   } else {
      icmpCb->dst.count++;
      icmpCb->dst.ttl = ip_ttl;
      icmpCb->dst.tos = ip_tos;
   }

   if (ep && !(icmpCb->status & MULTIADDR)) {
      if (rev) {
         if (bcmp ((char *)&ep->ether_dhost,
                    (char *)&ih->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_shost,
                    (char *)&ih->link.phys.etherdst, 6)) {
            icmpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&ih->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost,
                           (char *)&ih->link.phys.etherdst, 6);
         }
      } else
         if (bcmp ((char *)&ep->ether_shost,
                    (char *)&ih->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_dhost,
                    (char *)&ih->link.phys.etherdst, 6)) {
            icmpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_shost,
                           (char *)&ih->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&ih->link.phys.etherdst, 6);
         }
   }

   if (ip_options) {
      if (ip_rr)   icmpCb->status |= RECORDROUTE;
      if (ip_ts)   icmpCb->status |= TIMESTAMP;
      if (ip_sec)  icmpCb->status |= SECURITY;
      if (ip_lsrr) icmpCb->status |= LSRCROUTE;
      if (ip_ssrr) icmpCb->status |= SSRCROUTE;
      if (ip_satid) icmpCb->status |= SATNETID;
      if (ip_error) icmpCb->status |= IPOPTIONMASK;
   }

   if (icmpCb->src.count && icmpCb->dst.count)
      icmpCb->status |= CON_ESTABLISHED;

   update_queue_status ((struct QUEUE_HEADER *) &ih->qhdr);
   icmpCb->status |= MODIFIED;

   if (((tvp->tv_sec - icmpCb->startime.tv_sec) > icmptimeout) ||
         (((icmpCb->src.count == 1) && (icmpCb->dst.count == 1)) &&
              !(icmpCb->status & LOGGED)))
      log_icmp_connection (ih, tvp, CON_ESTABLISHED);
}

int icmpdeletecount = 0;

static void
delete_icmp (ih)
struct ICMP_OBJECT *ih;
{
   icmpdeletecount++;
   remove_hash_table_entry (&icmp_hash_table, &ih->tha);
   if (ih->tha.buffer) free (ih->tha.buffer);
   free (ih);
}


static struct ICMP_OBJECT *
new_icmp_hash_entry (ep, ip, icmp, rev, tvp, tha, len)
struct ether_header *ep;
struct ip *ip;
struct icmp *icmp;
int rev;
struct argtimeval *tvp;
struct THA_OBJECT *tha;
int len;
{
   struct ICMP_OBJECT *ptr = NULL;
   struct icmpcb *icmpCb;
   struct ip *oip;
   struct udphdr *ouh;
   int hlen, length;

   if (ptr = (struct ICMP_OBJECT *) calloc (1, sizeof (*ptr))) {
      icmpCb = &ptr->icmp_cb;
      icmpCb->status = ICMPPROTO | IP_INIT | MODIFIED;
      if (ip) icmpCb->status |= IPPROTO;
      if (dflag) icmpCb->status |= DETAIL;
      icmpCb->startime = *tvp;
      icmpCb->lasttime = *tvp;
      if (icmpCb->rev = rev)
         icmpCb->status |= REVERSE;

      if (ep) {
         if (icmpCb->rev) {
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.etherdst, 6);
    
         } else {
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
         }
      }

      ptr->qhdr.last_time = *tvp;

      if (ptr->tha.buffer = (u_char *) malloc (tha->size)) {
         bcopy (tha->buffer, (char *) ptr->tha.buffer, tha->size);
         ptr->tha.size = tha->size;
      }

      if (ip_options) {
         if (ip_rr)   icmpCb->status |= RECORDROUTE;
         if (ip_ts)   icmpCb->status |= TIMESTAMP;
         if (ip_sec)  icmpCb->status |= SECURITY;
         if (ip_lsrr) icmpCb->status |= LSRCROUTE;
         if (ip_ssrr) icmpCb->status |= SSRCROUTE;
         if (ip_satid) icmpCb->status |= SATNETID;
         if (ip_error) icmpCb->status |= IPOPTIONMASK;
      }

      icmpCb->src.count++;
      icmpCb->src.ttl = ip_ttl;
      icmpCb->src.tos = ip_tos;

      if (icmp && len) {
         ptr->icmp_cb.icmp.type = icmp->icmp_type;
         ptr->icmp_cb.icmp.code = icmp->icmp_code;
         if (ICMP_INFOTYPE(icmp->icmp_type)) {
            switch (icmp->icmp_type) {
               case ICMP_ECHO:
               case ICMP_ECHOREPLY:
               case ICMP_IREQ:
               case ICMP_IREQREPLY:
               case ICMP_MASKREQ:
               case ICMP_TSTAMP:
               case ICMP_TSTAMPREPLY:
                  break;
         
               case ICMP_MASKREPLY:
                  bcopy ((char *) &icmp->icmp_mask,
                             (char *) &ptr->icmp_cb.icmp.srcaddr,
                             sizeof (icmp->icmp_mask));
                  break;
            }
         } else {
            switch (icmp->icmp_type) {
               case ICMP_UNREACH:
                  bcopy ((char *) &icmp->icmp_ip.ip_src,
                             (char *) &ptr->icmp_cb.icmp.srcaddr,
                             sizeof (arg_uint32));
                  bcopy ((char *) &icmp->icmp_ip.ip_dst,
                             (char *) &ptr->icmp_cb.icmp.dstaddr,
                             sizeof (arg_uint32));
                  switch (icmp->icmp_code) {
                     case ICMP_UNREACH_PROTOCOL:
                        ptr->icmp_cb.icmp.data =
                             (unsigned short) icmp->icmp_ip.ip_p;
                        break;
      
                     case ICMP_UNREACH_PORT:
                        oip = &icmp->icmp_ip;
                        hlen = oip->ip_hl << 2;
                        ouh = (struct udphdr *) (((u_char *) oip) + hlen);
                        ptr->icmp_cb.icmp.data = ntohs((unsigned short) ouh->uh_dport);
                        bcopy ((char *) &oip->ip_p,
                             (char *) &ptr->icmp_cb.icmp.gwaddr,
                             sizeof (oip->ip_p));
                        break;
                  } break;
   
               case ICMP_REDIRECT:
                  bcopy ((char *) &icmp->icmp_ip.ip_src,
                             (char *) &ptr->icmp_cb.icmp.srcaddr,
                             sizeof (ptr->icmp_cb.icmp.srcaddr));
                  bcopy ((char *) &icmp->icmp_ip.ip_dst,
                             (char *) &ptr->icmp_cb.icmp.dstaddr,
                             sizeof (ptr->icmp_cb.icmp.dstaddr));
                  bcopy ((char *) &icmp->icmp_gwaddr,
                             (char *) &ptr->icmp_cb.icmp.gwaddr,
                             sizeof (ptr->icmp_cb.icmp.gwaddr));
                  break;
            }
         }
      }

      if (add_to_queue(&icmp_display_list,(struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&icmp_hash_table, tha, ptr);
         if (dflag) {
            log_icmp_connection (ptr, tvp, IP_INIT);
            icmpCb->status &= ~LOGGED;
         }
      } else
         delete_icmp (ptr);
   }

   return (ptr);
}


void
log_icmp_connection (ptr, tvp, state)
struct ICMP_OBJECT *ptr;
struct argtimeval *tvp;
int state;
{
   unsigned int status = 0;
   int src_count, dst_count;
   struct tha *tha;
   struct ICMP_CON_OBJECT *src, *dst;
   struct WriteStruct output;

   if (clienttags || wflag) {
      src = (struct ICMP_CON_OBJECT *) &ptr->icmp_cb.src;
      dst = (struct ICMP_CON_OBJECT *) &ptr->icmp_cb.dst;

      if ((src->count || dst->count) && (ptr->icmp_cb.status & MODIFIED)) {
         bzero ((char *) &output, sizeof (struct WriteStruct));
         ptr->icmp_cb.status |= state;
         status = ptr->icmp_cb.status;
         ptr->icmp_cb.status &= ~(MODIFIED | MULTIADDR | FRAGMENTS);
         output.ws_ip.startime =  ptr->icmp_cb.startime;

         ((u_short *)&output.ws_ip.lasttime.tv_sec)[1] =
                                (u_short)(ptr->icmp_cb.lasttime.tv_sec -
                                          ptr->icmp_cb.startime.tv_sec);
         output.ws_ip.lasttime.tv_usec =  ptr->icmp_cb.lasttime.tv_usec;

         ((u_char *)&output.ws_ip.lasttime.tv_sec)[0] = (u_char) src->ttl;
         ((u_char *)&output.ws_ip.lasttime.tv_sec)[1] = (u_char) dst->ttl;

         output.status   =  status;
         bcopy ((char *)&ptr->link.phys.ethersrc,
                             (char *)&output.ws_ip_phys.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst,
                             (char *)&output.ws_ip_phys.etherdst, 6);

         tha = (struct tha *) ptr->tha.buffer;
         output.ws_ip.src.s_addr = tha->src.s_addr;
         output.ws_ip.dst.s_addr = tha->dst.s_addr;
         output.ws_ip.port       = tha->port;

         if (state == IP_INIT) {
            bcopy ((char *)&ptr->icmp_cb.icmp, (char *) &output.ws_ip_icmp,
                        sizeof (output.ws_ip_icmp));
            writeOutData (&output, tvp);

         } else {
            unsigned int icmpdata;

            output.ws_ip_udp.src_count = src->count - src->lastcount;
            output.ws_ip_udp.dst_count = dst->count - dst->lastcount;

            ((unsigned char *)&icmpdata) [0] = ptr->icmp_cb.icmp.type;
            ((unsigned char *)&icmpdata) [1] = ptr->icmp_cb.icmp.code;
            ((unsigned short *)&icmpdata)[1] = ptr->icmp_cb.icmp.data;

            output.ws_ip_udp.src_bytes = icmpdata;
            switch (ptr->icmp_cb.icmp.type) {
               case ICMP_UNREACH:
                  if (ptr->icmp_cb.icmp.code == ICMP_UNREACH_PORT)
                     icmpdata = ptr->icmp_cb.icmp.gwaddr.s_addr;
                  else
                     icmpdata = ptr->icmp_cb.icmp.dstaddr.s_addr;
                  break;
               case ICMP_MASKREPLY:
                  icmpdata = ptr->icmp_cb.icmp.srcaddr.s_addr;
                  break;
               case ICMP_REDIRECT:
                  icmpdata = ptr->icmp_cb.icmp.gwaddr.s_addr;
                  break;
               default:
                  icmpdata = 0;  
                  break; 
            }
            output.ws_ip_udp.dst_bytes = icmpdata;

            if (output.ws_ip_udp.src_count || output.ws_ip_udp.dst_count) {
               if (ptr->icmp_cb.status & state) {
                  writeOutData (&output);
                  ptr->icmp_cb.status |= LOGGED;
               }
            }
            if (!(dflag && (state & IP_INIT)) &&
                               (ptr->icmp_cb.status & LOGGED)) {
               src->lastcount = src->count;
               dst->lastcount = dst->count;
               ptr->icmp_cb.startime = *tvp;
               ptr->icmp_cb.lasttime = *tvp;
            }
         }
      }
   }

   ptr->qhdr.logtime = *tvp;
}


int icmp_closed = 0;

void
check_icmp_timeouts (ptr, tvp)
struct ICMP_OBJECT *ptr;
struct argtimeval *tvp;
{
   int srccounts, dstcounts;

   if (!(ptr->icmp_cb.status & LOGGED)) {
      srccounts = ptr->icmp_cb.src.count;
      dstcounts = ptr->icmp_cb.dst.count;
   } else {
      srccounts = (ptr->icmp_cb.src.count - ptr->icmp_cb.src.lastcount);
      dstcounts = (ptr->icmp_cb.dst.count - ptr->icmp_cb.dst.lastcount);
   }

   if (srccounts || dstcounts) {
      ptr->icmp_cb.status |= TIMED_OUT;
      log_icmp_connection (ptr, tvp, TIMED_OUT);
   }  
   remove_from_queue (&icmp_display_list,
                          (struct QUEUE_HEADER *) &ptr->qhdr);
   delete_icmp (ptr);
}
