
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_ip.c - routines related to ip event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */


#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>
#include <pcap.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <cons_ip.h>

static int ip_eol = 0;
static int ip_nop = 0;

int ip_options = 0;
int ip_ts = 0;
int ip_rr = 0;
int ip_sec = 0;
int ip_lsrr = 0;
int ip_ssrr = 0;
int ip_satid = 0;
int ip_error = 0;

unsigned char ip_ttl = 0;
unsigned char ip_tos = 0;

static struct IP_OBJECT *new_ip_hash_entry ();
static void delete_ip ();
static void remove_ip_hash_entry ();
static struct IP_OBJECT *cons_ip ();
static int parse_options ();

void update_ip_record ();
void create_ip_tha ();

void
cons_ip_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &ip_display_list, sizeof (ip_display_list));
   bzero ((char *) &ip_hash_table, sizeof (ip_hash_table));

   if (ptr = (struct HASH_TABLE_HEADER **) calloc (IPSEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      ip_hash_table.hash_array = ptr;
      ip_hash_table.size = IPSEQ_HASHSIZE;
   } else
      perror ("cons_ip_init: calloc");

   ip_display_list.timerRoutine = check_ip_timeouts;
   ip_display_list.logRoutine   = log_ip_connection;
   iptimeout = iptimeout ? iptimeout : IPTIMEOUT;
}


#include <cons_frag.h>

void argus_ip_handler (ep, ip, length, tvp)
struct ether_header *ep;
struct ip *ip;
int length;
struct argtimeval *tvp;
{
   struct IP_OBJECT *obj = NULL;
   struct FRAG_OBJECT *frag = NULL;
   int hlen, len, optionlen;
   unsigned char *cp;
   u_char buf[1024];
   u_char *abuf = buf;
   extern int snaplen;
   arg_uint32 addr;

   if (length >= sizeof (struct ip)) {
      if ((long) ip & (sizeof (long) - 1)) {
         bcopy ((char *) ip, (char *) abuf, min (length, snaplen));
         ip = (struct ip *) abuf;
      }

      ip->ip_len = ntohs(ip->ip_len);
      ip->ip_id  = ntohs(ip->ip_id);
      ip->ip_off = ntohs(ip->ip_off);

      ip_ttl = ip->ip_ttl;
      ip_tos = ip->ip_tos;

      ip->ip_src.s_addr =  ntohl(ip->ip_src.s_addr);
      ip->ip_dst.s_addr =  ntohl(ip->ip_dst.s_addr);

      hlen = ip->ip_hl << 2;
      len = ip->ip_len - hlen;

      optionlen = (hlen - sizeof (struct ip));
      if (optionlen) {
         ip_options = 1;
         if (parse_options ((unsigned char *) (ip + 1), optionlen))
            return;
      } else {
         ip_options = ip_eol = ip_nop = ip_ts = ip_error = 0;
         ip_rr = ip_sec = ip_lsrr = ip_ssrr = ip_satid = 0;
      }
   
      if ((ip->ip_off & 0x1fff) == 0) {
         cp = (unsigned char *)ip + hlen;
         switch (ip->ip_p) {
            case IPPROTO_UDP:
               obj = (struct IP_OBJECT *)cons_udp (ep, cp, len, ip, tvp); break;
   
            case IPPROTO_ICMP:
               obj = (struct IP_OBJECT *)cons_icmp (ep, cp, len, ip, tvp); break;
   
            case IPPROTO_TCP:
               obj = (struct IP_OBJECT *)cons_tcp (ep, cp, len, ip, tvp); break;
   
            default:
               obj = (struct IP_OBJECT *)cons_ip (ep, len, ip, tvp); break;
         }
         if (ip->ip_off & IP_MF) {
            if (frag = cons_frag (ep, ip, len, tvp)) {
               frag->obj = (struct OBJECT *) obj;
               frag->frag_cb.status |= FRAG_OUT_OF_ORDER;
               obj->ip_cb.status |= FRAGMENTS;
            }
         }
      } else {
         if (frag = cons_frag (ep, ip, len, tvp)) {
            if (frag->frag_cb.status & NORMAL_CLOSE) {
               if (frag->obj)
                  adjustParentCounts (frag);
               remove_from_queue (&frag_display_list,
                          (struct QUEUE_HEADER *) &frag->qhdr);
               delete_frag (frag);
            }
         }
      }
   }
}



static int
parse_options (ptr, len)
unsigned char *ptr;
int len;
{
   int retn = 0, offset = 0;

   for (; len > 0; ptr += offset, len -= offset) {
      switch (*ptr) {
         case IPOPT_EOL:      ip_eol++; break;
         case IPOPT_NOP:      ip_nop++; break;
         case IPOPT_TS:       ip_ts++; break;
         case IPOPT_RR:       ip_rr++; break;
         case IPOPT_SECURITY: ip_sec++; break;
         case IPOPT_LSRR:     ip_lsrr++; break;
         case IPOPT_SSRR:     ip_ssrr++; break;
         case IPOPT_SATID:    ip_satid++; break;
         default:             ip_error++; break;
      }

      if ((*ptr == IPOPT_EOL) || (*ptr == IPOPT_NOP))
         offset = 1;
      else {
         offset = ptr[1];
         if (!(offset && (offset <= len))) {
            retn++; break;
         }
      }
   }

   return (retn);
}


static struct IP_OBJECT *
cons_ip (ep, length, ip, tvp)
struct ether_header *ep;
int length;
struct ip *ip;
struct argtimeval *tvp;
{
   struct IP_OBJECT *ih = NULL;
   int rev, status;
   struct tha buffer;
   struct THA_OBJECT thabuf, *tha = &thabuf;

   if (length && ip) {
      tha->buffer = (u_char *) &buffer;
      create_ip_tha (tha, ip, &rev);
 
      if (ih = (struct IP_OBJECT *) find_hash_entry (&ip_hash_table, tha))
         update_ip_record (ih, ep, length, tvp, rev);
      else
         ih = new_ip_hash_entry (ip, ep, rev, tvp, tha, length);
   }

   return (ih);
}

void
create_ip_tha (tha, ip, rev)
struct THA_OBJECT *tha;
struct ip *ip;
int *rev;
{
   struct tha thabuf;
   arg_uint32 src, dst;

   src = ip->ip_src.s_addr;
   dst = ip->ip_dst.s_addr;

   if (src < dst) {
      thabuf.src.s_addr = dst, thabuf.dst.s_addr = src;
      *rev = 1;
   } else {
      thabuf.src.s_addr = src, thabuf.dst.s_addr = dst;
      *rev = 0;
   }
   thabuf.port  = ip->ip_p;

   if (tha->buffer)
      bcopy ((u_char *)&thabuf, tha->buffer, sizeof (struct tha));

   tha->size = sizeof (struct tha);
}


void
update_ip_record (ih, ep, length, tvp, rev)
struct IP_OBJECT *ih;
struct ether_header *ep;
int length;
struct argtimeval *tvp;
int rev;
{
   struct ipcb *ipCb;

   ipCb = &ih->ip_cb;
   ipCb->lasttime = *tvp; ih->qhdr.last_time = *tvp;
   if (ipCb->rev == rev) {
      ipCb->src.count++;
      ipCb->src.bytes += length;
      ipCb->src.ttl = ip_ttl;
      ipCb->src.tos = ip_tos;
   } else {
      ipCb->dst.count++;
      ipCb->dst.bytes += length;
      ipCb->dst.ttl = ip_ttl;
      ipCb->dst.tos = ip_tos;
   }

   if (ep && !(ipCb->status & MULTIADDR)) {
      if (rev) {
         if (bcmp ((char *)&ep->ether_dhost,
                    (char *)&ih->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_shost,
                    (char *)&ih->link.phys.etherdst, 6)) {
            ipCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&ih->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost,
                           (char *)&ih->link.phys.etherdst, 6);
         }
      } else
         if (bcmp ((char *)&ep->ether_shost,
                    (char *)&ih->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_dhost,
                    (char *)&ih->link.phys.etherdst, 6)) {
            ipCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_shost,
                           (char *)&ih->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&ih->link.phys.etherdst, 6);
         }
   }

   if ((ipCb->src.count + ipCb->dst.count) == 2)
      ipCb->status |= CON_ESTABLISHED;
   update_queue_status ((struct QUEUE_HEADER *) &ih->qhdr);
   ipCb->status |= MODIFIED;

   if (ip_options) {
      if (ip_rr)    ipCb->status |= RECORDROUTE;
      if (ip_ts)    ipCb->status |= TIMESTAMP;
      if (ip_sec)   ipCb->status |= SECURITY;
      if (ip_lsrr)  ipCb->status |= LSRCROUTE;
      if (ip_ssrr)  ipCb->status |= SSRCROUTE;
      if (ip_satid) ipCb->status |= SATNETID;
      if (ip_error) ipCb->status |= IPOPTIONMASK;
   }

   if (((tvp->tv_sec - ih->ip_cb.startime.tv_sec) >= iptimeout) ||
         (((ipCb->src.count == 1) && (ipCb->dst.count == 1)) &&
              !(ipCb->status & LOGGED)))

      log_ip_connection (ih, tvp, CON_ESTABLISHED);
}

int ipdeletecount = 0;

static void
delete_ip (ih)
struct IP_OBJECT *ih;
{
   ipdeletecount++;
   remove_hash_table_entry (&ip_hash_table, &ih->tha);
   if (ih->tha.buffer) free (ih->tha.buffer);
   free (ih);
}


static struct IP_OBJECT *
new_ip_hash_entry (ip, ep, rev, tvp, tha)
struct ip *ip;
struct ether_header *ep;
int rev;
struct argtimeval *tvp;
struct THA_OBJECT *tha;
{
   struct IP_OBJECT *ptr = NULL;
   struct ipcb *ipCb;

   if (ptr = (struct IP_OBJECT *) calloc (1, sizeof (*ptr))) {
      ipCb = &ptr->ip_cb;
      ipCb->status = IPPROTO | IP_INIT | MODIFIED;
      if (dflag) ipCb->status |= DETAIL;
      ipCb->startime = *tvp;
      ipCb->lasttime = *tvp;
      if (ipCb->rev = rev)
         ipCb->status |= REVERSE;

      if (ep) {
         if (ipCb->rev) {
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.etherdst, 6);
         } else {
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
         }
      }
      ptr->qhdr.last_time = *tvp;

      if (ptr->tha.buffer = (u_char *) malloc (tha->size)) {
         bcopy (tha->buffer, (u_char *) ptr->tha.buffer, tha->size);
         ptr->tha.size = tha->size;
      }

      if (ip_options) {
         if (ip_rr)    ipCb->status |= RECORDROUTE;
         if (ip_ts)    ipCb->status |= TIMESTAMP;
         if (ip_sec)   ipCb->status |= SECURITY;
         if (ip_lsrr)  ipCb->status |= LSRCROUTE;
         if (ip_ssrr)  ipCb->status |= SSRCROUTE;
         if (ip_satid) ipCb->status |= SATNETID;
         if (ip_error) ipCb->status |= IPOPTIONMASK;
      }  

      ipCb->src.count++;
      ipCb->src.bytes += ip->ip_len - (ip->ip_hl << 2);
      ipCb->src.ttl = ip_ttl;
      ipCb->src.tos = ip_tos;

      if (add_to_queue (&ip_display_list,(struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&ip_hash_table, tha, ptr);
         if (dflag) {
            log_ip_connection (ptr, tvp, IP_INIT);
            ipCb->status &= ~LOGGED;
         }
      } else {
         delete_ip (ptr);
         ptr = NULL;
      }
   }

   return (ptr);
}


#include <sys/stat.h>
#include <sys/errno.h>
 
extern int sys_nerr;
extern int errno;
 
extern int clienttags;
extern int clients[];
 
 
void
log_ip_connection (ptr, tvp, state)
struct IP_OBJECT *ptr;
struct argtimeval *tvp;
int state;
{
   unsigned int status = 0;
   int src_count, dst_count, src_bytes, dst_bytes;
   struct tha *tha;
   struct IP_CON_OBJECT *src, *dst;
   struct WriteStruct output;

   if (clienttags || wflag) {
      src = (struct IP_CON_OBJECT *) &ptr->ip_cb.src;
      dst = (struct IP_CON_OBJECT *) &ptr->ip_cb.dst;

      if ((src->count || dst->count) && (ptr->ip_cb.status & MODIFIED)) {
         bzero ((char *) &output, sizeof (struct WriteStruct));
         ptr->ip_cb.status |= state;
         status = ptr->ip_cb.status;
         ptr->ip_cb.status &= ~(MODIFIED | MULTIADDR | FRAGMENTS);
         output.ws_ip.startime =  ptr->ip_cb.startime;

         ((u_short *)&output.ws_ip.lasttime.tv_sec)[1] =
                                (u_short)(ptr->ip_cb.lasttime.tv_sec -
                                          ptr->ip_cb.startime.tv_sec);
         output.ws_ip.lasttime.tv_usec =  ptr->ip_cb.lasttime.tv_usec;

         ((u_char *)&output.ws_ip.lasttime.tv_sec)[0] = (u_char) src->ttl;
         ((u_char *)&output.ws_ip.lasttime.tv_sec)[1] = (u_char) dst->ttl;

         output.status   =  status;
         bcopy ((char *)&ptr->link.phys.ethersrc,
                             (char *)&output.ws_ip_phys.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst,
                             (char *)&output.ws_ip_phys.etherdst, 6);

         tha = (struct tha *) ptr->tha.buffer;
         output.ws_ip.src.s_addr = tha->src.s_addr;
         output.ws_ip.dst.s_addr = tha->dst.s_addr;
         output.ws_ip.port       = tha->port;

         output.ws_ip_udp.src_count = src->count - src->lastcount;
         output.ws_ip_udp.dst_count = dst->count - dst->lastcount;
         output.ws_ip_udp.src_bytes = src->bytes - src->lastbytes;
         output.ws_ip_udp.dst_bytes = dst->bytes - dst->lastbytes;

         if ((output.ws_ip_udp.src_count || output.ws_ip_udp.src_bytes) || 
                 (output.ws_ip_udp.dst_count || output.ws_ip_udp.dst_bytes)) {

            if (ptr->ip_cb.status & state) {
               writeOutData (&output, tvp);
               ptr->ip_cb.status |= LOGGED;
            }

            if (!(dflag && (state & IP_INIT)) &&
                               (ptr->ip_cb.status & LOGGED)) {
               src->lastcount = src->count;
               src->lastbytes = src->bytes;
               dst->lastcount = dst->count;
               dst->lastbytes = dst->bytes;
               ptr->ip_cb.startime = *tvp;
               ptr->ip_cb.lasttime = *tvp;
            }
         }
      }
   }

   ptr->qhdr.logtime = *tvp;
}


void
check_ip_timeouts (ptr, tvp)
struct IP_OBJECT *ptr;
struct argtimeval *tvp;
{
   int srccounts, dstcounts;

   if (!(ptr->ip_cb.status & LOGGED)) {
      srccounts = ptr->ip_cb.src.count;
      dstcounts = ptr->ip_cb.dst.count;
   } else {
      srccounts = (ptr->ip_cb.src.count - ptr->ip_cb.src.lastcount);
      dstcounts = (ptr->ip_cb.dst.count - ptr->ip_cb.dst.lastcount);
   }  
 
   if (srccounts || dstcounts) {
      ptr->ip_cb.status |= TIMED_OUT;
      log_ip_connection (ptr, tvp, TIMED_OUT);
   }
   remove_from_queue (&ip_display_list,
                          (struct QUEUE_HEADER *) &ptr->qhdr);
   delete_ip (ptr);

}
