
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * cons_tcp.c - handle tcp specific protocol state machine,
 *              and routines related to tcp event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */

#define CONS_TCP

#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>
#include <pcap.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <cons_tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp_fsm.h>

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;
extern int ip_error;

extern unsigned char ip_ttl;
extern unsigned char ip_tos;

static struct TCP_OBJECT *new_tcp_hash_entry ();
static void delete_tcp ();
static int update_seq ();
static int tcp_state ();

void create_tcp_tha ();
void update_tcp_record ();

void
cons_tcp_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &tcp_display_list, sizeof (tcp_display_list));
   bzero ((char *) &tcp_timeout_list, sizeof (tcp_timeout_list));
   bzero ((char *) &tcp_hash_table, sizeof (tcp_hash_table));
   if (ptr = (struct HASH_TABLE_HEADER **) calloc (TSEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      tcp_hash_table.hash_array = ptr;
      tcp_hash_table.size = TSEQ_HASHSIZE;
   } else
      perror ("cons_tcp_init: calloc");

   tcp_display_list.timerRoutine = check_tcp_timeouts;
   tcp_display_list.logRoutine =   log_tcp_connection;
   tcp_timeout_list.timerRoutine = check_tcp_timeouts;
   tcp_timeout_list.logRoutine =   log_tcp_connection;

   tcptimeout = tcptimeout ? tcptimeout : TCPTIMEOUT;
}

struct TCP_CON_OBJECT *tcpsrc, *tcpdst;

struct TCP_OBJECT *
cons_tcp (ep, tp, length, ip, tvp)
struct ether_header *ep;
struct tcphdr *tp;
int length;
struct ip *ip;
struct argtimeval *tvp;
{
   struct TCP_OBJECT *th = NULL;
   int rev = 0;
   struct tha buffer;
   struct THA_OBJECT thabuf, *tha = &thabuf;

   if ((length >= sizeof(struct tcphdr)) && (tp)) {
      tp->th_dport = ntohs(tp->th_dport);
      tp->th_sport = ntohs(tp->th_sport);
      tp->th_seq = ntohl(tp->th_seq);
      tp->th_ack = ntohl(tp->th_ack);

      tha->buffer = (u_char *)&buffer;
      create_tcp_tha (tha, ip, tp, &rev);

      if (th = (struct TCP_OBJECT *) find_hash_entry (&tcp_hash_table, tha))
         update_tcp_record (th, ep, tp, length, tvp, rev, tp->th_flags);
      else
         th = new_tcp_hash_entry (tp, ep, rev, tvp, ip, tha,
                                           length - (tp->th_off<<2));
   }

   return (th);
}


void
create_tcp_tha (tha, ip, tp, rev)
struct THA_OBJECT *tha;
struct ip *ip;
struct tcphdr *tp;
int *rev;
{
   struct tha thabuf;
   arg_uint16 dport, sport;
   arg_uint32 daddr, saddr;

   dport = (tp->th_dport);
   sport = (tp->th_sport);
   daddr = ip->ip_dst.s_addr;
   saddr = ip->ip_src.s_addr;

   if (dport > sport || ((sport == dport) && (saddr < daddr))) {
      thabuf.src.s_addr = daddr, thabuf.dst.s_addr = saddr;
      ((unsigned short *)&thabuf.port)[0] = tp->th_dport;
      ((unsigned short *)&thabuf.port)[1] = tp->th_sport;
      *rev = 1;

   } else {
      thabuf.src.s_addr = saddr, thabuf.dst.s_addr = daddr;
      ((unsigned short *)&thabuf.port)[0] = tp->th_sport;
      ((unsigned short *)&thabuf.port)[1] = tp->th_dport;
      *rev = 0;
   }

   if (tha->buffer)
      bcopy ((u_char *)&thabuf, tha->buffer, sizeof (struct tha));

   tha->size = sizeof (struct tha);
}


void
update_tcp_record (th, ep, tp, length, tvp, rev, flags)
struct TCP_OBJECT *th;
struct ether_header *ep;
struct tcphdr *tp;
int length;
struct argtimeval *tvp;
int rev;
u_char flags;
{
   struct tcpcb *tcpCb;
   int newstate, oldstate;

   tcpCb = &th->tcp_cb;
   tcpCb->lasttime = *tvp; th->qhdr.last_time = *tvp;

   if (th->tcp_cb.rev == rev) {
      tcpsrc = (struct TCP_CON_OBJECT *) &tcpCb->src;
      tcpdst = (struct TCP_CON_OBJECT *) &tcpCb->dst;
      tcpCb->src.count++;
      tcpCb->src.ttl = ip_ttl;
      tcpCb->src.tos = ip_tos;
   } else {
      tcpsrc = (struct TCP_CON_OBJECT *) &tcpCb->dst;
      tcpdst = (struct TCP_CON_OBJECT *) &tcpCb->src;
      tcpCb->dst.count++;
      tcpCb->dst.ttl = ip_ttl;
      tcpCb->dst.tos = ip_tos;
   }

   if (ep && !(tcpCb->status & MULTIADDR)) {
      if (rev) {
         if (bcmp ((char *)&ep->ether_dhost,
                    (char *)&th->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_shost,
                    (char *)&th->link.phys.etherdst, 6)) {
            tcpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&th->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost,
                           (char *)&th->link.phys.etherdst, 6);
         }
      } else
         if (bcmp ((char *)&ep->ether_shost,
                    (char *)&th->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_dhost,
                    (char *)&th->link.phys.etherdst, 6)) {
            tcpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_shost,
                           (char *)&th->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&th->link.phys.etherdst, 6);
         }  
   }

   if (ip_options) { 
      if (ip_ts)   tcpCb->status |= TIMESTAMP;
      if (ip_sec)  tcpCb->status |= SECURITY; 
      if (ip_lsrr) tcpCb->status |= LSRCROUTE; 
      if (ip_ssrr) tcpCb->status |= SSRCROUTE; 
      if (ip_rr)   tcpCb->status |= RECORDROUTE; 
      if (ip_satid) tcpCb->status |= SATNETID; 
      if (ip_error) tcpCb->status |= IPOPTIONMASK; 
   }

   if (update_seq (tp, th, rev, length - (tp->th_off << 2))) {
      update_queue_status ((struct QUEUE_HEADER *) &th->qhdr);

      oldstate = th->tcp_cb.t_state;
      switch (newstate = tcp_state (flags, th, tvp)) {
         case TCPS_LISTEN:
            if (th->qhdr.queue == &tcp_display_list)
               log_tcp_connection (th, tvp, 0L);
            remove_from_queue (th->qhdr.queue,
                                 (struct QUEUE_HEADER *) &th->qhdr);
            delete_tcp (th);
            th = NULL;
            break;

         case TCPS_CLOSED:
         case TCPS_TIME_WAIT:
            if (th->qhdr.queue == &tcp_display_list) {
               if (!(th->tcp_cb.status & RESET))
                  th->tcp_cb.status |= NORMAL_CLOSE;

               log_tcp_connection (th, tvp, 0L);
               remove_from_queue (&tcp_display_list,
                                 (struct QUEUE_HEADER *) &th->qhdr);
               if (!(add_to_queue (&tcp_timeout_list,
                                 (struct QUEUE_HEADER *) &th->qhdr))) {
                  tcpclosed++;
                  delete_tcp (th);
               }
            }
            break;
      }

      if (th && (!(tcpCb->status & MODIFIED))) {
         tcpCb->status |= MODIFIED;
         if (oldstate == newstate)
            tcpCb->inttime = *tvp;
         else
            tcpCb->inttime = tcpCb->lasttime;
      }
   }
}


static int
update_seq (tp, th, rev, len)
struct tcphdr *tp;
struct TCP_OBJECT *th;
int rev, len;
{
   int retn = 1, maxseq = 0;
   tcp_seq seq = tp->th_seq;
   tcp_seq newseq = seq + len;
   u_char flags = tp->th_flags;


   if (!(tcpsrc->win = ntohs(tp->th_win)) && !(flags & (TH_FIN|TH_RST)))
      tcpsrc->status |= WINDOW_SHUT;

   tcpsrc->bytes += len;
   tcpsrc->winbytes += len;  
   tcpsrc->ttl = ip_ttl;  
   tcpsrc->tos = ip_tos;  

   if (newseq < seq) {
      tcpsrc->seq_base = newseq;
      tcpsrc->seq = newseq;
   } else {
      if (!tcpsrc->seq_base) {
         tcpsrc->seq_base = seq;
         tcpsrc->seq = newseq;
      } else {
         if (len) {
            maxseq = (newseq > tcpsrc->seq) ? newseq : tcpsrc->seq;
            if (tcpsrc->win) {
               if (tcpsrc->winbytes > ((maxseq - 1) - tcpsrc->ack)) {
                  tcpsrc->retrans += len;
                  tcpsrc->status |= PKTS_RETRANS;
               }
            }
            tcpsrc->seq = maxseq;
         }
      }
   }

   if (tp->th_ack && (flags | TH_ACK)) {
      tcpdst->ack = tp->th_ack - 1;
      if (tcpdst->seq > tcpdst->ack)
         tcpdst->winbytes = (tcpdst->seq - 1) - tcpdst->ack;  
   }
   if (tp->th_seq < tcpsrc->ack)
      tcpsrc->strays++;

   return (retn);
}


static int
tcp_state (flags, th, tvp)
u_char flags;
struct TCP_OBJECT *th;
struct argtimeval *tvp;
{
   int state = th->tcp_cb.t_state;

   if ((flags &= ~TH_PUSH) & TH_RST) {
      if (!(th->tcp_cb.status & RESET)) {
         th->tcp_cb.status |= RESET;
         tcpsrc->status |= RESET;
      }

      state = TCPS_CLOSED;
   } else {
      switch (state) {
         case TCPS_LISTEN:
         case TCPS_SYN_SENT:
            if (flags == TH_SYN) {
               th->tcp_cb.startime = *tvp;
               tcpsrc->status |= PKTS_RETRANS;
            } else {
               if (flags == (TH_SYN|TH_ACK)) {
                  state = TCPS_SYN_RECEIVED;
                  th->tcp_cb.status |= SAW_SYN_SENT;
                  tcpsrc->status |= SAW_SYN_SENT;
                  if (dflag) log_tcp_connection (th, tvp, DETAIL);
               } else {
                  if (flags & TH_ACK) {
                     state = TCPS_ESTABLISHED;
                     th->tcp_cb.status |= CON_ESTABLISHED;
                     tcpsrc->status |= CON_ESTABLISHED;
                     if (dflag) log_tcp_connection (th, tvp, DETAIL);
                  }
               }
            }
 
            if (flags & TH_FIN)
               state = TCPS_CLOSING;
            break;
 
         case TCPS_SYN_RECEIVED:
            if (flags == (TH_FIN|TH_ACK))
               state = TCPS_FIN_WAIT_1;
            else if (!(flags & TH_SYN) && (flags & TH_ACK)) {
               state = TCPS_ESTABLISHED;
               th->tcp_cb.status |= CON_ESTABLISHED;
               tcpsrc->status |= CON_ESTABLISHED;
               if (dflag) log_tcp_connection (th, tvp, DETAIL);
            }
            break;
 
         case TCPS_ESTABLISHED:
            if (flags == (TH_FIN|TH_ACK)) {
               state = TCPS_FIN_WAIT_1;
               th->tcp_cb.status |= CLOSE_WAITING;
               tcpsrc->status |= CLOSE_WAITING;
            }
            if (flags == (TH_FIN)) {
               state = TCPS_CLOSE_WAIT;
               th->tcp_cb.status |= CLOSE_WAITING;
               tcpsrc->status |= CLOSE_WAITING;
            }
            break;
    
         case TCPS_CLOSE_WAIT:
         case TCPS_FIN_WAIT_1:
            if (flags & TH_SYN)
               state = TCPS_LISTEN;
            else {
               if ((flags & (TH_FIN|TH_ACK)) == (TH_FIN|TH_ACK)) {
                  if ((tcpsrc->seq == tcpsrc->ack) &&
                             (tcpdst->seq == tcpdst->ack))
                     state=TCPS_CLOSED;
               } else
                  if (flags & TH_FIN)
                     state = TCPS_CLOSING;
                  else 
                     if (flags & TH_ACK)
                        if (tcpdst->seq == tcpdst->ack)
                           state = TCPS_FIN_WAIT_2;
            }
            break;
      
      case TCPS_LAST_ACK:
      case TCPS_FIN_WAIT_2:
         if ((flags & (TH_FIN))) {
            if ((tcpsrc->seq == tcpsrc->ack) &&
                         (tcpdst->seq == tcpdst->ack))
               state = TCPS_CLOSED;
            else
               state = TCPS_CLOSING;
         } else 
            if (flags & TH_ACK)
               if ((tcpsrc->seq == tcpsrc->ack) &&
                         (tcpdst->seq == tcpdst->ack))
                  state = TCPS_CLOSED;
         break;
      
      case TCPS_CLOSING:
      case TCPS_TIME_WAIT:
         if (flags & TH_SYN)
            state = TCPS_LISTEN;
         else if (flags & TH_ACK)
            if ((tcpsrc->seq == tcpsrc->ack) &&
                         (tcpdst->seq == tcpdst->ack))
               state = TCPS_CLOSED;
         break;
      
      case TCPS_CLOSED:
         if (flags & TH_SYN)
            state = TCPS_LISTEN;
         break;
      }
   }
  
   if (state != TCPS_LISTEN)
      th->tcp_cb.t_state = state;
   return (state);
}


int tcpdeletecount = 0;

static void
delete_tcp (th)
struct TCP_OBJECT *th;
{
   tcpdeletecount++;
   remove_hash_table_entry (&tcp_hash_table, &th->tha);
   if (th->tha.buffer)
      free (th->tha.buffer);
   free (th);
}


      
static struct TCP_OBJECT *
new_tcp_hash_entry (tp, ep, rev, tvp, ip, tha, len)
struct tcphdr *tp;
struct ether_header *ep;
int rev;
struct argtimeval *tvp;
struct ip *ip;
struct THA_OBJECT *tha;
int len;
{
   struct TCP_OBJECT *ptr = NULL;
   struct tcpcb *tcpCb;
   u_char flags = tp->th_flags;
   u_char etherbuf[6];

   if (!(flags & TH_RST)) {
      if (ptr = (struct TCP_OBJECT *) calloc (1, sizeof (*ptr))) {
         tcpCb = &ptr->tcp_cb;

         tcpCb->status = TCPPROTO;
         if (ip) tcpCb->status |= IPPROTO;
         tcpCb->status |= MODIFIED;
         if (tcpCb->rev = rev)
            tcpCb->status |= REVERSE;

         tcpCb->startime = *tvp;
         tcpCb->lasttime = *tvp;
         tcpCb->inttime  = *tvp;

         if (ep) {
            if (tcpCb->rev) {
               bcopy ((char *)&ep->ether_dhost,
                          (char *)&ptr->link.phys.ethersrc, 6);
               bcopy ((char *)&ep->ether_shost,
                          (char *)&ptr->link.phys.etherdst, 6);
            } else {
               bcopy ((char *)&ep->ether_shost,
                          (char *)&ptr->link.phys.ethersrc, 6);
               bcopy ((char *)&ep->ether_dhost,
                          (char *)&ptr->link.phys.etherdst, 6);
            }
         }

         if (ptr->tha.buffer = (u_char *) malloc (tha->size)) {
            bcopy (tha->buffer, (u_char *) ptr->tha.buffer, tha->size);
            ptr->tha.size = tha->size;
         }

         ptr->qhdr.last_time = *tvp;

         if (ip_options) { 
            if (ip_ts)   tcpCb->status |= TIMESTAMP;
            if (ip_sec)  tcpCb->status |= SECURITY; 
            if (ip_lsrr) tcpCb->status |= LSRCROUTE; 
            if (ip_ssrr) tcpCb->status |= SSRCROUTE; 
            if (ip_rr)   tcpCb->status |= RECORDROUTE; 
            if (ip_satid) tcpCb->status |= SATNETID; 
            if (ip_error) tcpCb->status |= IPOPTIONMASK; 
         }

         switch (flags) {
            case (TH_SYN):
               tcpCb->t_state = TCPS_SYN_SENT;
               tcpCb->status |= SAW_SYN;
               tcpCb->src.count = 1;
               tcpCb->src.bytes = len;
               tcpCb->src.ttl = ip_ttl;
               tcpCb->src.tos = ip_tos;
               tcpCb->src.seq_base = tp->th_seq; 
               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            case (TH_SYN|TH_ACK):
               tcpCb->t_state = TCPS_SYN_RECEIVED;
               tcpCb->status |= SAW_SYN_SENT;
               tcpCb->rev = rev ? 0 : 1;
               tcpCb->status ^= REVERSE;
               bcopy ((char *)&ptr->link.phys.ethersrc, (char *) &etherbuf, 6);
               bcopy ((char *)&ptr->link.phys.etherdst,
                      (char *)&ptr->link.phys.ethersrc, 6);
               bcopy ((char *) &etherbuf, (char *)&ptr->link.phys.ethersrc, 6);

               tcpCb->dst.count = 1;
               tcpCb->dst.bytes = len;
               tcpCb->dst.ttl = ip_ttl;
               tcpCb->dst.tos = ip_tos;
               tcpCb->dst.seq_base = tp->th_seq;
               tcpCb->src.ack = tp->th_ack;
               tcpCb->src.seq_base = tp->th_ack - 1;

               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            case (TH_ACK):
            case (TH_PUSH|TH_ACK):
            case (TH_URG|TH_ACK):
            case (TH_PUSH|TH_URG|TH_ACK):
               tcpCb->dst.ack = tp->th_ack - 1;
               tcpCb->dst.seq_base = tp->th_ack - 1;
            case (TH_PUSH):
            case (TH_URG):
            case (TH_PUSH|TH_URG):
               tcpCb->src.count = 1;
               tcpCb->src.bytes = len;
               tcpCb->src.ttl = ip_ttl;
               tcpCb->src.tos = ip_tos;
               tcpCb->src.seq_base = tp->th_seq - 1;
               tcpCb->src.seq = tp->th_seq + ((len > 20) ? (len - (tp->th_off << 2)) : 0);
               tcpCb->t_state = TCPS_ESTABLISHED;
               tcpCb->status |= CON_ESTABLISHED;
               if (dflag) log_tcp_connection (ptr, tvp, DETAIL);
               break;

            default:
               tcpCb->t_state = TCPS_CLOSING;
               break;
         }

         if (tcpCb->t_state == TCPS_CLOSING) {
            if (add_to_queue (&tcp_timeout_list,
                           (struct QUEUE_HEADER *) &ptr->qhdr))
               (void) add_hash_table_entry (&tcp_hash_table, tha, ptr);

            else
               tcpclosed++;
         } else
            if (add_to_queue (&tcp_display_list,
                           (struct QUEUE_HEADER *) &ptr->qhdr))
               (void) add_hash_table_entry (&tcp_hash_table, tha, ptr);

            else {
               delete_tcp (ptr);
               ptr = NULL;
            }
      }
   }

   return (ptr);
}


#include <sys/stat.h>
#include <sys/errno.h>

extern int errno;

void
log_tcp_connection (ptr, tvp, state)
struct TCP_OBJECT *ptr;
struct argtimeval *tvp;
int state;
{
   arg_uint32 status = 0;
   int src_count, dst_count, src_bytes, dst_bytes;
   struct TCP_CON_OBJECT *src, *dst;
   struct WriteStruct output;
   struct tha *tha;

   if (clienttags || wflag) {
      bzero ((char *) &output, sizeof (struct WriteStruct));
      ptr->tcp_cb.status |= state;

      src = (struct TCP_CON_OBJECT *) &ptr->tcp_cb.src;
      dst = (struct TCP_CON_OBJECT *) &ptr->tcp_cb.dst;

      ptr->tcp_cb.status &= ~RESET;

      if (src->status & PKTS_RETRANS)
         ptr->tcp_cb.status |= SRC_PKTS_RETRANS;

      if (src->status & RESET)
         ptr->tcp_cb.status |= SRC_RESET;

      if (src->status & WINDOW_SHUT) 
         ptr->tcp_cb.status |= SRC_WINDOW_SHUT;

      if (dst->status & PKTS_RETRANS)
         ptr->tcp_cb.status |= DST_PKTS_RETRANS;

      if (dst->status & RESET)       
         ptr->tcp_cb.status |= DST_RESET;

      if (dst->status & WINDOW_SHUT) 
         ptr->tcp_cb.status |= DST_WINDOW_SHUT;

      status = ptr->tcp_cb.status;

      ptr->tcp_cb.status &= ~(MODIFIED | MULTIADDR | FRAGMENTS);

      if (dflag && (!(status & (NORMAL_CLOSE|RESET|TIMED_OUT)))) {
         output.ws_ip.startime =  ptr->tcp_cb.inttime;
      } else {
         output.ws_ip.startime =  ptr->tcp_cb.startime;
      }

      ((u_short *)&output.ws_ip.lasttime.tv_sec)[1] =
                             (u_short)(ptr->tcp_cb.lasttime.tv_sec -
                                      output.ws_ip.startime.tv_sec);
      output.ws_ip.lasttime.tv_usec =  ptr->tcp_cb.lasttime.tv_usec;
      ((u_char *)&output.ws_ip.lasttime.tv_sec)[0] = (u_char) src->ttl;
      ((u_char *)&output.ws_ip.lasttime.tv_sec)[1] = (u_char) dst->ttl;

      output.status   =  status;
      bcopy ((char *)&ptr->link.phys.ethersrc,
                           (char *)&output.ws_ip_phys.ethersrc, 6);
      bcopy ((char *)&ptr->link.phys.etherdst,
                           (char *)&output.ws_ip_phys.etherdst, 6);

      tha = (struct tha *) ptr->tha.buffer;
      output.ws_ip.src.s_addr =  tha->src.s_addr;
      output.ws_ip.dst.s_addr =  tha->dst.s_addr;
      output.ws_ip.port       =  tha->port;

      output.ws_ip_tcp.src_count = src->count;
      output.ws_ip_tcp.dst_count = dst->count;

      if (dflag && ((status & (SAW_SYN | SAW_SYN_SENT)) &&
                         (!(status & (CON_ESTABLISHED | RESET))))) {
         if (status & SAW_SYN_SENT)
            output.ws_ip_inittcp.seq = ptr->tcp_cb.dst.seq_base;
         else
            output.ws_ip_inittcp.seq = ptr->tcp_cb.src.seq_base;
      } else {
         output.ws_ip_tcp.src_bytes = src->bytes;
         output.ws_ip_tcp.dst_bytes = dst->bytes;
      }

      if (dflag && (status & CON_ESTABLISHED) &&
              (!(status & (NORMAL_CLOSE|RESET|TIMED_OUT)))) {
         src_count = output.ws_ip_tcp.src_count;
         dst_count = output.ws_ip_tcp.dst_count;
         src_bytes = output.ws_ip_tcp.src_bytes;
         dst_bytes = output.ws_ip_tcp.dst_bytes;
         output.ws_ip_tcp.src_count = src_count - src->lastcount;
         output.ws_ip_tcp.dst_count = dst_count - dst->lastcount;
         output.ws_ip_tcp.src_bytes = src_bytes - src->lastbytes;
         output.ws_ip_tcp.dst_bytes = dst_bytes - dst->lastbytes;
         src->lastcount = src_count;
         src->lastbytes = src_bytes;
         dst->lastcount = dst_count;
         dst->lastbytes = dst_bytes;
      }

      if ((output.ws_ip_tcp.src_count || output.ws_ip_tcp.dst_count) ||
          (output.ws_ip_tcp.src_bytes || output.ws_ip_tcp.dst_bytes))
         writeOutData (&output, tvp);

      ptr->tcp_cb.status &= ~(PKTS_RETRANS & WINDOW_SHUT);

      src->status &= ~(PKTS_RETRANS & WINDOW_SHUT);
      dst->status &= ~(PKTS_RETRANS & WINDOW_SHUT);

      ptr->tcp_cb.status |= LOGGED;
      ptr->tcp_cb.status ^= state;
   }

   ptr->qhdr.logtime = *tvp;
}


void
check_tcp_timeouts (ptr, tvp)
struct TCP_OBJECT *ptr;
struct argtimeval *tvp;
{
   switch (ptr->tcp_cb.t_state) {
      case TCPS_SYN_SENT:
      case TCPS_SYN_RECEIVED:
      case TCPS_ESTABLISHED:
      case TCPS_CLOSE_WAIT:
      case TCPS_FIN_WAIT_1:
      case TCPS_LAST_ACK:
      case TCPS_FIN_WAIT_2:
      case TCPS_CLOSING:
      case TCPS_TIME_WAIT:
         ptr->tcp_cb.status |= TIMED_OUT;
         if (ptr->qhdr.queue == &tcp_display_list)
            log_tcp_connection (ptr, tvp, 0);
         break;
   }

   remove_from_queue (ptr->qhdr.queue, (struct QUEUE_HEADER *)&ptr->qhdr);
   tcpclosed++;
   delete_tcp (ptr);
}
