
/*
 * Copyright (c) 1997 Carter Bullard
 * All applicable rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is restricted to personal use only.  Use, sale
 * or retransmission of this software for commercial purposes, 
 * including but not limited to use as a commerical product or
 * in support of a commercial endeavor requires licensing from Carter
 * Bullard.
 *
 * CARTER BULLARD DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL CARTER BULLARD BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */


/*
 * cons_udp.c - handle udp specific protocol state machine,
 *              and routines related to udp event reporting.
 *
 * written by Carter Bullard
 * Software Engineering Institute
 * Carnegie Mellon Univeristy
 *
 */


#if !defined(HAVE_SOLARIS)
#include <stdlib.h>
#include <unistd.h>
#endif

#include <compat.h>
#include <pcap.h>

#include <netinet/in.h>
#include <interface.h>
#include <argus.h>

#include <cons_udp.h>

extern int clienttags;
extern int clients[];
extern int ip_options;
extern int ip_eol;
extern int ip_nop;
extern int ip_ts;
extern int ip_rr;
extern int ip_sec;
extern int ip_lsrr;
extern int ip_ssrr;
extern int ip_satid;
extern int ip_error;

extern unsigned char ip_ttl;
extern unsigned char ip_tos;

static struct UDP_OBJECT *new_udp_hash_entry ();
static void delete_udp ();
static void remove_udp_hash_entry ();

void create_udp_tha ();
void adjust_group_port ();
void update_udp_record ();


void
cons_udp_init ()
{
   struct HASH_TABLE_HEADER **ptr;

   bzero ((char *) &udp_display_list, sizeof (udp_display_list));
   bzero ((char *) &udp_hash_table, sizeof (udp_hash_table));

   if (ptr = (struct HASH_TABLE_HEADER **) calloc (USEQ_HASHSIZE,
        sizeof (struct HASH_TABLE_HEADER *))) {
      udp_hash_table.hash_array = ptr;
      udp_hash_table.size = USEQ_HASHSIZE;
   } else
      perror ("cons_udp_init: calloc");

   udptimeout = udptimeout ? udptimeout : UDPTIMEOUT; 
   udp_display_list.timerRoutine = check_udp_timeouts;
   udp_display_list.logRoutine   = log_udp_connection;
}

struct UDP_OBJECT *
cons_udp (ep, up, length, ip, tvp)
struct ether_header *ep;
struct udphdr *up;
int length;
struct ip *ip;
struct argtimeval *tvp;
{
   struct UDP_OBJECT *uh = NULL;
   int rev;
   struct tha buffer;
   struct THA_OBJECT thabuf, *tha = &thabuf;


   if ((length >= sizeof(struct udphdr)) && (up)) {
      up->uh_ulen  = ntohs(up->uh_ulen);
      up->uh_sport = ntohs(up->uh_sport);
      up->uh_dport = ntohs(up->uh_dport);

      tha->buffer = (u_char *) &buffer;
      create_udp_tha (tha, ip, up, &rev);

      if (uh = (struct UDP_OBJECT *) find_hash_entry (&udp_hash_table, tha))
         update_udp_record (uh, ep, up, (int)up->uh_ulen, tvp, rev);
      else
         uh = new_udp_hash_entry (up, ep, rev, tvp, ip, tha, (int)up->uh_ulen);
   }

   return (uh);
}


void
create_udp_tha (tha, ip, up, rev)
struct THA_OBJECT *tha;
struct ip *ip;
struct udphdr *up;
int *rev;
{
   struct tha thabuf;

   adjust_group_port (&up->uh_sport, &up->uh_dport);

   if ((up->uh_dport > up->uh_sport) || ((up->uh_dport == up->uh_sport) &&
            (ip->ip_src.s_addr < ip->ip_dst.s_addr))) {
      thabuf.src = ip->ip_dst, thabuf.dst = ip->ip_src;
      ((unsigned short *)&thabuf.port)[0] = up->uh_dport;
      ((unsigned short *)&thabuf.port)[1] = up->uh_sport;
      *rev = 1;

   } else {
      thabuf.src = ip->ip_src, thabuf.dst = ip->ip_dst;
      ((unsigned short *)&thabuf.port)[0] = up->uh_sport;
      ((unsigned short *)&thabuf.port)[1] = up->uh_dport;
      *rev = 0;
   }

   if (tha->buffer)
      bcopy ((u_char *)&thabuf, tha->buffer, sizeof (struct tha));

   tha->size = sizeof (struct tha);
}

void
adjust_group_port (sport, dport)
unsigned short *sport, *dport;
{
   int is_src_server = 0, is_dst_server = 0;
   extern unsigned short *udpServicePorts;
   unsigned short *ptr = udpServicePorts, *end = NULL;
 
   if (ptr && *ptr) {
      do {
         if (*sport == *ptr) is_src_server = 1;
         if (*dport == *ptr) is_dst_server = 1;
         ptr++;
      } while (*ptr);

      if ((is_src_server || is_dst_server) &&
                              !(is_src_server && is_dst_server)) {
         if (is_src_server) *dport = 0;
         if (is_dst_server) *sport = 0;
      }
   }
}


void
update_udp_record (uh, ep, up, length, tvp, rev)
struct UDP_OBJECT *uh;
struct ether_header *ep;
struct udphdr *up;
int length;
struct argtimeval *tvp;
int rev;
{
   struct udpcb *udpCb;

   udpCb = &uh->udp_cb;
   udpCb->lasttime = *tvp; uh->qhdr.last_time = *tvp;
   if (udpCb->rev == rev) {
      udpCb->src.count++;
      udpCb->src.bytes += length;
      udpCb->src.ttl = ip_ttl;
      udpCb->src.tos = ip_tos;
   } else {
      udpCb->dst.count++;
      udpCb->dst.bytes += length;
      udpCb->dst.ttl = ip_ttl;
      udpCb->dst.tos = ip_tos;
   }

   if (ep && !(udpCb->status & MULTIADDR)) {
      if (rev) {
         if (bcmp ((char *)&ep->ether_dhost,
                    (char *)&uh->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_shost,
                    (char *)&uh->link.phys.etherdst, 6)) {
            udpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&uh->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost,
                           (char *)&uh->link.phys.etherdst, 6);
         }
      } else
         if (bcmp ((char *)&ep->ether_shost,
                    (char *)&uh->link.phys.ethersrc, 6) ||
             bcmp ((char *)&ep->ether_dhost,
                    (char *)&uh->link.phys.etherdst, 6)) {
            udpCb->status |= MULTIADDR;
            bcopy ((char *)&ep->ether_shost,
                           (char *)&uh->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost,
                           (char *)&uh->link.phys.etherdst, 6);
         }  
   }

   if (ip_options) { 
      if (ip_rr)   udpCb->status |= RECORDROUTE; 
      if (ip_ts)   udpCb->status |= TIMESTAMP;
      if (ip_sec)  udpCb->status |= SECURITY; 
      if (ip_lsrr) udpCb->status |= LSRCROUTE; 
      if (ip_ssrr) udpCb->status |= SSRCROUTE; 
      if (ip_satid) udpCb->status |= SATNETID; 
      if (ip_error) udpCb->status |= IPOPTIONMASK; 
   }

   if ((udpCb->src.count + udpCb->dst.count) == 2)
      udpCb->status |= CON_ESTABLISHED;

   update_queue_status ((struct QUEUE_HEADER *) &uh->qhdr);
   udpCb->status |= MODIFIED;

   if (((tvp->tv_sec - udpCb->startime.tv_sec) >= udptimeout) ||
         (((udpCb->src.count == 1) && (udpCb->dst.count == 1)) &&
              !(udpCb->status & LOGGED)))
      log_udp_connection (uh, tvp, CON_ESTABLISHED);
}
 

int udpdeletecount = 0;

static void
delete_udp (uh)
struct UDP_OBJECT *uh;
{
   udpdeletecount++;
   remove_hash_table_entry (&udp_hash_table, &uh->tha);
   if (uh->tha.buffer) free (uh->tha.buffer);
   free (uh);
}


static struct UDP_OBJECT *
new_udp_hash_entry (up, ep, rev, tvp, ip, tha, len)
struct udphdr *up;
struct ether_header *ep;
int rev;
struct argtimeval *tvp;
struct ip *ip;
struct THA_OBJECT *tha;
int len;
{
   struct UDP_OBJECT *ptr = NULL;
   struct udpcb *udpCb;

   if (ptr = (struct UDP_OBJECT *) calloc (1, sizeof (*ptr))) {
      udpCb = &ptr->udp_cb;
      udpCb->status = UDPPROTO | UDP_INIT | MODIFIED;
      if (ip) udpCb->status |= IPPROTO;
      if (dflag) udpCb->status |= DETAIL;
      udpCb->startime = *tvp;
      udpCb->lasttime = *tvp;
      if (udpCb->rev = rev)
         udpCb->status |= REVERSE;

      if (ep) {
         if (udpCb->rev) {
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.etherdst, 6);
   
         } else {
            bcopy ((char *)&ep->ether_shost, (char *)&ptr->link.phys.ethersrc, 6);
            bcopy ((char *)&ep->ether_dhost, (char *)&ptr->link.phys.etherdst, 6);
         }
      }

      ptr->qhdr.last_time = *tvp;

      if (ptr->tha.buffer = (u_char *) malloc (tha->size)) {
         bcopy (tha->buffer, (u_char *) ptr->tha.buffer, tha->size);
         ptr->tha.size = tha->size;
      }

      if (ip_options) { 
         if (ip_rr)   udpCb->status |= RECORDROUTE; 
         if (ip_ts)   udpCb->status |= TIMESTAMP;
         if (ip_sec)  udpCb->status |= SECURITY; 
         if (ip_lsrr) udpCb->status |= LSRCROUTE; 
         if (ip_ssrr) udpCb->status |= SSRCROUTE; 
         if (ip_satid) udpCb->status |= SATNETID; 
         if (ip_error) udpCb->status |= IPOPTIONMASK; 
      }

      udpCb->src.count++;
      udpCb->src.bytes += len;
      udpCb->src.ttl = ip_ttl;
      udpCb->src.tos = ip_tos;

      if (add_to_queue (&udp_display_list, (struct QUEUE_HEADER *)&ptr->qhdr)) {
         (void) add_hash_table_entry (&udp_hash_table, tha, ptr);
         if (dflag) {
            log_udp_connection (ptr, tvp, UDP_INIT);
            udpCb->status &= ~LOGGED;
         }
      } else {
         delete_udp (ptr);
         ptr = NULL;
      }
   }

   return (ptr);
}


#include <sys/stat.h>
#include <sys/errno.h>

void
log_udp_connection (ptr, tvp, state)
struct UDP_OBJECT *ptr;
struct argtimeval *tvp;
int state;
{
   unsigned int status = 0;
   struct tha *tha;
   int src_count, dst_count, src_bytes, dst_bytes;
   struct UDP_CON_OBJECT *src, *dst;
   struct WriteStruct output;

   if (clienttags || wflag) {
      src = (struct UDP_CON_OBJECT *) &ptr->udp_cb.src;
      dst = (struct UDP_CON_OBJECT *) &ptr->udp_cb.dst;

      if ((src->count || dst->count) && (ptr->udp_cb.status & MODIFIED)) {
         bzero ((char *) &output, sizeof (struct WriteStruct));
         ptr->udp_cb.status |= state;
         status = ptr->udp_cb.status;
         ptr->udp_cb.status &= ~(MODIFIED | MULTIADDR | FRAGMENTS);
         output.ws_ip.startime =  ptr->udp_cb.startime;


         ((u_short *)&output.ws_ip.lasttime.tv_sec)[1] =
                                (u_short)(ptr->udp_cb.lasttime.tv_sec -
                                          ptr->udp_cb.startime.tv_sec);
         output.ws_ip.lasttime.tv_usec =  ptr->udp_cb.lasttime.tv_usec;
         ((u_char *)&output.ws_ip.lasttime.tv_sec)[0] = (u_char) src->ttl;
         ((u_char *)&output.ws_ip.lasttime.tv_sec)[1] = (u_char) dst->ttl;

         output.status   =  status;
         bcopy ((char *)&ptr->link.phys.ethersrc,
                             (char *)&output.ws_ip_phys.ethersrc, 6);
         bcopy ((char *)&ptr->link.phys.etherdst,
                             (char *)&output.ws_ip_phys.etherdst, 6);

         tha = (struct tha *) ptr->tha.buffer;
         output.ws_ip.src.s_addr = tha->src.s_addr;
         output.ws_ip.dst.s_addr = tha->dst.s_addr;
         output.ws_ip.port       = tha->port;

         output.ws_ip_udp.src_count = src->count - src->lastcount;
         output.ws_ip_udp.dst_count = dst->count - dst->lastcount;
         output.ws_ip_udp.src_bytes = src->bytes - src->lastbytes;
         output.ws_ip_udp.dst_bytes = dst->bytes - dst->lastbytes;

         if ((output.ws_ip_udp.src_count || output.ws_ip_udp.src_bytes) || 
                 (output.ws_ip_udp.dst_count || output.ws_ip_udp.dst_bytes)) {

            if (ptr->udp_cb.status & state) {
               writeOutData (&output, tvp);
               ptr->udp_cb.status |= LOGGED;
            }

            if (!(dflag && (state & UDP_INIT)) &&
                               (ptr->udp_cb.status & LOGGED)) {
               src->lastcount = src->count;
               src->lastbytes = src->bytes;
               dst->lastcount = dst->count;
               dst->lastbytes = dst->bytes;
               ptr->udp_cb.startime = *tvp;
               ptr->udp_cb.lasttime = *tvp;
            }
         }
      }
   }

   ptr->qhdr.logtime = *tvp;
}
   

void
check_udp_timeouts (ptr, tvp)
struct UDP_OBJECT *ptr;
struct argtimeval *tvp;
{
   int srccounts, dstcounts;

   if (!(ptr->udp_cb.status & LOGGED)) {
      srccounts = ptr->udp_cb.src.count;
      dstcounts = ptr->udp_cb.dst.count;
   } else {
      srccounts = (ptr->udp_cb.src.count - ptr->udp_cb.src.lastcount);
      dstcounts = (ptr->udp_cb.dst.count - ptr->udp_cb.dst.lastcount);
   }  
 
   if (srccounts || dstcounts) {
      ptr->udp_cb.status |= TIMED_OUT;
      log_udp_connection (ptr, tvp, TIMED_OUT);
   }
   remove_from_queue (&udp_display_list,
                       (struct QUEUE_HEADER *) &ptr->qhdr);
   delete_udp (ptr);
}
