/* crlfetch.c - LDAP access
 *      Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *      Copyright (C) 2003 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>

#include <stdio.h>
#include <errno.h>

#include "crlfetch.h"
#include "dirmngr.h"
#include "misc.h"
#include "http.h"

struct fun_cookie_s {
  unsigned char *value; /* Malloced buffer with the data to deliver. */
  size_t valuelen;      /* Size of this buffer. */
  size_t nread;         /* Bytes already read from value. */
};


static int 
fun_reader (void *cookie, char *buffer, int size)
{
  struct fun_cookie_s *h = cookie;

  if (h->nread >= h->valuelen)
    return 0; /* EOF */
  if (h->nread + size > h->valuelen)
    size = h->valuelen - h->nread;
  memcpy (buffer, h->value + h->nread, size);
  h->nread += size;
  return size;
}

static int
fun_closer (void *cookie)
{
  struct fun_cookie_s *h = cookie;

  xfree (h->value);
  xfree (h);
  return 0;
}

/* Helper to prepare a stream from VALUE and VALUELNE and return that
   stream in R_FP.  In the error case VALUE will be freed. */
static gpg_error_t
setup_funopen (unsigned char *value, size_t valuelen, FILE **r_fp)
{
  gpg_error_t err = 0;
  struct fun_cookie_s *h;
  FILE *fp;

  h = xtrymalloc (sizeof *h);
  if (!h)
    {
      err = gpg_error_from_errno (errno);
      xfree (value);
    }
  else 
    {
      h->value = value;
      value = NULL;
      h->valuelen = valuelen;
      h->nread = 0;
      fp = funopen (h, fun_reader, NULL, NULL, fun_closer);
      if (!fp)
        {
          err = gpg_error_from_errno (errno);
          xfree (h->value);
          xfree (h);
        }
      else
        *r_fp = fp;
    }
  return err;
}

/* Fetch CRL from URL and return the entire CRL as a newly opened
   stream returned in R_FP.

   FIXME: we should look for a way to return the stream direct from
   the ldap core and not use a temporary buffer.
 */
gpg_error_t
crl_fetch (const char *url, FILE **r_fp)
{
  gpg_error_t err;
  unsigned char *value = NULL;
  size_t valuelen;
  parsed_uri_t uri;

  *r_fp = NULL;

  err = http_parse_uri (&uri, url);
  http_release_parsed_uri (uri);
  if (!err) /* Yes, our HTTP code groks that. */
    {
      struct http_context_s hd;
      
      err = http_open_document (&hd, url, 0);
      if (err)
        log_error ("error retrieving `%s': %s\n", url, gpg_strerror (err));
      else if (hd.status_code != 200)
        {
          log_error ("error retrieving `%s': http status %u\n",
                     url, hd.status_code);
          err = gpg_error (GPG_ERR_NO_DATA);
          http_close (&hd, 0);
        }
      else
        {
          *r_fp = hd.fp_read;
          http_close (&hd, 1);
        }
    }
  else /* Let the LDAP code try other schemes. */
    {
      err = url_fetch_ldap (url, "certificateRevocationList;binary",
                            &value, &valuelen);
      if (!err)
        err = setup_funopen (value, valuelen, r_fp);
      else if (err && *value)
        xfree (value);
    }
  return err;
}


/* Fetch CRL for ISSUER using a default server. Return the entire CRL
   as a newly opened stream returned in R_FP. */
gpg_error_t
crl_fetch_default (const char *issuer, FILE **r_fp)
{
  gpg_error_t err;
  unsigned char *value = NULL;
  size_t valuelen;
  
  *r_fp = NULL;
  err = attr_fetch_ldap (issuer, "certificateRevocationList;binary", 
                         &value, &valuelen);
  if (!err)
    err = setup_funopen (value, valuelen, r_fp);
  else if (err && value)
    xfree (value);
  return err;
}


/* Fetch a CA certificate for DN using the default server. */
int
ca_cert_fetch (const char *dn, unsigned char **value, size_t *valuelen)
{
  int rc;

  *value = NULL;
  rc = attr_fetch_ldap (dn, "cACertificate;binary", value, valuelen);
  if (rc && *value)
    {
      xfree (*value);
      *value = NULL;
      *valuelen = 0;
    }
  return rc;
}


gpg_error_t
start_cert_fetch (cert_fetch_context_t *context,
                  strlist_t patterns, const ldap_server_t server)
{
  return start_cert_fetch_ldap (context, patterns, server);
}

gpg_error_t
fetch_next_cert (cert_fetch_context_t context,
                 unsigned char **value, size_t * valuelen)
{
  return fetch_next_cert_ldap (context, value, valuelen);
}

void
end_cert_fetch (cert_fetch_context_t context)
{
  return end_cert_fetch_ldap (context);
}
