/* dirmngr.c - LDAP access 
 *	Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifndef DIRMNGR_H
#define DIRMNGR_H

#include <sys/time.h>
#include <unistd.h>
#include <gcrypt.h> /* we need this for the memory function protos */
#include <gpg-error.h>
#include <ksba.h>

/* to pass hash functions to libksba we need to cast it */
#define HASH_FNC ((void (*)(void *, const void*,size_t))gcry_md_write)

/* get all the stuff from jnlib */
#include "../jnlib/logging.h"
#include "../jnlib/argparse.h"
#include "../jnlib/stringhelp.h"
#include "../jnlib/mischelp.h"
#include "../jnlib/strlist.h"
#include "../jnlib/dotlock.h"

#include "error.h"

/* handy malloc macros  - use only them */
#define xtrymalloc(a)    gcry_malloc ((a))
#define xtrycalloc(a,b)  gcry_calloc ((a),(b))
#define xtryrealloc(a,b) gcry_realloc ((a),(b))
#define xtrystrdup(a)    gcry_strdup ((a))
#define xfree(a)         gcry_free ((a))

#define xmalloc(a)       gcry_xmalloc ((a))
#define xcalloc(a,b)     gcry_xcalloc ((a),(b))
#define xrealloc(a,b)    gcry_xrealloc ((a),(b))
#define xstrdup(a)       gcry_xstrdup ((a))

#define seterr(a)  (GNUPG_ ## a)

/* some macros to replace ctype ones and avoid locale problems */
#define spacep(p)   (*(p) == ' ' || *(p) == '\t')
#define digitp(p)   (*(p) >= '0' && *(p) <= '9')
#define hexdigitp(a) (digitp (a)                     \
                      || (*(a) >= 'A' && *(a) <= 'F')  \
                      || (*(a) >= 'a' && *(a) <= 'f'))
/* the atoi macros assume that the buffer has only valid digits */
#define atoi_1(p)   (*(p) - '0' )
#define atoi_2(p)   ((atoi_1(p) * 10) + atoi_1((p)+1))
#define atoi_4(p)   ((atoi_2(p) * 100) + atoi_2((p)+2))
#define xtoi_1(p)   (*(p) <= '9'? (*(p)- '0'): \
                     *(p) <= 'F'? (*(p)-'A'+10):(*(p)-'a'+10))
#define xtoi_2(p)   ((xtoi_1(p) * 16) + xtoi_1((p)+1))


/* This objects keeps information about a particular LDAP server and
   is used as item of a single linked list of servers. */
struct ldap_server_s {
  struct ldap_server_s* next;

  char *host;
  int   port;
  char *user;
  char *pass;
  char *base;
};
typedef struct ldap_server_s *ldap_server_t;



/* A large struct name "opt" to keep global flags. */
struct {
  unsigned int debug; /* debug flags (DBG_foo_VALUE) */
  int verbose;        /* verbosity level */
  int quiet;          /* be as quiet as possible */
  int dry_run;        /* don't change any persistent data */
  int batch;          /* batch mode */
  const char *homedir;/* configuration directory name */

  int force;          /* Force loading outdated CRLs. */

  int max_replies;
  struct timeval ldaptimeout;
  
  ldap_server_t ldapservers;
  int add_new_ldapservers;

  strlist_t ocsp_responders; /* List of OCSP responder URLs. */

} opt;


#define DBG_X509_VALUE    1	/* debug x.509 parsing */
#define DBG_LOOKUP_VALUE  2	/* debug lookup details */
#define DBG_CRYPTO_VALUE  4	/* debug low level crypto */
#define DBG_MEMORY_VALUE  32	/* debug memory allocation stuff */
#define DBG_CACHE_VALUE   64	/* debug the caching */
#define DBG_MEMSTAT_VALUE 128	/* show memory statistics */
#define DBG_HASHING_VALUE 512	/* debug hashing operations */
#define DBG_ASSUAN_VALUE  1024  /* debug assuan communication */

#define DBG_X509    (opt.debug & DBG_X509_VALUE)
#define DBG_LOOKUP  (opt.debug & DBG_LOOKUP_VALUE)
#define DBG_CRYPTO  (opt.debug & DBG_CRYPTO_VALUE)
#define DBG_MEMORY  (opt.debug & DBG_MEMORY_VALUE)
#define DBG_CACHE   (opt.debug & DBG_CACHE_VALUE)
#define DBG_HASHING (opt.debug & DBG_HASHING_VALUE)
#define DBG_ASSUAN   (opt.debug & DBG_ASSUAN_VALUE)



struct server_local_s;

struct server_control_s {
  int no_server;     /* We are not running under server control. */
  int status_fd;     /* Only for non-server mode. */
  struct server_local_s *server_local;
};

typedef struct server_control_s *ctrl_t;


/*-- dirmngr.c --*/
void dirmngr_exit( int );  /* Wrapper for exit() */
void dirmngr_init_default_ctrl (ctrl_t ctrl);

/*-- server.c --*/
ksba_cert_t get_issuer_cert_local (ctrl_t ctrl, const char *issuer);
void start_command_handler (void);



#endif /*DIRMNGR_H*/


