/* ldap.c - LDAP access
 *      Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *      Copyright (C) 2003 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ldap.h>
#include <errno.h>

#include "crlfetch.h"
#include "dirmngr.h"


#define CERTATTR "userCertificate"
#define CERTATTRBIN CERTATTR ";binary"
#define CACERTATTR "caCertificate"
#define CACERTATTRBIN CACERTATTR ";binary"

struct cert_fetch_context_s {
  LDAP *ld;
  strlist_t query_list; /* List of all prepared queries. */
  strlist_t query;      /* Pointer into this list. */
  LDAPMessage *result;
  int bercount;
  struct berval **bervalues;
  char *base;
};


/* Fetch the value attr, put result in value, length in valuelen.
   Returns 0 on success or an gpg error code.
 */
static gpg_error_t
get_attr_from_result_ldap (LDAP * ld, LDAPMessage * msg, const char *attr,
			   unsigned char **value, size_t * valuelen)
{
  LDAPMessage *item;
  struct berval **values;

  item = ldap_first_entry (ld, msg);
  if (!item)
    return gpg_error (GPG_ERR_NO_CRL_KNOWN);

  if (DBG_LOOKUP)
     log_info ("get_attr_from_result_ldap `%s'\n", attr);

  values = ldap_get_values_len (ld, item, attr);

  if (values && opt.verbose)
    log_info ("found attribute `%s'\n", attr);

  if (!values)
    {
      /* Try again without or with ";binary" .. which worked with some
         strange LDAP servers.  */
      const char *s;
      char *newattr;

      s = memistr (attr, strlen (attr), ";binary");
      if (s)
        { 
          newattr = xstrdup (attr);
          newattr[s - newattr] = 0;
        }
      else
        {
          newattr = xmalloc (strlen (attr) + 7 + 1);
          strcpy (stpcpy (newattr, attr), ";binary");
        }

      values = ldap_get_values_len (ld, item, newattr);
      if (values && opt.verbose)
	    log_info ("found attributes `%s' at second try\n", newattr);
      xfree (newattr);
    }

  if (values)
    {
      *valuelen = values[0]->bv_len;
      *value = xmalloc (*valuelen * sizeof (char));
      memcpy (*value, values[0]->bv_val, *valuelen);
      ldap_value_free_len (values);
      return 0;
    }
  else
    return gpg_error (GPG_ERR_NO_CRL_KNOWN);
}


/* Perform an LDAP query.  Returns an gpg error code or 0 on success.
   The values are returned in VALUE and VALUELEN. */
static gpg_error_t
attr_fetch_ldap_internal (const char *host, int port,
			  const char *user, const char *pass,
			  const char *dn, const char *attr,
			  unsigned char **value, size_t * valuelen)
{
  gpg_error_t err;
  char *attrs[2];
  int rc;
  int msgid;
  LDAPMessage *result;
  LDAP *ld;

  attrs[0] = xstrdup (attr); 
  attrs[1] = NULL;

  if (DBG_LOOKUP)
    log_debug ("fetching attribute `%s' from '%s:%d' for DN '%s'\n",
               attr, host, port, dn);

  ld = ldap_init (host, port);
  if (!ld)
    {
      log_error ("ldap_init failed for '%s:%d': %s\n",
                 host, port, strerror (errno));
      /* We do not return the system error because the host might
         simply not be available and the user is not intersted inthese
         details. */
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }
  msgid = ldap_simple_bind (ld, user, pass);
  if (msgid == -1)
    {
      log_error ("ldap_simple_bind failed for user '%s': %s\n",
                 user? user : "[none]", strerror (errno));
      /* According to the documentation a variable ld->ld_errno will
         be set.  However, that structure is private and thus we can't
         access.  So we assume that the standard errno is used. */
      xfree (attrs[0]);
      ldap_unbind (ld);
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }

  rc = ldap_result (ld, msgid, 0, &opt.ldaptimeout, &result);
  if (rc != LDAP_RES_BIND)
    {
      if (rc == -1)
        log_error ("ldap_result for bind failed: %s\n", strerror (errno));
      else if (!rc)
        log_error ("timeout while waiting for ldap_bind to complete\n");
      else
        {
          log_error ("unexpected message type %d returned for ldap_bind\n",
                     rc);
          ldap_msgfree (result);
        }
      xfree (attrs[0]);
      ldap_unbind (ld);
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }
  ldap_msgfree (result);
  result = NULL;

  rc = ldap_search_st (ld, dn, LDAP_SCOPE_BASE, "objectclass=*", attrs, 0,
                       &opt.ldaptimeout, &result);
  if (rc)
    {
      log_error ("ldap_search failed: %s\n", ldap_err2string (rc));
      xfree (attrs[0]);
      ldap_unbind (ld);
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }

  err = get_attr_from_result_ldap (ld, result, attr, value, valuelen);

  ldap_msgfree (result);
  xfree (attrs[0]);
  return err;
}


/* Perform an LDAP query on all configured servers.  On error the
   error code of the last try is retruned.  */
gpg_error_t
attr_fetch_ldap (const char *dn, const char *attr,
                 unsigned char **value, size_t * valuelen)
{
  struct ldap_server_s *server;
  gpg_error_t err = gpg_error (GPG_ERR_CONFIGURATION);

  for (server = opt.ldapservers; server; server = server->next)
    {
      err = attr_fetch_ldap_internal (server->host, server->port,
				     server->user, server->pass,
				     dn, attr, value, valuelen);
      if (!err)
        break; /* Found a result. Ready. */
    }
  return err;
}


/* Helper for the URL based LDAP query. */
static gpg_error_t
url_fetch_ldap_internal (const LDAPURLDesc *ludp, const char *attr,
			 unsigned char **value, size_t * valuelen)
{
  gpg_error_t err;
  LDAPMessage *result;
  LDAP *ld;
  int rc;

  ld = ldap_init (ludp->lud_host, ludp->lud_port);
  if (!ld)
    {
      log_error ("ldap_init failed for '");
      print_sanitized_string (log_get_stream (), ludp->lud_host, 0);
      log_printf (":%d': %s\n", ludp->lud_port, strerror (errno));
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }
  if (ldap_simple_bind_s (ld, 0, 0))
    {
      log_error ("ldap_simple_bind failed: %s\n", strerror (errno));
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }

  rc = ldap_search_st (ld, ludp->lud_dn, ludp->lud_scope,
                       ludp->lud_filter, ludp->lud_attrs, 0,
                       &opt.ldaptimeout, &result);
  if (rc)
    {
      log_error ("ldap_search failed: %s\n", ldap_err2string (rc));
      return gpg_error (GPG_ERR_NO_CRL_KNOWN);
    }

  err = get_attr_from_result_ldap (ld, result,
                                   ludp->lud_attrs? ludp->lud_attrs[0] : attr,
                                   value, valuelen);
  ldap_msgfree (result);
  return err;
}


/* Add HOST and PORT to our list of LDAP servers.  Fixme: We should
   better use an extra list of servers. */
static void
add_server_to_servers (const char *host, int port)
{
  ldap_server_t server;
  ldap_server_t last = NULL;
  const char *s;

  for (server=opt.ldapservers; server; server = server->next)
    {
      if (!strcmp (server->host, host) && server->port == port)
	  return; /* already in list... */
      last = server;
    }

  /* We assume that the host names are all supplied by our
     configuration files and thus are sane.  To keep this assumption
     we must reject all invalid host names. */
  for (s=host; *s; s++)
    if (!strchr ("abcdefghijklmnopqrstuvwxyz"
                 "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                 "01234567890.-", *s))
      {
        log_error ("invalid char 0x%02x in host name"
                   " - not added", *s);
        return;
      }

  log_info ("adding `%s:%d' to the ldap server list\n", host, port);
  server = xcalloc (1, sizeof *s);
  server->host = xstrdup (host);
  server->port = port;
  if (last)
    last->next = server;
  else
    opt.ldapservers = server;
}


/* Perform a LDAP query using a given URL. */
gpg_error_t
url_fetch_ldap (const char *url, const char *attr, unsigned char **value,
		size_t *valuelen)
{
  gpg_error_t err;
  LDAPURLDesc *ludp = NULL;

  *value = NULL;
  *valuelen = 0;
  if (DBG_LOOKUP)
    {
      log_debug ("url_fetch_ldap: url=`");
      print_sanitized_string (log_get_stream (), url, 0);
      log_printf ("' attr=`%s'\n", attr);
    }

  if (!ldap_is_ldap_url (url))
    {
      log_error ("'");
      print_sanitized_string (log_get_stream (), url, 0);
      log_printf ("' is not an LDAP URL\n");
      return gpg_error (GPG_ERR_INV_URI);
    }

  if (ldap_url_parse (url, &ludp))
    {
      log_error ("'");
      print_sanitized_string (log_get_stream (), url, 0);
      log_printf ("' is an invalid LDAP URL\n");
      return gpg_error (GPG_ERR_BAD_URI);
    }

  if (ludp->lud_host)
    {
      /* URL has host part: try a simple query. */
      if (opt.add_new_ldapservers)
	add_server_to_servers (ludp->lud_host, ludp->lud_port);
      err = url_fetch_ldap_internal (ludp, attr, value, valuelen);
    }
  else
    err = gpg_error (GPG_ERR_NO_CRL_KNOWN);


  if (gpg_err_code (err) == GPG_ERR_NO_CRL_KNOWN)
    {
      /* CRL not found so far. */
      ldap_server_t server;
      char *savehost;
      int saveport;
      
      if (DBG_LOOKUP)
        log_debug ("no hostname in URL or query failed; "
                   "trying default hostnames\n");
      savehost = ludp->lud_host;
      saveport = ludp->lud_port;
      for (server = opt.ldapservers; server; server = server->next)
	{
	  ludp->lud_host = server->host;
	  ludp->lud_port = server->port;
	  err = url_fetch_ldap_internal (ludp, attr, value, valuelen);
	  if (!err)
	    break;
	}
      ludp->lud_host = savehost;
      ludp->lud_port = saveport;
    }

  ldap_free_urldesc (ludp);
  return err;
}





/* Parse PATTERN and return a new strlist to be used for the actual
   LDAP query.  Bit 0 of the flags field is set if that pattern is
   actually a base specification.  Caller must release the returned
   strlist.  NULL is returned on error.

 * Possible patterns:
 *
 *   KeyID
 *   Fingerprint
 *   OpenPGP userid
 * x Email address  Indicated by a left angle bracket.
 *   Exact word match in user id or subj. name
 * x Subj. DN  indicated bu a leading slash
 *   Issuer DN
 *   Serial number + subj. DN
 * x Substring match indicated by a leading '*; is also the default.
 */

strlist_t
parse_one_pattern (const char *pattern)
{
  strlist_t result = NULL;
  char *p;

  switch (*pattern)
    {
    case '<':			/* Email. */
      {
        pattern++;
	result = xmalloc (sizeof *result + 5 + strlen (pattern));
        result->next = NULL;
        result->flags = 0;
	p = stpcpy (stpcpy (result->d, "mail="), pattern);
	if (p[-1] == '>')
	  *--p = 0;
        if (!*result->d) /* Error. */
          {
            xfree (result);
            result = NULL;
          }
	break;
      }
    case '/':			/* Subject DN. */
      pattern++;
      if (*pattern)
        {
          result = xmalloc (sizeof *result + strlen (pattern));
          result->next = NULL;
          result->flags = 1; /* Base spec. */
          strcpy (result->d, pattern);
        }
      break;
    case '#':			/* Issuer DN. */
      pattern++;
      if (*pattern == '/')  /* Just issuer DN. */
        {
          pattern++;
	}
      else  /* Serial number + issuer DN */
	{
        }
      break;
    case '*':
      pattern++;
    default:			/* Take as substring match. */
      {
	const char format[] = "(|(sn=*%s*)(|(cn=*%s*)(mail=*%s*)))";
        
        if (*pattern)
          {
            result = xmalloc (sizeof *result
                              + strlen (format) + 3 * strlen (pattern));
            result->next = NULL;
            result->flags = 0; 
            sprintf (result->d, format, pattern, pattern, pattern);
          }
      }
      break;
    }
  
  return result;
}


/* Prepare an LDAP query to return certificates maching PATTERNS using
   the SERVER.  This function retruns an error code or 0 and a CONTEXT
   on success. */
gpg_error_t
start_cert_fetch_ldap (cert_fetch_context_t *context,
                       strlist_t patterns, const ldap_server_t server)
{
  gpg_error_t err;
  int rc;
  const char *host;
  int port;
  const char *user;
  const char *pass;
  const char *base;
  LDAP *ld;
  int msgid;
  LDAPMessage *result;
  strlist_t query_list = NULL, *query_list_tail, sl;
  
  *context = NULL;
  if (server)
    {
      host = server->host;
      port = server->port;
      user = server->user;
      pass = server->pass;
      base = server->base;
    }
  else /* Use a default server. */
    return gpg_error (GPG_ERR_NOT_IMPLEMENTED);

  if (!base)
    base = "";

  if (DBG_LOOKUP)
    {
      log_debug ("start querying LDAP host=`%s:%d' base='", host, port);
      print_sanitized_string (log_get_stream (), base, 0);
      log_printf ("'\n");
    }

  query_list_tail = &query_list;
  for (; patterns; patterns = patterns->next)
    {
      sl = parse_one_pattern (patterns->d);
      if (!sl)
        {
          log_error ("start_cert_fetch: invalid pattern `%s'\n", patterns->d);
          free_strlist (query_list);
          return gpg_error (GPG_ERR_INV_USER_ID);
        }
      *query_list_tail = sl;
      query_list_tail = &sl;
    }


  ld = ldap_init (host, port);
  if (!ld)
    {
      err = gpg_error_from_errno (errno);
      log_error ("ldap_init failed for '%s:%d': %s\n",
                 host, port, strerror (errno));
      free_strlist (query_list);
      return err;
    }

  msgid = ldap_simple_bind (ld, user, pass);
  if (msgid == -1)
    {
      err = gpg_error_from_errno (errno);
      log_error ("ldap_simple_bind failed for user '%s': %s\n",
                 user? user : "[none]", strerror (errno));
      ldap_unbind (ld);
      free_strlist (query_list);
      return err;
    }

  rc = ldap_result (ld, msgid, 0, &opt.ldaptimeout, &result);
  if (rc != LDAP_RES_BIND)
    {
      if (rc == -1)
        {
          err = gpg_error_from_errno (errno);
          log_error ("ldap_result for bind failed: %s\n", strerror (errno));
        }
      else if (!rc)
        {
          err = gpg_error (GPG_ERR_TIMEOUT);
          log_error ("timeout while waiting for ldap_bind to complete\n");
        }
      else
        {
          log_error ("unexpected message type %d returned for ldap_bind\n", rc);
          ldap_msgfree (result);
          err = gpg_error (GPG_ERR_BUG);
        }
      ldap_unbind (ld);
      free_strlist (query_list);
      return err;
    }
  ldap_msgfree (result);

  *context = xcalloc (1, sizeof **context);
  (*context)->ld = ld;
  (*context)->query_list = query_list;
  (*context)->base = xstrdup (base);

  return 0;
}

/* Fixme: This is a strange helper.  It does for example never return an
   error code but instead sets a last value to whatever semantics LDAP
   has. */
static int
get_cert_ldap (cert_fetch_context_t context,
               unsigned char **value, size_t * valuelen, int *last)
{
  int try_ca = 0;
  char *dn;

  *last = 0;
  if (DBG_LOOKUP)
    {
      dn = ldap_get_dn (context->ld, context->result);
      log_debug ("get_cert_ldap: examining DN=`");
      print_sanitized_string (log_get_stream (), dn?dn:"[none]", 0);
      log_printf ("'\n");
    }

 again:
  if (!context->bervalues)
    {
      context->bercount = 0;
      context->bervalues = ldap_get_values_len (context->ld,
                                                context->result, 
                                                try_ca? CACERTATTR
                                                : CERTATTR);
      if (!context->bervalues)
	context->bervalues = ldap_get_values_len (context->ld,
                                                  context->result,
                                                  try_ca? CACERTATTRBIN
                                                        : CERTATTRBIN);
    }

  if (context->bervalues)
    {
      if (DBG_LOOKUP)
        log_debug ("get_cert_ldap: got a %sCertificate\n", try_ca? "ca":"user");

      *valuelen = context->bervalues[context->bercount]->bv_len;
      *value = xmalloc (*valuelen);
      memcpy (*value, context->bervalues[context->bercount]->bv_val, *valuelen);
      context->bercount++;
      if (!context->bervalues[context->bercount])
	{
	  ldap_value_free_len (context->bervalues);
	  context->bervalues = 0;
	  *last = 1;
	}
    }
  else if (!try_ca)
    {
      try_ca = 1;
      goto again;
    }
  else
    {
      *last = 1;
    }

  return 0;
}


/* Fetch the next certificate. Return 0 on success, GPG_ERR_EOF
   when no (more) certifciates are available or any other error
   code. */
gpg_error_t
fetch_next_cert_ldap (cert_fetch_context_t context,
                      unsigned char **value, size_t *valuelen)
{
  gpg_error_t err;
  int last;
  int rc;
  int truncated = 0;
  char *attrs[5];

  attrs[0] = CERTATTR;
  attrs[1] = CERTATTRBIN;
  attrs[2] = CACERTATTR;  
  attrs[3] = CACERTATTRBIN;
  attrs[4] = NULL;


 next:
  *value = NULL;
  *valuelen = 0;

  if (!context->result)
    {
      context->query = context->query? context->query->next
                                     : context->query_list;
      if (!context->query)
        return gpg_error (GPG_ERR_EOF); /* No more patterns. */

      if ((context->query->flags & 1))
        {
          if (DBG_LOOKUP)
            log_debug ("ldap_search_st using base `%s'\n", context->query->d);
          rc = ldap_search_st (context->ld, context->query->d,
                               LDAP_SCOPE_SUBTREE,
                               "objectClass=*", attrs, 0,
                               &opt.ldaptimeout, &context->result);
        }
      else
        {
          if (DBG_LOOKUP)
            log_debug ("ldap_search_st using filter `%s', base `%s'\n",
			context->query->d, context->base);
          rc = ldap_search_st (context->ld, context->base,
                               LDAP_SCOPE_SUBTREE,
                               context->query->d, attrs, 0,
                               &opt.ldaptimeout, &context->result);
        }

      if (rc)
        {
          if (rc == LDAP_SIZELIMIT_EXCEEDED)
            {
              truncated = 1;
              log_info ("ldap_search hit the size limit of the server\n");
              /* Fixme: Shouldn't we bail out here? */
            }
          else if (rc == LDAP_NO_SUCH_OBJECT)
            {
              /* Ignore this error. */
              if (DBG_LOOKUP)
                log_debug ("ldap_search returned with no object\n");
              /* Fixme: Shouldn't we bail out here? */
            }
          else
            {
              log_error ("ldap_search failed: %s\n",
                         ldap_err2string (ldap_result2error
                                          (context->ld,
                                           context->result, 0)));
              /* FIXME we should write a ldap to gpg error mapper but
                 before we can do that we need to dig deeper into the
                 LDAP API definition. */
              return gpg_error (GPG_ERR_GENERAL);
            }
        }
      context->result = ldap_first_entry (context->ld, context->result);
    }


  if (!context->result)
    goto next;

  err = get_cert_ldap (context, value, valuelen, &last);
  if (err)
    return err;

  if (last)
    context->result = ldap_next_entry (context->ld, context->result);

  if (!*value)
    goto next;

  return truncated? gpg_error (GPG_ERR_TRUNCATED) : 0;
}

void
end_cert_fetch_ldap (cert_fetch_context_t context)
{
  if (context)
    {
      free_strlist (context->query_list);
      xfree (context->base);
      ldap_unbind (context->ld);
      xfree (context);

      /* fixme: do we need to release the BERVALUES. */
    }
}
