/* misc.c - miscellaneous
 *	Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *      Copyright (C) 2002, 2003 Free Software Foundation, Inc.
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "misc.h"
#include "dirmngr.h"

static unsigned long timewarp;
static enum { NORMAL = 0, FROZEN, FUTURE, PAST } timemode;

/* Wrapper for the time(3).  We use this here so we can fake the time
   for tests. */
time_t 
get_time () 
{
  time_t current = time (NULL);
  if (timemode == NORMAL)
    return current;
  else if (timemode == FROZEN)
    return timewarp;
  else if (timemode == FUTURE)
    return current + timewarp;
  else
    return current - timewarp;
}



/* Return the current time in ISO format. */
void
get_isotime (dirmngr_isotime_t timebuf)
{
  time_t atime = get_time ();
    
  if (atime < 0)
    *timebuf = 0;
  else 
    {
      struct tm *tp;
#ifdef HAVE_GMTIME_R
      struct tm tmbuf;
      
      tp = gmtime_r (&atime, &tmbuf);
#else
      tp = gmtime (&atime);
#endif
      sprintf (timebuf,"%04d%02d%02dT%02d%02d%02d",
               1900 + tp->tm_year, tp->tm_mon+1, tp->tm_mday,
               tp->tm_hour, tp->tm_min, tp->tm_sec);
    }
}


/* Set the time to NEWTIME so that get_time returns a time starting
   with this one.  With FREEZE set to 1 the returned time will never
   change.  Just for completeness, a value of (time_t)-1 for NEWTIME
   gets you back to reality. Note that this is obviously not
   thread-safe but this is not required. */
void
set_time (time_t newtime, int freeze)
{
  time_t current = time (NULL);

  if ( newtime == (time_t)-1 || current == newtime)
    {
      timemode = NORMAL;
      timewarp = 0;
    }
  else if (freeze)
    {
      timemode = FROZEN;
      timewarp = current;
    }
  else if (newtime > current)
    {
      timemode = FUTURE;
      timewarp = newtime - current;
    }
  else
    {
      timemode = PAST;
      timewarp = current - newtime;
    }
}

/* Returns true when we are in timewarp mode. */
int
faked_time_p (void)
{
  return timemode;
}




/* Check that the 15 bytes in ATIME represent a valid ISO time.  Note
   that this function does not expect a string but a plain 15 byte
   isotime buffer. */
gpg_error_t
check_isotime (const dirmngr_isotime_t atime)
{
  int i;
  const char *s;

  if (!*atime)
    return gpg_error (GPG_ERR_NO_VALUE);
  
  for (s=atime, i=0; i < 8; i++, s++)
    if (!digitp (s))
      return gpg_error (GPG_ERR_INV_TIME);
  if (*s != 'T')
      return gpg_error (GPG_ERR_INV_TIME);
  for (s++, i=9; i < 15; i++, s++)
    if (!digitp (s))
      return gpg_error (GPG_ERR_INV_TIME);
  return 0;
}




/* Convert the hex encoded STRING back into binary and store the
   result into the provided buffer RESULT.  The actual size of that
   buffer will be returned.  The caller should provide RESULT of at
   least strlen(STRING)/2 bytes.  The is no error detection, the
   parsing simply stops at the firt non hex character. */
size_t
unhexify (unsigned char *result, const char *string)
{
  const char *s;
  size_t n;

  for (s=string,n=0; hexdigitp (s) && hexdigitp(s+1); s += 2)
    result[n++] = xtoi_2 (s);
  return n;
}


char* 
hashify_data( const char* data, size_t len )
{
  unsigned char buf[20];
  gcry_md_hash_buffer (GCRY_MD_SHA1, buf, data, len);
  return hexify_data( buf, 20 );
}

char* 
hexify_data( const unsigned char* data, size_t len )
{
  int i;
  char* result = xmalloc( sizeof( char ) * (2*len+1));

  for( i = 0; i < 2*len; i+=2 )
    sprintf( result+i, "%02X", *data++);
  return result;
}

char *
serial_hex (ksba_sexp_t serial )
{
  unsigned char* p = serial;
  char *endp;
  unsigned long n;
  char *certid;

  if (!p)
    return NULL;
  else {
    p++; /* ignore initial '(' */
    n = strtoul (p, (char**)&endp, 10);
    p = endp;
    if (*p!=':')
      return NULL;
    else {
      int i = 0;
      certid = xmalloc( sizeof( char )*(2*n + 1 ) );
      for (p++; n; n--, p++) {
	sprintf ( certid+i , "%02X", *p);
	i += 2;
      }
    }
  }
  return certid;
}


/* Take an S-Expression encoded blob and return a pointer to the
   actual data as well as its length.  Return NULL for an invalid
   S-Expression.*/
const unsigned char *
serial_to_buffer (const ksba_sexp_t serial, size_t *length)
{
  unsigned char *p = serial;
  char *endp;
  unsigned long n;

  if (!p || *p != '(')
    return NULL;
  p++;
  n = strtoul (p, &endp, 10);
  p = endp;
  if (*p != ':')
    return NULL;
  p++;
  *length = n;
  return p;
}


/* Do an in-place percent unescaping of STRING. Returns STRING. Noet
   that this function does not do a '+'-to-space unescaping.*/
char *
unpercent_string (char *string)
{
  char *s = string;
  char *d = string;

  while (*s)
    {
      if (*s == '%' && s[1] && s[2])
        { 
          s++;
          *d++ = xtoi_2 ( s);
          s += 2;
        }
      else
        *d++ = *s++;
    }
  *d = 0; 
  return string;
}


