/* ocsp.c - OCSP management
 *      Copyright (C) 2004 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "dirmngr.h"
#include "misc.h"
#include "ocsp.h"

#if 0 /* Code not yet ready. */
/* The maximum size we allow as a response from an OCSP reponder. */
#define MAX_RESPONSE_SIZE 65536


/* Read from FP and return a newly allocated buffer in R_BUFFER with the
   entire data read from FP. */
gpg_error_t
read_response (FILE *fp, unsigned char **r_buffer, size_t *r_buflen)
{
  unsigned char *buffer;
  size_t bufsize, nbytes;

  *r_buffer = NULL;
  *r_buflen = 0;

  bufsize = 4096;
  buffer = xtrymalloc (bufsize);
  if (!buffer)
    return gpg_error_from_errno (errno);

  nbytes = 0;
  for (;;)
    {
      unsigned char *tmp;
      size_t nread = 0;

      assert (nbytes < bufsize);
      nread = fread (buffer+nbytes, 1, bufsize-nbytes, fp);
      if (nread < bufsize-nbytes && ferror (fp))
        {
          err = gpg_error_from_errno (errno);
          log_error ("error reading from responder: %s\n", strerror (errno));
          xfree (buffer);
          return err;
        }
      if ( !(nread == bufsize-nbytes && !feof (fp)))
        { /* Response succesfully received. */
          nbytes += nread;
          *r_buffer = buffer;
          *r_buflen = nbytes;
          return 0;
        }

      nbytes += nread;

      /* Need to enlarge the buffer. */
      if (bufsize >= MAX_RESPONSE_SIZE)
        {
          log_error ("response from server too large; limit is %d bytes\n",
                     MAX_RESPONSE_SIZE);
          xfree (buffer);
          return gpg_error (GPG_ERR_TOO_LARGE);
        }

      bufsize += 4096;
      tmp = xtryrealloc (buffer, bufsize);
      if (!tmp)
        {
          err = gpg_error_from_errno (errno);
          xfree (buffer);
          return err;
        }
      buffer = tmp;
    }
}


/* Construct an OCSP request, send it to the configured OCSP responder
   and parse the response. On success the OCSP context may be used to
   further process the reponse. */
gpg_error_t
do_ocsp_request (ctrl_t ctrl, ksba_ocsp_t ocsp,
                 ksba_cert_t cert, ksba_cert_t issuer_cert, FILE *r_fp)
{
  gpg_error_t err;
  unsigned char *request, *response;
  size_t requestlen, responselen;
  struct http_context_s http;
  ksba_ocsp_response_status_t response_status;
      
  err = ksba_ocsp_add_target (ocsp, cert, issuer_cert);
  if (err)
    {
      log_error ("error setting OCSP target: %s\n", gpg_strerror (err));
      return err;
    }

  err = ksba_ocsp_build_request (ocsp, &request, &requestlen);
  if (err)
    {
      log_error ("error building OCSP request: %s\n", gpg_strerror (err));
      return err;
    }

  err = http_open (&http, HTTP_REQ_POST, url, 0);
  if (err)
    {
      log_error ("error connecting to `%s': %s\n", url, gpg_strerror (err));
      return err;
    }

  fprintf (http.fp_write,
           "Content-Type: application/ocsp-request\r\n"
           "Content-Length: %ul\r\n", 
           (unsigned long)requestlen );
  http_start_data (&http);
  if (fwrite (request, requestlen, 1, http.fp_write) != 1)
    {
      err = gpg_error_from_errno (errno);
      log_error ("error sending request to `%s': %s\n", url, strerror (errno));
      http_close (&http, 0);
      xfree (request);
      return err;
    }
  xfree (request);
  request = NULL;

  err = http_wait_response (&http, NULL);
  if (err || http.status_code != 200)
    {
      if (err)
        log_error ("error reading HTTP response for `%s': %s\n",
                   url, gpg_strerror (err));
      else
        {
          log_error ("error accessing `%s': http status %u\n",
                     url, hd.status_code);
          err = gpg_error (GPG_ERR_NO_DATA);
        }
      http_close (&http, 0);
      return err
    }

  err = read_response (http.fp_read, &response, &responselen);
  http_close (&http, 0);
  if (err)
    {
      log_error ("error reading HTTP response for `%s': %s\n",
                 url, gpg_strerror (err));
      return err;
    }

  err = ksba_ocsp_parse_response (ocsp, response, responselen,
                                  &response_status);
  xfree (response);
  response = NULL;
  if (err)
    {
      log_error ("error parsing OCSP response for `%s': %s\n",
                 url, gpg_strerror (err));
      return err;
    }

  switch (response_status)
    {
    case KSBA_OCSP_RSPSTATUS_SUCCESS:      t = "success"; break;
    case KSBA_OCSP_RSPSTATUS_MALFORMED:    t = "malformed"; break;  
    case KSBA_OCSP_RSPSTATUS_INTERNAL:     t = "internal error"; break;  
    case KSBA_OCSP_RSPSTATUS_TRYLATER:     t = "try later"; break;      
    case KSBA_OCSP_RSPSTATUS_SIGREQUIRED:  t = "must sign request"; break;  
    case KSBA_OCSP_RSPSTATUS_UNAUTHORIZED: t = "unauthorized"; break;  
    case KSBA_OCSP_RSPSTATUS_REPLAYED:     t = "replay detected"; break;  
    case KSBA_OCSP_RSPSTATUS_OTHER:        t = "other (unknown)"; break;  
    case KSBA_OCSP_RSPSTATUS_NONE:         t = "no status"; break;
    default:                               t = "[unknown status]"; break;
    }
  if (response_status == KSBA_OCSP_RSPSTATUS_SUCCESS)
    {
      if (opt.verbose)
        log_info ("ocsp_responder at `%s' status: %s\n", t);
    }
  else
    {
      log_error ("ocsp responder at `%s' status: %s\n", t);
      err = gpg_error (GPG_ERR_GENERAL);
    }

  return err;
}
#endif 


gpg_error_t
ocsp_isvalid (ctrl_t ctrl, const char *cert_fpr)
{
  gpg_error_t err;
  ksba_ocsp_t ocsp = NULL;
  ksba_cert_t cert = NULL;
  ksba_cert_t issuer_cert = NULL;

  if (!opt.ocsp_responders)
    {
      log_error ("no OCSP responder defined\n");
      err = gpg_error (GPG_ERR_CONFIGURATION);
      goto leave;
    }

  err = ksba_ocsp_new (&ocsp);
  if (err)
    {
      log_error ("failed to allocate OCSP context: %s\n", gpg_strerror (err));
      goto leave;
    }

  /* We don't have any caching yet implemented thus we need to get the
     certificate right now. */
  cert = get_issuer_cert_local (ctrl, cert_fpr);
  if (!cert)
    {
      log_error ("caller did not return the certificate as expected\n");
      err = gpg_error (GPG_ERR_GENERAL);
      goto leave;
    }

   /* Hack for testing: */
  issuer_cert = cert;
  ksba_cert_ref (cert);

  
  err = gpg_error (GPG_ERR_NOT_IMPLEMENTED); 
       /*do_ocsp_request (ctrl, ocsp, cert, issuer_cert);*/
  if (err)
    goto leave;



 leave:
  ksba_cert_release (issuer_cert);
  ksba_cert_release (cert);
  ksba_ocsp_release (ocsp);
  return err;
}


