/* dirmngr.c - LDAP access
 *	Copyright (C) 2002 Klarlvdalens Datakonsult AB
 *      Copyright (C) 2003 g10 Code GmbH
 *
 * This file is part of DirMngr.
 *
 * DirMngr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DirMngr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <assuan.h> /* needed for the malloc hooks */


#define JNLIB_NEED_LOG_LOGV
#include "crlcache.h"
#include "crlfetch.h"
#include "dirmngr.h"
#include "ocsp.h"

/* no i18n needed */
#define N_(a) a
#define _(a) a


#define PARM_ERROR(t) assuan_set_error (ctx, ASSUAN_Parameter_Error, (t))


/* Data used to associate an Assuan context with local server data */
struct server_local_s {
  ASSUAN_CONTEXT assuan_ctx;
  crl_cache_t     crl_cache;
  int message_fd;
};


/* Map Assuan error code ERR to an GPG_ERR_ code.  We need to
   distinguish between genuine (and legacy) Assuan error codes and
   application error codes shared with all GnuPG modules.  The rule is
   simple: All errors with a gpg_err_source of UNKNOWN are genuine;
   all other Assuan codes are passed verbatim through. */
static gpg_error_t
map_assuan_err (int err)
{
  gpg_err_code_t ec;

  if (gpg_err_source (err))
    return err;

  switch (err)
    {
    case -1:                     ec = GPG_ERR_EOF; break;
    case 0:                      ec = 0; break;

    case ASSUAN_Canceled:        ec = GPG_ERR_CANCELED; break;
    case ASSUAN_Invalid_Index:   ec = GPG_ERR_INV_INDEX; break;

    case ASSUAN_Not_Implemented: ec = GPG_ERR_NOT_IMPLEMENTED; break;
    case ASSUAN_Server_Fault:    ec = GPG_ERR_ASSUAN_SERVER_FAULT; break;
    case ASSUAN_No_Public_Key:   ec = GPG_ERR_NO_PUBKEY; break;
    case ASSUAN_No_Secret_Key:   ec = GPG_ERR_NO_SECKEY; break;

    case ASSUAN_Cert_Revoked:    ec = GPG_ERR_CERT_REVOKED; break;
    case ASSUAN_No_CRL_For_Cert: ec = GPG_ERR_NO_CRL_KNOWN; break;       
    case ASSUAN_CRL_Too_Old:     ec = GPG_ERR_CRL_TOO_OLD; break;        

    case ASSUAN_Not_Trusted:     ec = GPG_ERR_NOT_TRUSTED; break;

    case ASSUAN_Card_Error:      ec = GPG_ERR_CARD; break;
    case ASSUAN_Invalid_Card:    ec = GPG_ERR_INV_CARD; break;
    case ASSUAN_No_PKCS15_App:   ec = GPG_ERR_NO_PKCS15_APP; break;
    case ASSUAN_Card_Not_Present: ec= GPG_ERR_CARD_NOT_PRESENT; break;
    case ASSUAN_Not_Confirmed:   ec = GPG_ERR_NOT_CONFIRMED; break;
    case ASSUAN_Invalid_Id:      ec = GPG_ERR_INV_ID; break;

    default:
      ec = err < 100? GPG_ERR_ASSUAN_SERVER_FAULT : GPG_ERR_ASSUAN;
      break;
    }
  return gpg_err_make (GPG_ERR_SOURCE_UNKNOWN, ec);
}

/* Map GPG_xERR_xx error codes to Assuan status codes */
static int
map_to_assuan_status (int rc)
{
  gpg_err_code_t   ec = gpg_err_code (rc);
  gpg_err_source_t es = gpg_err_source (rc);

  if (!rc)
    return 0;
  if (!es)
    {
      es = GPG_ERR_SOURCE_USER_4; /*  This should not happen, but we
                                      need to make sure to pass a new
                                      Assuan error code along. */
      log_debug ("map_to_assuan_status called with no error source\n");
    }

  if (ec == -1)
    ec = GPG_ERR_NO_DATA;  /* That used to be ASSUAN_No_Data_Available. */

  return gpg_err_make (es, ec);
}



/* Copy the % and + escaped string S into the buffer D and replace the
   escape sequences.  Note, that it is sufficient to allocate the
   target string D as long as the source string S, i.e.: strlen(s)+1.
   NOte further that If S contains an escaped binary nul the resulting
   string D will contain the 0 as well as all other characters but it
   will be impossible to kno whether this is the original eos or a
   copied nul. */
static void
strcpy_escaped_plus (char *d, const unsigned char *s)
{
  while (*s)
    {
      if (*s == '%' && s[1] && s[2])
        {
          s++;
          *d++ = xtoi_2 ( s);
          s += 2;
        }
      else if (*s == '+')
        *d++ = ' ', s++;
      else
        *d++ = *s++;
    }
  *d = 0;
}



/* Ask back to return a certificate for ISSUER, given as a regular
   gpgsm certifciate indentidicates (e.g. fingerprint or one of the
   other methods).  Either return the certificate in a KSBA object or
   NULL if it is not available. 

   Note: get_issuer_cert_loal is a misnomer as the function allows us
   to get back any certificate.
*/
ksba_cert_t 
get_issuer_cert_local (ctrl_t ctrl, const char *issuer)
{
  unsigned char *value;
  size_t valuelen; 
  int rc;
  char *buf;
  ksba_cert_t issuer_cert;

  if (!ctrl || !ctrl->server_local->assuan_ctx)
    {
      log_debug ("get_issuer_cert_local called w/o context\n");
      return NULL;
    }

  buf = xmalloc ( 9 + strlen(issuer) + 1);
  strcpy (stpcpy (buf, "SENDCERT "), issuer);

  /* FIXME: When changed to run as a system wide server, we should
     setup a maxlen limit for the certificate.  The problem is that
     libassuan allocates a buffer of that size in advance which we
     usually don't need. As long as dirmngr is used in server mode we
     don't need to care about it because the user can toast his
     applications in many other ways.*/
  rc = assuan_inquire (ctrl->server_local->assuan_ctx, buf,
                       &value, &valuelen, 0);
  xfree (buf);
  if (rc)
    {
      log_error (_("assuan_inquire failed: %s\n"), assuan_strerror (rc));
      return NULL;
    }
  
  rc = ksba_cert_new (&issuer_cert);
  if (!rc)
    {
      rc = ksba_cert_init_from_mem (issuer_cert, value, valuelen);
      if (rc)
        {
          ksba_cert_release (issuer_cert);
          issuer_cert = NULL;
        }
    }
  xfree (value);
  return issuer_cert;
}


/* Ask the client to return the certificate asscociated with the
   current command. This is sometime needed because the client usually
   sends us just the cert ID, assuming that the request can be
   satisfied from the cache, where the cert ID is used as key. */

/* FIXME: We need to check the logic and better rename the function
   becuase it does more than merely get a certificate - it is actually
   the core of the the CRL retrievel. */
static int
inquire_cert (assuan_context_t ctx)
{
  ctrl_t ctrl = assuan_get_pointer (ctx);
  gpg_error_t err;
  assuan_error_t ae;
  FILE *crl_fp = NULL;
  unsigned char *value = NULL;
  size_t valuelen; 
  char *issuer = NULL;
  ksba_cert_t issuer_cert = NULL;
  int seq;
  int found_dist_point = 0;
  ksba_name_t distpoint = NULL;
  ksba_name_t issuername = NULL;
  char *distpoint_uri = NULL;
  char *issuername_uri = NULL;

  ae = assuan_inquire( ctx, "SENDCERT", &value, &valuelen, 0);
  if (ae)
    return map_assuan_err (ae);

/*   { */
/*     FILE *fp = fopen ("foo.der", "r"); */
/*     value = xmalloc (2000); */
/*     valuelen = fread (value, 1, 2000, fp); */
/*     fclose (fp); */
/*   } */

  if (!valuelen) /* No data returned; return a comprehensible error. */
    return gpg_error (GPG_ERR_MISSING_CERT);

  err = ksba_cert_new (&issuer_cert);
  if (err)
    goto leave;
  err = ksba_cert_init_from_mem (issuer_cert, value, valuelen);
  if(err)
    goto leave;

  xfree (value); 
  value = NULL;

  issuer = ksba_cert_get_issuer (issuer_cert, 0);
  if (!issuer) 
    return gpg_error (GPG_ERR_BAD_CERT); /* No issuer in certificate. */

  /* Loop over all distribution points, get the CRLs and put them
     into the cache. */
  if (DBG_X509)
    log_debug ("checking distribution points\n");
  seq = 0;
  while ( !(err = ksba_cert_get_crl_dist_point (issuer_cert, seq++,
                                                &distpoint,
                                                &issuername, NULL )))
    {
      if (!distpoint && !issuername)
        {
          if (DBG_X509)
            log_debug ("no issuername and no distpoint\n");
          break; /* Not allowed, so don't keep trying. */
        }

      /* Get the URIs. */
      xfree (distpoint_uri); distpoint_uri = NULL;
      xfree (issuername_uri); issuername_uri = NULL;
      distpoint_uri = ksba_name_get_uri (distpoint, 0); 
      issuername_uri = ksba_name_get_uri (issuername, 0); 
      ksba_name_release (distpoint); distpoint = NULL;
      ksba_name_release (issuername); issuername = NULL;
      found_dist_point = 1;
      
      xfree (value); value = NULL;
      err = crl_fetch (distpoint_uri, &crl_fp);
      if (err)
        {
          log_error ("crl_fetch via DP failed: %s\n", gpg_strerror (err));
          goto leave;
        }
      
      err = crl_cache_insert (ctrl, ctrl->server_local->crl_cache,
                              distpoint_uri, crl_fp); 
      if (err)
        {
          log_error ("crl_cache_insert via DP failed: %s\n",
                     gpg_strerror (err));
          goto leave;
        }
    }
  if (gpg_err_code (err) == GPG_ERR_EOF)
    err = 0;

  /* If we did not found any distpoint, try something reasonable. */
  if (!found_dist_point )
    {
      if (DBG_X509)
        log_debug ("no distribution point - trying issuer name\n");
      
      if (crl_fp)
        {
          fclose (crl_fp);
          crl_fp = NULL;
        }
      xfree (value); value = NULL;
      err = crl_fetch_default (issuer, &crl_fp);
      if (err)
          {
            log_error ("crl_fetch via issuer failed: %s\n", gpg_strerror (err));
            goto leave;
          }

      err = crl_cache_insert (ctrl, ctrl->server_local->crl_cache,
                             "default location(s)", crl_fp);
      if (err)
        {
          log_error ("crl_cache_insert via issuer failed: %s\n",
                     gpg_strerror (err));
          goto leave;
        }
    }

 leave:
  if (crl_fp)
    fclose (crl_fp);
  xfree (distpoint_uri);
  xfree (issuername_uri);
  ksba_name_release (distpoint); 
  ksba_name_release (issuername); 
  ksba_free (issuer);
  ksba_cert_release (issuer_cert);
  xfree (value);
  return err;
}



/* IS_VALID <certificate_id>|<certificate_fpr>
  
   This command checks whether the certificate identified by the
   certificate_id is valid.  This is done by consulting CRLs or
   whatever has been configured.  Note, that the returned error codes
   are from gpg-error.h.  The command may callback using the inquire
   function.  See the manual for details.
 
   The certificate_id is a hex encoded string consisting of two parts,
   delimited by a single dot.  The first part is the SHA-1 hash of the
   issuer name and the second part the serial number.

   Alternatively the certificate's fingerprint may be given in which
   case an OCSP request is done before consulting the CRL.
 */

static int
cmd_isvalid (assuan_context_t ctx, char *line)
{
  ctrl_t ctrl = assuan_get_pointer (ctx);
  char *issuerhash, *serialno;
  gpg_error_t err;
  int did_inquire = 0;
  int ocsp_mode = 0;
  
  issuerhash = xstrdup (line); /* We need to work on a copy of the
                                  line because that same Assuan
                                  context may be used for an inquiry.
                                  That is because Assuan reuses its
                                  line buffer.
                                   */

  serialno = strchr (issuerhash, '.');
  if (serialno)
    *serialno++ = 0;
  else
    {
      char *endp = strchr (issuerhash, ' ');
      if (endp)
        *endp = 0;
      if (strlen (issuerhash) != 40)
        {
          xfree (issuerhash);
          return PARM_ERROR ("serialno missing in cert ID");
        }
      ocsp_mode = 1;
    }


 again:
  if (ocsp_mode)
    {
      char *cert_fpr = issuerhash;

      err = ocsp_isvalid (ctrl, cert_fpr);
      /* Fixme: If we got no ocsp response we should fall back to CRL
         mode.  Thus we need to clear OCSP_MODE, get the issuerhash
         and the serialno from the current certificate and jump to
         again. */
    }
  else
    {
      switch (crl_cache_isvalid (ctrl->server_local->crl_cache,
                                 issuerhash, serialno))
        {
        case CRL_CACHE_VALID:
          err = 0;
          break;
        case CRL_CACHE_INVALID:
          err = gpg_error (GPG_ERR_CERT_REVOKED);
          break;
        case CRL_CACHE_DONTKNOW: 
          if (did_inquire)
            err = gpg_error (GPG_ERR_NO_CRL_KNOWN);
          else if (!(err = inquire_cert (ctx)))
            {
              did_inquire = 1;
              goto again;
            }
          break;
        default:
          log_fatal ("crl_cache_isvalid returned invalid code\n");
        }
    }

  if (err)
    log_error ("command ISVALID failed: %s\n", gpg_strerror (err));
  xfree (issuerhash);
  return map_to_assuan_status (err);
}



/* LOOKUP <pattern>

   Lookup certificates matching PATTERN.  To allow for multiple
   patterns (which are ORed) quoting is required: Spaces are to be
   translated into "+" or into "%20"; obviously this requires that the
   usual escape quoting rules are applied.
*/

static int
cmd_lookup (assuan_context_t ctx, char *line)
{
  gpg_error_t err = 0;
  assuan_error_t ae;
  char *p;
  strlist_t sl, list = NULL;
  int truncated = 0, truncation_forced = 0;
  int count = 0;
  unsigned char *value = NULL;
  size_t valuelen; 
  ldap_server_t ldapserver;
  cert_fetch_context_t fetch_context;

  /* Break the line down into an STRLIST */
  for (p=line; *p; line = p)
    {
      while (*p && *p != ' ')
        p++;
      if (*p) 
        *p++ = 0;

      if (*line)
        {
          sl = xtrymalloc (sizeof *sl + strlen (line));
          if (!sl)
            {
              err = gpg_error_from_errno (errno);
              goto leave;
            }
          memset (sl, 0, sizeof *sl);
          strcpy_escaped_plus (sl->d, line);
          sl->next = list;
          list = sl;
        }
    }

  /* Loop over all configured servers. */
  for (ldapserver = opt.ldapservers;
       ldapserver && ldapserver->host && !truncation_forced;
       ldapserver = ldapserver->next)
    {
      
      if (DBG_LOOKUP)
        log_debug ("cmd_lookup: trying %s:%d base=%s\n", 
                   ldapserver->host, ldapserver->port,
                   ldapserver->base?ldapserver->base : "[default]");

      /* Fetch certificates matching pattern */
      err = start_cert_fetch (&fetch_context, list, ldapserver);
      if (err)
        {
          log_error ("start_cert_fetch failed: %s\n", gpg_strerror (err));
          goto leave;
        }

      /* Fetch the certificates for this query. */
      while (!truncation_forced)
        {
          xfree (value); value = NULL;
          err = fetch_next_cert (fetch_context, &value, &valuelen);
          if (gpg_err_code (err) == GPG_ERR_TRUNCATED)
            {
              truncated = 1;
              err = 0;
            }
          if (gpg_err_code (err) == GPG_ERR_EOF)
            {
              err = 0;
              break; /* Ready. */
            }
          if (!err && !value)
            {
              err = gpg_error (GPG_ERR_BUG);
              goto leave;
            }
          if (err)
            {
              log_error ("fetch_next_cert failed: %s\n", gpg_strerror (err));
              end_cert_fetch (fetch_context);
              goto leave;
            }
          
          if (DBG_LOOKUP)
            log_debug ("cmd_lookup: returning one cert%s\n",
                       truncated? " (truncated)":"");
          
          /* Send the data, flush the buffer and then send an END line
             as a certificate delimiter. */
          ae = assuan_send_data (ctx, value, valuelen);      
          if (!ae)
            ae = assuan_send_data (ctx, NULL, 0);
          if (!ae)
            ae = assuan_write_line (ctx, "END");
          if (ae) 
            {
              log_error ("error sending data: %s\n", assuan_strerror (ae));
              err = map_assuan_err (ae);
              end_cert_fetch (fetch_context);
              goto leave;
            }
          
          if (++count >= opt.max_replies )
            {
              truncation_forced = 1;
              log_info ("max_replies %d exceeded\n", opt.max_replies );
            }
        }

      end_cert_fetch (fetch_context);
    }

  if (truncated || truncation_forced )
    {
      char str[50];

      sprintf (str, "%d", count);
      assuan_write_status (ctx, "TRUNCATED", str);    
    }

 leave:
  if (err)
    log_error ("Command LOOKUP failed: %s\n", gpg_strerror (err));
  free_strlist (list);
  return map_to_assuan_status (err);
}


/* LOADCRL <filename>

   Load the CRL in the file with name FILENAME into our cache.  Note
   that FILENAME should be given with an absolute path because
   Dirmngrs cwd is not known.  This command is usually used by gpgsm
   using the invocation "gpgsm --call-dirmngr loadcrl <filename>".  A
   direct onvocation of Dirmngr is not useful because gpgsm might need
   to callback gpgsm to ask for the CA's certificate.
*/

static int
cmd_loadcrl (assuan_context_t ctx, char *line)
{
  ctrl_t ctrl = assuan_get_pointer (ctx);
  gpg_error_t err;
  char *buf;

  buf = xmalloc (strlen (line)+1);
  strcpy_escaped_plus (buf, line);
  err = crl_cache_load (ctrl, ctrl->server_local->crl_cache, buf);
  xfree (buf);
  if (err)
    log_error ("command LOADCRL failed: %s\n", gpg_strerror (err));
  return map_to_assuan_status (err);
}

/* LISTCRLS 

   List the content of all CRLs in a readable format.  This command is
   usually used by gpgsm using the invocation "gpgsm --call-dirmngr
   listcrls".  It may also be used directly using "dirmngr
   --list-crls".
*/

static int
cmd_listcrls (assuan_context_t ctx, char *line)
{
  gpg_error_t err;
  ctrl_t ctrl = assuan_get_pointer (ctx);
  FILE *fp = assuan_get_data_fp (ctx);

  if (!fp)
    return PARM_ERROR ("no data stream");

  err = crl_cache_list (ctrl->server_local->crl_cache, fp);
  if (err)
    log_error ("command LISTCRLS failed: %s\n", gpg_strerror (err));
  return map_to_assuan_status (err);
}



/* Tell the assuan library about our commands. */
static int
register_commands (ASSUAN_CONTEXT ctx)
{
  static struct {
    const char *name;
    int (*handler)(ASSUAN_CONTEXT, char *line);
  } table[] = {
    { "ISVALID",    cmd_isvalid },
    { "LOOKUP",     cmd_lookup },
    { "LOADCRL",    cmd_loadcrl },
    { "LISTCRLS",   cmd_listcrls },
    { "INPUT",      NULL },
    { "OUTPUT",     NULL },
    { NULL }
  };
  int i, j, rc;

  for (i=j=0; table[i].name; i++)
    {
      rc = assuan_register_command (ctx, table[i].name, table[i].handler);
      if (rc)
        return rc;
    }
  return 0;
}


/* Startup the server and run the main loop. */
void
start_command_handler (void)
{
  int rc;
  int filedes[2];
  assuan_context_t ctx;
  struct server_control_s ctrl;

  memset (&ctrl, 0, sizeof ctrl);
  dirmngr_init_default_ctrl (&ctrl);

  /* For now we use a simple pipe based server so that we can work
     from scripts.  We will later add options to run as a daemon and
     wait for requests on a Unix domain socket. */
  filedes[0] = 0;
  filedes[1] = 1;
  rc = assuan_init_pipe_server (&ctx, filedes);
  if (rc)
    {
      log_error (_("failed to initialize the server: %s\n"),
                 assuan_strerror(rc));
      dirmngr_exit (2);
    }

  rc = register_commands (ctx);
  if (rc)
    {
      log_error (_("failed to the register commands with Assuan: %s\n"),
                 assuan_strerror(rc));
      dirmngr_exit (2);
    }
  assuan_set_hello_line (ctx, "Dirmngr " VERSION " at your service");


  assuan_set_pointer (ctx, &ctrl);
  ctrl.server_local = xcalloc (1, sizeof *ctrl.server_local);
  ctrl.server_local->assuan_ctx = ctx;
  ctrl.server_local->crl_cache = crl_cache_init ();
  ctrl.server_local->message_fd = -1;

  if (DBG_ASSUAN)
    assuan_set_log_stream (ctx, log_get_stream ());

  for (;;) 
    {
      rc = assuan_accept (ctx);
      if (rc == -1)
        break;
      if (rc)
        {
          log_info (_("Assuan accept problem: %s\n"), assuan_strerror (rc));
          break;
        }

      rc = assuan_process (ctx);
      if (rc)
        {
          log_info (_("Assuan processing failed: %s\n"), assuan_strerror (rc));
          continue;
        }
    }
  
  crl_cache_deinit (ctrl.server_local->crl_cache);

  assuan_deinit_server (ctx);
}
