#include "config.h"
#include "IIOP.h"
#include "IIOP-private.h"
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <ctype.h>
#include <string.h>
#include <sys/time.h>
#ifdef HAVE_POLL
#include <sys/poll.h>
#endif

static void giop_connection_add_to_list(GIOPConnection *cnx);
static void giop_connection_remove_from_list(GIOPConnection *cnx);

void (*IIOPAddConnectionHandler)(GIOPConnection *newcnx) = NULL;
void (*IIOPRemoveConnectionHandler)(GIOPConnection *oldcnx) = NULL;
void (*IIOPIncomingMessageHandler)(GIOPRecvBuffer *recv_buffer) = NULL;

static void iiop_init(void);
static IIOPConnection *iiop_connection_new(char *host, gushort port);
static void iiop_connection_destroy(IIOPConnection *connection);
static int  iiop_get_fd(GIOPConnection *giop_connection);

DEFINE_LOCK(giop_connection_list);
GIOPConnectionList giop_connection_list;

/**** giop_init
      Inputs: None
      Outputs: None
      Side effects: Initializes giop_connection_list
      Global data structures used: giop_connection_list

      Description: Initializes giop_connection_list. Calls
                   giop_message_buffer_init() to initialize the
                   message_buffer subsystem. Calls iiop_init()
		   to perform IIOP-specific initialization.
*/

void giop_init(void)
{
  giop_message_buffer_init();

  INIT_LOCK(giop_connection_list);

  giop_connection_list.list = NULL;
  giop_connection_list.connection_list_changed = FALSE;
# ifdef HAVE_POLL
  giop_connection_list.pollset = g_array_new(FALSE);
#else
  FD_ZERO(&giop_connection_list.selectset_rd);
  FD_ZERO(&giop_connection_list.selectset_ex);
#endif
  giop_connection_list.fd_to_connection_mapping = g_ptr_array_new();

  /* This also needs to do any transport-specific initialization
     as appropriate */
  iiop_init();
}

/*** giop_connection_init

     Inputs: 'giop_connection' - memory region allocated for use as a
                                 GIOPConnection.
	     'cnxclass' - the class of connection that will be stored here
	                  (SERVER, CLIENT)

     Outputs: None

     Side effects: Initializes 'giop_connection'.

     Description: Basic setup of a GIOPConnection.
                  Sets is_valid to FALSE because it is the responsibility of
		  the transport-specific initialization routine to make
		  a connection valid.
 */

static void giop_connection_init(GIOPConnection *giop_connection,
				 GIOPConnectionClass cnxclass)
{
  giop_connection->connection_type = GIOP_CONNECTION_NONE;
  giop_connection->refcount = 0;
  giop_connection->connection_class = cnxclass;
  giop_connection->is_valid = FALSE;
}

/**** giop_connection_free
      Inputs: 'connection'
      Outputs: None
      Side effects: Makes the 'connection' invalid as a GIOPConnection
                    and as a gpointer.

      Description: Calls giop_connection_remove_from_list() to
                   stop the connection from being used for incoming.

		   If a transport-specific finalization function has
		   been provided, call it.

		   Free the memory block at '*connection'.

*/
void giop_connection_free(GIOPConnection *connection)
{
  g_return_if_fail(connection != NULL);

  giop_connection_remove_from_list(connection);

  if(connection->destroy_func)
    connection->destroy_func(connection);

  g_free(connection);
}

/**** giop_connection_list_recreate
      Inputs: None
      Outputs: None
      Side effects: giop_connection_list changes.

      Global data structures used: giop_connection_list

      Description:
           When new connections are added to giop_connection_list.list,
	   the data structures passed to poll() or select() (OS-dependant)
	   must be recreated to match this list.

	   [We do this at add-connection/remove-connection time
	    instead of every time a poll/select is done in order to
	    speed things up a little]

	    This function reinitializes the OS-specific file
	    descriptor data structure and then adds all the file
	    descriptors in the list to it.

	    It also regenerates the array that maps file descriptors
	    into GIOPConnection*'s

*/
static void
giop_connection_list_recreate(void)
{
  int curfd;
  GList *item;
  GIOPConnection *cnx;
#ifdef HAVE_POLL
  struct pollfd new_poll;
#endif

  giop_connection_list.max_fd = 0;
  for(item = giop_connection_list.list; item; item = g_list_next(item))
    {
      cnx = item->data;
      curfd = GIOP_CONNECTION_GET_FD(cnx);

      if(curfd > giop_connection_list.max_fd)
	giop_connection_list.max_fd = curfd;
  }

  g_ptr_array_set_size(giop_connection_list.fd_to_connection_mapping,
		       giop_connection_list.max_fd);

#ifdef HAVE_POLL
  g_array_truncate(giop_connection_list.pollset, struct pollfd, 0);
#else
  FD_ZERO(&giop_connection_list.selectset_rd);
  FD_ZERO(&giop_connection_list.selectset_ex);
#endif

  for(item = giop_connection_list.list; item; item = g_list_next(item))
    {
      cnx = item->data;
      curfd = GIOP_CONNECTION_GET_FD(cnx);

      giop_connection_list.fd_to_connection_mapping->pdata[curfd] = cnx;

#     ifdef HAVE_POLL
      new_poll.fd = curfd;
      new_poll.events = POLLIN|POLLPRI|POLLERR|POLLHUP|POLLNVAL;
      g_array_append_val(giop_connection_list.pollset, 
			 struct pollfd,
			 new_poll);
#     else
      FD_SET(curfd, &giop_connection_list.selectset_rd);
      FD_SET(curfd, &giop_connection_list.selectset_ex);
#     endif
  }
}

/**** giop_connection_add_to_list
      Inputs: 'cnx' - a GIOPConnection that the user wishes added to the list
      Outputs: None
      Side effects: Modifies giop_connection_list
      Global data structures used: giop_connection_list
      Bugs: Does not check for duplicate additions.

      Description:
           Adds a connection to the list of active connections.
*/
static void
giop_connection_add_to_list(GIOPConnection *cnx)
{
  g_return_if_fail(cnx->is_valid == FALSE);

  cnx->is_valid = TRUE;

  GET_LOCK(giop_connection_list);
  giop_connection_list.list = g_list_prepend(giop_connection_list.list, cnx);

  giop_connection_list_recreate();

  RELEASE_LOCK(giop_connection_list);

  if(IIOPAddConnectionHandler)
    IIOPAddConnectionHandler(cnx);
}

/**** giop_connection_remove_from_list
      Inputs: 'cnx' - a GIOPConnection that the user wishes
      Outputs: None
      Side effects: Modifies giop_connection_list
      Global data structures used: giop_connection_list

      Description:
           Removes a connection from the list of active connections.
	   Calls the library user's "I removed connection" handler if it
	   exists.

      Bugs: Does not check for duplicate removals. This may not be "bad" though.
*/
void
giop_connection_remove_from_list(GIOPConnection *cnx)
{
  g_return_if_fail(cnx->is_valid == TRUE);

  if(IIOPRemoveConnectionHandler && cnx->is_valid)
    IIOPRemoveConnectionHandler(cnx);

  cnx->is_valid = FALSE;

  GET_LOCK(giop_connection_list);

  giop_connection_list.list = g_list_remove_link(giop_connection_list.list,
						 g_list_find(giop_connection_list.list,
							     cnx));
  giop_connection_list_recreate();
  RELEASE_LOCK(giop_connection_list);
}

/************************************************
 * Routines specific to the IIOP/IPv4 transport *
 ************************************************/

/* iiop_init - no IIOP-specific initialization to perform at this time */
static void
iiop_init(void)
{
}

/*** server_cnx
     Data structure holding the IIOPConnection for the 'server' socket
     (the socket that listens on the network).

     Comments: In the future, we will need to handle this in a manner
     that will let us listen on e.g. both IPv4 and IPv6 sockets.
 */
DEFINE_LOCK(server_cnx);
IIOPConnection *server_cnx = NULL;

/**** iiop_connection_init
      Inputs: 'connection' - a memory region that needs to be initialized as
                             an 'IIOPConnection'.

      Side effects: initializes 'connection'

      Description: Performs the IIOP-specific initialization of an
                   IIOPConnection. giop_connection_init is called.

*/

void iiop_connection_init(IIOPConnection *connection, GIOPConnectionClass cnxclass)
{
  giop_connection_init(GIOP_CONNECTION(connection), cnxclass);

  GIOP_CONNECTION(connection)->connection_type =
    GIOP_CONNECTION_IIOP;

  GIOP_CONNECTION(connection)->destroy_func =
    (gpointer)iiop_connection_destroy;

  GIOP_CONNECTION(connection)->get_fd_func =
    (gpointer)iiop_get_fd;
}

/* For accept()'d fds, basically */
IIOPConnection *
iiop_connection_from_fd(int fd)
{
  IIOPConnection *fd_cnx;
  struct hostent *hent;
  char hn_tmp[65];
  int n;

  fd_cnx = g_new0(IIOPConnection, 1);

  iiop_connection_init(fd_cnx, GIOP_CONNECTION_CLIENT);

  fd_cnx->fd = fd;

  if(fd_cnx->fd < 0) {
    ORBit_Trace(TraceMod_IIOP, TraceLevel_Debug,
		"iiop_connection_new: socket_error: %s\n",
		g_strerror(errno));
    goto failed;
  }

  fd_cnx->location.sin_port = 0;
  fd_cnx->location.sin_family = AF_INET;

  n = sizeof(struct sockaddr_in);
  getpeername(fd_cnx->fd, &fd_cnx->location, &n);

  hn_tmp[64] = '\0';
  gethostname(hn_tmp, 64);
  hent = gethostbyname(hn_tmp);
  if(!hent) {
    g_warning("Couldn't find FQDN of this host [%s]. Your /etc/hosts file needs work.\n", hn_tmp);
    goto failed;
  }
  hent = gethostbyaddr(hent->h_addr, hent->h_length, AF_INET);

  fd_cnx->hostname = NULL;

  giop_connection_add_to_list(GIOP_CONNECTION(fd_cnx));

  return fd_cnx;

failed:
  close(fd_cnx->fd);
  fd_cnx->fd = -1;
  giop_connection_free(GIOP_CONNECTION(fd_cnx));
  fd_cnx = NULL;
  return NULL;
}

IIOPConnection *
iiop_connection_server(void)
{
  struct hostent *hent;

  GET_LOCK(server_cnx);

  if(!server_cnx) {
    char hn_tmp[65];
    int n;

    server_cnx = g_new0(IIOPConnection, 1);

    iiop_connection_init(server_cnx, GIOP_CONNECTION_SERVER);

    server_cnx->is_serversock = TRUE;
    server_cnx->fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

    if(server_cnx->fd < 0) {
      ORBit_Trace(TraceMod_IIOP, TraceLevel_Debug, "iiop_connection_new: socket_error: %s\n", strerror(errno));
      goto failed;
    }

    server_cnx->location.sin_port = 0;
    server_cnx->location.sin_family = AF_INET;

    bind(server_cnx->fd, &server_cnx->location, sizeof(struct sockaddr_in));

    n = sizeof(struct sockaddr_in);
    getsockname(server_cnx->fd, &server_cnx->location, &n);

    hn_tmp[64] = '\0';
    gethostname(hn_tmp, 64);
    hent = gethostbyname(hn_tmp);
    if(!hent) {
      g_warning("Couldn't find FQDN of this host [%s]. Your /etc/hosts file needs work.\n", hn_tmp);
      goto failed;
    }
    hent = gethostbyaddr(hent->h_addr, hent->h_length, AF_INET);

    server_cnx->hostname = g_strdup(hent->h_name);

    listen(server_cnx->fd, 5);

    giop_connection_add_to_list(GIOP_CONNECTION(server_cnx));
  }

  RELEASE_LOCK(server_cnx);

  return server_cnx;

failed:
  close(server_cnx->fd);
  server_cnx->fd = -1;
  giop_connection_free(GIOP_CONNECTION(server_cnx));
  server_cnx = NULL;
  RELEASE_LOCK(server_cnx);
  return NULL;
}

IIOPConnection *
iiop_connection_get(char *host, gushort port)
{
  IIOPConnection *cnx = NULL, *tmp;
  GList *link;

  GET_LOCK(giop_connection_list);
  for(link = giop_connection_list.list; link; link = link->next)
    {
      if(GIOP_CONNECTION(link->data)->connection_type != GIOP_CONNECTION_IIOP
	 || !GIOP_CONNECTION(link->data)->is_valid)
	continue;

      tmp = IIOP_CONNECTION(link->data);
      if(  ((!host && !tmp->hostname)
	     || (host && tmp->hostname && !strcmp(host, tmp->hostname)) )
	 && htons(port) == tmp->location.sin_port  )
	{
	  cnx = tmp;
	  break;
	}
    }
  RELEASE_LOCK(iiop_connection_list);

  if(!cnx)
    cnx = iiop_connection_new(host, port);

  return cnx;
}

IIOPConnection *
iiop_connection_new(char *host, gushort port)
{
  IIOPConnection *retval;

  g_return_val_if_fail(host != NULL && port != 0, NULL);

  retval = g_new0(IIOPConnection, 1);

  iiop_connection_init(retval, GIOP_CONNECTION_CLIENT);

  retval->fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if(retval->fd < 0) {
    ORBit_Trace(TraceMod_IIOP, TraceLevel_Debug, "iiop_connection_new: socket_error: %s\n", strerror(errno));
    goto failed;
  }

  retval->hostname = g_strdup(host);

  retval->location.sin_port = htons(port);
  retval->location.sin_family = AF_INET;
  if(!inet_aton(host, &retval->location.sin_addr))
    {
      struct hostent *hent;
      hent = gethostbyname(host);
      if(!hent) {
	/* a (char *)h_strerror(int) function would be nice here */
	ORBit_Trace(TraceMod_IIOP, TraceLevel_Debug, "iiop_connection_new: gethostbyname error: %d\n", h_errno);
	goto failed;
      }
      memcpy(&retval->location.sin_addr, hent->h_addr, (size_t) sizeof(retval->location.sin_addr));
    }
  if(connect(retval->fd, (struct sockaddr *)&retval->location, sizeof(retval->location)) < 0) {
    ORBit_Trace(TraceMod_IIOP, TraceLevel_Debug, "iiop_connection_new: connect error: %s\n", strerror(errno));
    goto failed;
  }

  if(server_cnx)
    GIOP_CONNECTION(retval)->orb_data = GIOP_CONNECTION(server_cnx)->orb_data;
  giop_connection_add_to_list(GIOP_CONNECTION(retval));

  return retval;

failed:
  close(retval->fd);
  retval->fd = -1;
  giop_connection_free(GIOP_CONNECTION(retval));
  return NULL;
}

static void
iiop_connection_destroy(IIOPConnection *iiop_connection)
{
  const GIOPMessageHeader mh = {"GIOP", {1,0}, FLAG_ENDIANNESS,
				GIOP_CLOSECONNECTION, 0};

  if(iiop_connection->fd >= 0)
    {
      write(iiop_connection->fd, &mh, sizeof(mh));
      close(iiop_connection->fd);
    }

  g_free(iiop_connection->hostname);
}

static int
iiop_get_fd(GIOPConnection *giop_connection)
{
  return IIOP_CONNECTION(giop_connection)->fd;
}

static int giop_nloops = 0;

void giop_main_quit(void) { giop_nloops--; }

void
giop_main(void)
{
  int looplevel = ++giop_nloops;

  while(giop_nloops > 0) {

    giop_main_iterate(TRUE);

    if(giop_nloops != looplevel) {
      giop_nloops = --looplevel;
      return;
    }
  }
}

GIOPRecvBuffer *
giop_main_next_message(gboolean blocking)
{
    GIOPConnection *connection;
    GIOPRecvBuffer *recv_buffer = NULL;

    do {
      recv_buffer = giop_received_list_pop();
      if(recv_buffer)
	break;

      connection = giop_check_connections(blocking);

      if(!connection)
	return NULL;

      if(connection->connection_class == GIOP_CONNECTION_SERVER) {
	struct sockaddr_in sock;
	int n;
	int newfd;
	GIOPConnection *newcnx;

	newfd = accept(GIOP_CONNECTION_GET_FD(connection), &sock, &n);

	newcnx = GIOP_CONNECTION(iiop_connection_from_fd(newfd));
	GIOP_CONNECTION(newcnx)->orb_data = connection->orb_data;
	giop_connection_ref(newcnx);
      } else
	recv_buffer = giop_recv_message_buffer_use(connection);
    } while(!recv_buffer);

    return recv_buffer;
}

void
giop_main_handle_connection(GIOPConnection *connection)
{
  GIOPRecvBuffer *recv_buffer;

  g_return_if_fail(connection != NULL);
  g_return_if_fail(connection->is_valid);

  if(connection->connection_class == GIOP_CONNECTION_SERVER) {
    struct sockaddr_in sock;
    int n;
    int newfd;
    GIOPConnection *newcnx;

    newfd = accept(GIOP_CONNECTION_GET_FD(connection), &sock, &n);

    newcnx = GIOP_CONNECTION(iiop_connection_from_fd(newfd));
    GIOP_CONNECTION(newcnx)->orb_data = connection->orb_data;
    giop_connection_ref(newcnx);
    return;
  } else
    recv_buffer = giop_recv_message_buffer_use(connection);

  if(recv_buffer) {
    if(IIOPIncomingMessageHandler)
      IIOPIncomingMessageHandler(recv_buffer);
    else
      giop_received_list_push(recv_buffer);
  }
}

/**** giop_main_handle_connection_exception

   Input: GIOPConnection *connection

   Output:

   Side effects: invalidates connection

   Description:
       When poll() or select() indicates that a file descriptor
       has been closed at the remote end, we must invalidate the associated
       GIOPConnection structure.
*/
void
giop_main_handle_connection_exception(GIOPConnection *connection)
{
  g_return_if_fail(connection != NULL);
  g_return_if_fail(connection->is_valid);

  giop_connection_unref(connection);
  g_warning("Gratuitously removing connection %p from poll list\n",
	    connection);
  giop_connection_remove_from_list(connection);
}

/**** giop_main_iterate
      Input: 'blocking' - flag to indicate whether to wait for incoming
             messages (TRUE), or whether to return immediately if no
	     incoming messages are available (FALSE).
      Output: None
      Description:
             Gets the next message into recv_buffer (see
             giop_main_next_message) If we have a handler for incoming
             messages, then pass recv_buffer to the handler (handler
             becomes the new owner of recv_buffer's contents). Otherwise,
	     tosses it onto the list of received-but-unprocessed buffers.

      Warnings:
             If you don't have an IIOPIncomingMessageHandler set, you're
	     probably really screwed in the long run.
 */
void
giop_main_iterate(gboolean blocking)
{
  GIOPRecvBuffer *recv_buffer;

  recv_buffer = giop_main_next_message(blocking);
  
  if(IIOPIncomingMessageHandler)
    IIOPIncomingMessageHandler(recv_buffer);
  else
    giop_received_list_push(recv_buffer);
}

/**** giop_check_connections
      Inputs: 'block_for_reply' - If no incoming data is immediately available
              should this routine wait for incoming data (TRUE) or return
	      immediately (FALSE).

      Outputs: 'connection' - the first connection that has incoming
               data available for reading (supposedly a GIOP message, but
	       could be anything).

      Side effects: Removes closed connections from the active list.

      Global data structures used: giop_connection_list
      
      Description: Does a poll or select (OS-dependant) on the list of file
                   descriptors in giop_connection_list.

		   If a file descriptor has been closed, call
		   giop_connection_handle_exception() on it and (as
		   appropriated by 'block_for_reply') either return
		   NULL or do another poll/select.

		   If a file descriptor has data available for
		   reading, find the associated GIOPConnection (using
		   giop_connection_list.fd_to_connection_mapping) and
		   return that.
          
 */
GIOPConnection *
giop_check_connections(gboolean block_for_reply)
{
  GIOPConnection *connection = NULL;
  int pollret;
  int numcnx_checks;
  int i;

#ifndef HAVE_POLL
  struct timeval immediate_timeout = {0,0};
#endif

 do_read_msg:

#ifdef HAVE_POLL
  pollret = poll((struct pollfd *)giop_connection_list.pollset->data,
		 g_array_length(giop_connection_list.pollset,
			      struct pollfd),
		 block_for_reply?-1:0);

#     else /* !HAVE_POLL */

  pollret = select (giop_connection_list.max_fd + 1,
		    &giop_connection_list.selectset_rd,
		    NULL, &giop_connection_list.selectset_ex,
		    block_for_reply?NULL:&immediate_timeout);
#     endif /* !HAVE_POLL */

  if(pollret <= 0) {
    if(pollret < 0)
      g_warning("Error code from select/poll: %s\n", g_strerror(errno));
    return NULL;
  }

#ifdef HAVE_POLL
  numcnx_checks = g_array_length(giop_connection_list.pollset, struct pollfd);
#else
  numcnx_checks = giop_connection_list.max_fd+1;
#endif

  for(i = 0; i < numcnx_checks; i++)
    {
#ifdef HAVE_POLL
      struct pollfd *p = 
      &g_array_index(giop_connection_list.pollset,
		     struct pollfd, 
		     i);

      connection = giop_connection_list.fd_to_connection_mapping->pdata[p->fd];
      if(p->revents & (POLLHUP|POLLNVAL)) {
	giop_main_handle_connection_exception(connection);
	connection = NULL;
      } else if(p->revents & POLLIN)
	break;
#else /* !HAVE_POLL */

      connection = giop_connection_list.fd_to_connection_mapping->pdata[i];

      if(FD_ISSET(i, &giop_connection_list.selectset_ex)) {
	giop_main_handle_connection_exception(connection);
	connection = NULL;
      } else if (FD_ISSET(i, &giop_connection_list.selectset_rd))
	break;
#endif /* !HAVE_POLL */

    }

  if(!connection && block_for_reply)
    goto do_read_msg;

  return connection;
}
