#include "orbit-c-backend.h"
#include <glib.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef struct {
  FILE *of;
  IDL_ns ns;
  IDL_tree tree;
} CBEStubInfo;

/* Pass either DATA_IN or DATA_OUT, tells you how many parameters
   (including the return value) will cause data to be sent or received */

static void orbit_cbe_stub_process_piece(CBEStubInfo *ski);
static void cbe_stub_do_attr_dcl(CBEStubInfo *sti);
static void cbe_stub_do_binop(CBEStubInfo *sti);
static void cbe_stub_do_boolean(CBEStubInfo *sti);
static void cbe_stub_do_case_stmt(CBEStubInfo *sti);
static void cbe_stub_do_char(CBEStubInfo *sti);
static void cbe_stub_do_const_dcl(CBEStubInfo *sti);
static void cbe_stub_do_except_dcl(CBEStubInfo *sti);
static void cbe_stub_do_fixed(CBEStubInfo *sti);
static void cbe_stub_do_float(CBEStubInfo *sti);
static void cbe_stub_do_forward_dcl(CBEStubInfo *sti);
static void cbe_stub_do_gentree(CBEStubInfo *sti);
static void cbe_stub_do_ident(CBEStubInfo *sti);
static void cbe_stub_do_integer(CBEStubInfo *sti);
static void cbe_stub_do_interface(CBEStubInfo *sti);
static void cbe_stub_do_list(CBEStubInfo *sti);
static void cbe_stub_do_member(CBEStubInfo *sti);
static void cbe_stub_do_module(CBEStubInfo *sti);
static void cbe_stub_do_none(CBEStubInfo *sti);
static void cbe_stub_do_op_dcl(CBEStubInfo *sti);
#if 0
static void cbe_stub_do_param_dcl_marshal(CBEStubInfo *sti);
static void cbe_stub_do_param_dcl_demarshal(CBEStubInfo *sti);
#endif
static void cbe_stub_do_string(CBEStubInfo *sti);
static void cbe_stub_do_type_any(CBEStubInfo *sti);
static void cbe_stub_do_type_array(CBEStubInfo *sti);
static void cbe_stub_do_type_boolean(CBEStubInfo *sti);
static void cbe_stub_do_type_char(CBEStubInfo *sti);
static void cbe_stub_do_type_dcl(CBEStubInfo *sti);
static void cbe_stub_do_type_enum(CBEStubInfo *sti);
static void cbe_stub_do_type_fixed(CBEStubInfo *sti);
static void cbe_stub_do_type_float(CBEStubInfo *sti);
static void cbe_stub_do_type_integer(CBEStubInfo *sti);
static void cbe_stub_do_type_object(CBEStubInfo *sti);
static void cbe_stub_do_type_octet(CBEStubInfo *sti);
static void cbe_stub_do_type_sequence(CBEStubInfo *sti);
static void cbe_stub_do_type_string(CBEStubInfo *sti);
static void cbe_stub_do_type_struct(CBEStubInfo *sti);
static void cbe_stub_do_type_union(CBEStubInfo *sti);
static void cbe_stub_do_type_wide_char(CBEStubInfo *sti);
static void cbe_stub_do_type_wide_string(CBEStubInfo *sti);
static void cbe_stub_do_unaryop(CBEStubInfo *sti);
static void cbe_stub_do_wide_char(CBEStubInfo *sti);
static void cbe_stub_do_wide_string(CBEStubInfo *sti);

void
orbit_cbe_write_stubs(FILE *outfile, IDL_ns ns, IDL_tree tree,
		      const char *header_filename)
{
  CBEStubInfo sti;
  sti.of = outfile;
  sti.ns = ns;
  sti.tree = tree;

  fprintf(outfile, "#include <string.h>\n");
  fprintf(outfile, "#include \"%s\"\n\n", header_filename);
  fprintf(outfile, "#define GET_ATOM(x) ({ GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->decoder(&x, (GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur), sizeof(x)); GIOP_RECV_BUFFER(_ORBIT_recv_buffer)->cur += sizeof(x); })\n");

  orbit_cbe_stub_process_piece(&sti);
}

static void
orbit_cbe_stub_process_piece(CBEStubInfo *sti)
{
  g_return_if_fail(sti != NULL);
  g_return_if_fail(sti->tree != NULL);

  switch(IDL_NODE_TYPE(sti->tree)) {
  case IDLN_ATTR_DCL:
    cbe_stub_do_attr_dcl(sti);
    break;
  case IDLN_BINOP:
    cbe_stub_do_binop(sti);
    break;
  case IDLN_BOOLEAN:
    cbe_stub_do_boolean(sti);
    break;
  case IDLN_CASE_STMT:
    cbe_stub_do_case_stmt(sti);
    break;
  case IDLN_CHAR:
    cbe_stub_do_char(sti);
    break;
  case IDLN_CONST_DCL:
    cbe_stub_do_const_dcl(sti);
    break;
  case IDLN_EXCEPT_DCL:
    cbe_stub_do_except_dcl(sti);
    break;
  case IDLN_FIXED:
    cbe_stub_do_fixed(sti);
    break;
  case IDLN_FLOAT:
    cbe_stub_do_float(sti);
    break;
  case IDLN_FORWARD_DCL:
    cbe_stub_do_forward_dcl(sti);
    break;
  case IDLN_GENTREE:
    cbe_stub_do_gentree(sti);
    break;
  case IDLN_IDENT:
    cbe_stub_do_ident(sti);
    break;
  case IDLN_INTEGER:
    cbe_stub_do_integer(sti);
    break;
  case IDLN_INTERFACE:
    cbe_stub_do_interface(sti);
    break;
  case IDLN_LIST:
    cbe_stub_do_list(sti);
    break;
  case IDLN_MEMBER:
    cbe_stub_do_member(sti);
    break;
  case IDLN_MODULE:
    cbe_stub_do_module(sti);
    break;
  case IDLN_NONE:
    cbe_stub_do_none(sti);
    break;
  case IDLN_OP_DCL:
    cbe_stub_do_op_dcl(sti);
    break;
  case IDLN_PARAM_DCL:
    g_assert(!"Can't do a PARAM outside of an operation!\n");
    break;
  case IDLN_STRING:
    cbe_stub_do_string(sti);
    break;
  case IDLN_TYPE_ANY:
    cbe_stub_do_type_any(sti);
    break;
  case IDLN_TYPE_ARRAY:
    cbe_stub_do_type_array(sti);
    break;
  case IDLN_TYPE_BOOLEAN:
    cbe_stub_do_type_boolean(sti);
    break;
  case IDLN_TYPE_CHAR:
    cbe_stub_do_type_char(sti);
    break;
  case IDLN_TYPE_DCL:
    cbe_stub_do_type_dcl(sti);
    break;
  case IDLN_TYPE_ENUM:
    cbe_stub_do_type_enum(sti);
    break;
  case IDLN_TYPE_FIXED:
    cbe_stub_do_type_fixed(sti);
    break;
  case IDLN_TYPE_FLOAT:
    cbe_stub_do_type_float(sti);
    break;
  case IDLN_TYPE_INTEGER:
    cbe_stub_do_type_integer(sti);
    break;
  case IDLN_TYPE_OBJECT:
    cbe_stub_do_type_object(sti);
    break;
  case IDLN_TYPE_OCTET:
    cbe_stub_do_type_octet(sti);
    break;
  case IDLN_TYPE_SEQUENCE:
    cbe_stub_do_type_sequence(sti);
    break;
  case IDLN_TYPE_STRING:
    cbe_stub_do_type_string(sti);
    break;
  case IDLN_TYPE_STRUCT:
    cbe_stub_do_type_struct(sti);
    break;
  case IDLN_TYPE_UNION:
    cbe_stub_do_type_union(sti);
    break;
  case IDLN_TYPE_WIDE_CHAR:
    cbe_stub_do_type_wide_char(sti);
    break;
  case IDLN_TYPE_WIDE_STRING:
    cbe_stub_do_type_wide_string(sti);
    break;
  case IDLN_UNARYOP:
    cbe_stub_do_unaryop(sti);
    break;
  case IDLN_WIDE_CHAR:
    cbe_stub_do_wide_char(sti);
    break;
  case IDLN_WIDE_STRING:
    cbe_stub_do_wide_string(sti);
    break;
  case IDLN_ANY:
    g_error("%s not handled in stubs\n",
	    IDL_tree_type_names[IDL_NODE_TYPE(sti->tree)]);
    break;
  }
}

static void
cbe_stub_do_attr_dcl(CBEStubInfo *sti)
{
  IDL_tree curop, curitem;
  GString *attrname = g_string_new(NULL);
  CBEStubInfo substi = *sti;

  for(curitem = IDL_ATTR_DCL(sti->tree).simple_declarations;
      curitem; curitem = IDL_LIST(curitem).next) {

    /* Fake the attribute get/set methods as operation declarations */
    IDL_tree ident, ns_data_save;
    int i;

    for (i = 0; i < 2; ++i) {

	    if (i && IDL_ATTR_DCL(sti->tree).f_readonly)
		    break;

	    /* Output the operation on this attribute */
	    g_string_sprintf(attrname, i ? "_set_%s" : "_get_%s",
			     IDL_IDENT(IDL_LIST(curitem).data).str);
	    ident = IDL_ident_new(strdup(attrname->str));
	    
	    /* Tell the ident where our namespace node is, and request
	       a return value if this is the _get operation */

	    IDL_IDENT_TO_NS(ident) = IDL_IDENT_TO_NS(IDL_LIST(curitem).data);
	    curop = IDL_op_dcl_new(0, i == 0 ?
				   IDL_ATTR_DCL(sti->tree).param_type_spec : NULL,
				   ident, NULL, NULL, NULL);
	    
	    curop->up = sti->tree->up;
	    substi.tree = curop;
	    
	    /* Save the namespace ident (IDL_GENTREE data) reference, assign
	       back to the temporary tree, output the operation, then restore
	       the namespace ident link */
	    ns_data_save = IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data;
	    IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data = ident;

	    if (i) {
		    /* The set routine also needs the value, so we
		       temporarily add that to the operation
		       declaration */
		    IDL_OP_DCL(curop).parameter_dcls = IDL_list_new(
			    IDL_param_dcl_new(IDL_PARAM_IN,
					      IDL_ATTR_DCL(sti->tree).param_type_spec,
					      IDL_ident_new(strdup("value"))));
	    }
	    
	    orbit_cbe_stub_process_piece(&substi);

	    /* Restore the fake link to the original in the namespace */
	    IDL_GENTREE(IDL_IDENT_TO_NS(IDL_LIST(curitem).data)).data = ns_data_save;

	    if (i) {
		    /* Free only what we've created for the fake node, so remove 
		       the attribute node element and then free the rest */
		    IDL_PARAM_DCL(IDL_LIST(
			    IDL_OP_DCL(curop).parameter_dcls).data).param_type_spec = NULL;
	    }
	    
	    /* Remove what we've "borrowed" from ATTR_DCL from the
	       fake curop node then free the rest */
	    IDL_OP_DCL(curop).op_type_spec = NULL;
	    IDL_tree_free(curop);
    }
  }

  g_string_free(attrname, TRUE);
}

static void
cbe_stub_do_binop(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_boolean(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_case_stmt(CBEStubInfo *sti)
{
  /* Nothing to do in stubs (???) */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_char(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_const_dcl(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_except_dcl(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_fixed(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_float(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_forward_dcl(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_gentree(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_ident(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_integer(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_interface(CBEStubInfo *sti)
{
  sti->tree = IDL_INTERFACE(sti->tree).body;
  orbit_cbe_stub_process_piece(sti);
}

static void
cbe_stub_do_list(CBEStubInfo *sti)
{
  IDL_tree curitem;
  CBEStubInfo substi = *sti;

  for(curitem = sti->tree; curitem; curitem = IDL_LIST(curitem).next) {
    substi.tree = IDL_LIST(curitem).data;
    orbit_cbe_stub_process_piece(&substi);
  }
}

static void
cbe_stub_do_member(CBEStubInfo *sti)
{
  /* Nothing to do in stubs (?) */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_module(CBEStubInfo *sti)
{
  char *id;

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_MODULE(sti->tree).ident), "_", 1);
  fprintf(sti->of, "/***************** Begin module %s ***************/\n", id);

  sti->tree = IDL_MODULE(sti->tree).definition_list;
  orbit_cbe_stub_process_piece(sti);
  fprintf(sti->of, "/***************** End module %s ***************/\n", id);
  free(id);
}

static void
cbe_stub_do_none(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_string(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_type_any(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_array(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_boolean(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_char(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_dcl(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_enum(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_fixed(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_float(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_integer(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_object(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_octet(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_type_sequence(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_type_string(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_type_struct(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_union(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  /* g_assert(!"Not yet implemented"); */
}

static void
cbe_stub_do_type_wide_char(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_type_wide_string(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_unaryop(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_wide_char(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}

static void
cbe_stub_do_wide_string(CBEStubInfo *sti)
{
  /* Nothing to do in stubs. */
  g_assert(!"Not yet implemented");
}


/* There be dragons in here... */
void
cbe_print_var_dcl(FILE *of, IDL_tree tree)
{
  /* This is a variant of print_param_dcl */

  if(IDL_NODE_TYPE(tree) != IDLN_PARAM_DCL) {
    g_warning("cbe_print_param_dcl: Outputting \"retval\" for type %s\n",
	      IDL_tree_type_names[IDL_NODE_TYPE(tree)]);
    orbit_cbe_write_typespec(of, tree);
    fprintf(of, " _ORBIT_retval");
  } else {

    orbit_cbe_write_typespec(of,
			     IDL_PARAM_DCL(tree).param_type_spec);

    fprintf(of, " %s", IDL_IDENT(IDL_PARAM_DCL(tree).simple_declarator).str);
  }
}

void
cbe_print_param_dcl(FILE *of, IDL_tree tree)
{
  IDL_ParamRole r = DATA_IN;

  if(IDL_NODE_TYPE(tree) != IDLN_PARAM_DCL) {
    orbit_cbe_write_typespec(of, tree);
    orbit_cbe_param_printptrs(of, tree, DATA_RETURN);
    fprintf(of, " _ORBIT_retval");
  } else {

    orbit_cbe_write_typespec(of,
			     IDL_PARAM_DCL(tree).param_type_spec);

    switch(IDL_PARAM_DCL(tree).attr) {
    case IDL_PARAM_IN: r = DATA_IN; break;
    case IDL_PARAM_INOUT: r = DATA_INOUT; break;
    case IDL_PARAM_OUT: r = DATA_OUT; break;
    default:
      g_error("Unknown IDL_PARAM type\n");
    }

    orbit_cbe_param_printptrs(of,
			      IDL_PARAM_DCL(tree).param_type_spec, r);
    fprintf(of, " %s", IDL_IDENT(IDL_PARAM_DCL(tree).simple_declarator).str);
  }
}

static void
cbe_stub_do_op_dcl(CBEStubInfo *sti)
{
  /* XXX we don't handle contexts here yet - fix that */
  char *id, *id2;
  IDL_tree curitem;
  int level, i, n;
  CBEMarshalInfo mi = {sti->of, NULL, NULL, NULL, NULL, FALSE};
  CBEDemarshalInfo dmi = {sti->of, NULL, NULL, NULL, NULL, TRUE, TRUE};
  GString *tmpstr = g_string_new(NULL);

  id = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_OP_DCL(sti->tree).ident),
			       "_", 0);

  if(IDL_OP_DCL(sti->tree).op_type_spec && IDL_OP_DCL(sti->tree).f_oneway) {
    g_error("[%s] You cannot have a return value from a oneway operation!\n", id);
  }

  orbit_cbe_write_typespec(sti->of, IDL_OP_DCL(sti->tree).op_type_spec);
  orbit_cbe_param_printptrs(sti->of,
			    IDL_OP_DCL(sti->tree).op_type_spec, DATA_RETURN);

  curitem = IDL_get_parent_node(sti->tree, IDLN_INTERFACE, &level);

  g_assert(curitem);
  id2 = IDL_ns_ident_to_qstring(IDL_IDENT_TO_NS(IDL_INTERFACE(curitem).ident), "_", 0);
  fprintf(sti->of, "\n%s(%s obj, ", id, id2);
  free(id); free(id2);

  for(curitem = IDL_OP_DCL(sti->tree).parameter_dcls; curitem;
      curitem = IDL_LIST(curitem).next) {
    cbe_print_param_dcl(sti->of, IDL_LIST(curitem).data);
    fprintf(sti->of, ", ");
  }

  fprintf(sti->of, "CORBA_Environment *ev)\n{\n");
  fprintf(sti->of, "  GIOP_unsigned_long _ORBIT_request_id;\n");
  fprintf(sti->of, "  GIOPSendBuffer *_ORBIT_send_buffer;\n");
  fprintf(sti->of, "  GIOPRecvBuffer *_ORBIT_recv_buffer;\n");
  fprintf(sti->of, "  static const struct { CORBA_unsigned_long len; char opname[%d]; } _ORBIT_operation_name_data = { %d, \"%s\" };\n",
	  strlen(IDL_IDENT(IDL_OP_DCL(sti->tree).ident).str) + 1,
	  strlen(IDL_IDENT(IDL_OP_DCL(sti->tree).ident).str) + 1,
	  IDL_IDENT(IDL_OP_DCL(sti->tree).ident).str);
  fprintf(sti->of, "  static const struct iovec _ORBIT_operation_vec = {(gpointer)&_ORBIT_operation_name_data, %d};\n",
	  sizeof(CORBA_unsigned_long)+
	  strlen(IDL_IDENT(IDL_OP_DCL(sti->tree).ident).str) + 1);

  if(IDL_OP_DCL(sti->tree).op_type_spec) {
    fprintf(sti->of, "  ");
    cbe_print_param_dcl(sti->of, IDL_OP_DCL(sti->tree).op_type_spec);
    fprintf(sti->of, ";\n");
  }
  fprintf(sti->of, "\n");

  fprintf(sti->of, "  _ORBIT_request_id = giop_get_request_id();\n");
  fprintf(sti->of, "  _ORBIT_send_buffer = \n");
  fprintf(sti->of, "    giop_send_request_buffer_use(obj->connection, NULL,\n");
  fprintf(sti->of, "				 _ORBIT_request_id, %s, &(obj->object_key_vec),\n",
	  IDL_OP_DCL(sti->tree).f_oneway?"CORBA_FALSE":"CORBA_TRUE");
  fprintf(sti->of, "				 &_ORBIT_operation_vec, NULL);\n\n");


  for(curitem = IDL_OP_DCL(sti->tree).parameter_dcls;
      curitem; curitem = IDL_LIST(curitem).next) {

    if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_IN
       || IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_INOUT) {

      if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_IN)
	n = orbit_cbe_param_numptrs(mi.param, DATA_IN);
      else
	n = orbit_cbe_param_numptrs(mi.param, DATA_INOUT) /* only getting, not setting */ - 1;

      mi.previous_param = mi.param;
      mi.param = IDL_LIST(curitem).data;

      g_string_assign(tmpstr, "");
      for(i = 0; i < n; i++)
	g_string_append_c(tmpstr, '*');
      g_string_sprintfa(tmpstr, "%s",
			IDL_IDENT(IDL_PARAM_DCL(mi.param).simple_declarator).str);
      mi.param_name = tmpstr->str;
      fprintf(sti->of, "\n\n  /* marshal parameter %s */\n", mi.param_name);
      cbe_output_marshaller(mi);
    }
  }

  fprintf(sti->of, "  giop_send_buffer_write(_ORBIT_send_buffer);\n");

  /* NOTE: I've put the "unuse" before the reply receive because I
     figured that this data structure shuffling is best done while the
     network is sending the data and the remote end is processing the
     request. If I'm wrong, please move it elsewhere */

  fprintf(sti->of, "  giop_send_buffer_unuse(_ORBIT_send_buffer);\n");

  fprintf(sti->of, "  _ORBIT_recv_buffer = giop_recv_reply_buffer_use(_ORBIT_request_id, TRUE);\n");

  fprintf(sti->of, "  if(_ORBIT_recv_buffer == NULL || _ORBIT_recv_buffer->message_buffer.message_header.message_type != GIOP_REPLY) {\n    CORBA_exception_set_system(ev, ex_CORBA_COMM_FAILURE, CORBA_COMPLETED_MAYBE);\n    if(_ORBIT_recv_buffer) giop_recv_buffer_unuse(_ORBIT_recv_buffer);\n    return;\n  }\n");

    fprintf(sti->of, "  if(_ORBIT_recv_buffer->message.u.reply.reply_status != GIOP_NO_EXCEPTION) {\n");
    /* This should also pass a structure that lists info about the
       operation-specific exceptions raised. */
    fprintf(sti->of, "    ORBit_handle_exception(_ORBIT_recv_buffer, ev, NULL);\n");
    fprintf(sti->of, "    giop_recv_buffer_unuse(_ORBIT_recv_buffer);\n");
    fprintf(sti->of, "    return;\n");
    fprintf(sti->of, "  }\n");

  if(cbe_op_dcl_nparams(sti->tree, DATA_INOUT|DATA_OUT|DATA_RETURN) > 0) {

    dmi.param = NULL;

    fprintf(sti->of, "  if(giop_msg_conversion_needed(GIOP_MESSAGE_BUFFER(_ORBIT_recv_buffer))) {\n");

    do { /* This do loop is here basically so we generate two
	    different versions of the same code - the "byteswapping needed"
	    and "no byteswapping needed" versions */

      if(IDL_OP_DCL(sti->tree).op_type_spec) {
	fprintf(sti->of, "  /* demarshal return value */\n");
	dmi.param = IDL_OP_DCL(sti->tree).op_type_spec;
	dmi.param_name = "_ORBIT_retval";
	cbe_output_demarshaller(dmi);
      }

      for(curitem = IDL_OP_DCL(sti->tree).parameter_dcls;
	  curitem; curitem = IDL_LIST(curitem).next) {


	if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_OUT
	   || IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_INOUT) {

	  dmi.previous_param = dmi.param;
	  dmi.param = IDL_LIST(curitem).data;

	  if(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr == IDL_PARAM_OUT)
	    n = orbit_cbe_param_numptrs(dmi.param, DATA_OUT);
	  else
	    n = orbit_cbe_param_numptrs(dmi.param, DATA_INOUT); /* Clarify */

	  g_string_assign(tmpstr, "");
	  for(i = 0; i < n; i++)
	    g_string_append_c(tmpstr, '*');
	  g_string_sprintfa(tmpstr, "%s",
			    IDL_IDENT(IDL_PARAM_DCL(dmi.param).simple_declarator).str);
	  dmi.param_name = tmpstr->str;

	  fprintf(sti->of, "  /* demarshal parameter %s */\n", dmi.param_name);
	  cbe_output_demarshaller(dmi);
	}
      }

      if(dmi.byteswap_version)
	fprintf(sti->of, "  } else {\n");

    } while(dmi.byteswap_version-- > 0);

    fprintf(sti->of, "  }\n");
  }

  fprintf(sti->of, "  giop_recv_buffer_unuse(_ORBIT_recv_buffer);\n");

  fprintf(sti->of, "  ev->_major = CORBA_NO_EXCEPTION;\n");
  if(IDL_OP_DCL(sti->tree).op_type_spec) {
    fprintf(sti->of, "  return _ORBIT_retval;\n");
  }

  fprintf(sti->of, "}\n\n");
  g_string_free(tmpstr, TRUE);
}

int cbe_op_dcl_nparams(IDL_tree op, IDL_ParamRole role)
{
  IDL_tree curitem;
  int retval;

  for(retval = 0, curitem = IDL_OP_DCL(op).parameter_dcls;
      curitem; curitem = IDL_LIST(curitem).next) {
    switch(IDL_PARAM_DCL(IDL_LIST(curitem).data).attr) {
    case IDL_PARAM_IN: if(role & DATA_IN) retval++;
      break;
    case IDL_PARAM_OUT: if(role & DATA_OUT) retval++;
      break;
    case IDL_PARAM_INOUT: if(role & DATA_INOUT) retval++;
      break;
    }
  }

  if((role & DATA_RETURN) && IDL_OP_DCL(op).op_type_spec)
    retval++;

  return retval;
}
