/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Dick Porter <dick@cymru.net>
 *
 */

/*
 * CORBA_Environment handling functions
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "orbit.h"

struct SysExInfo {
	const char *repo_id;
	const int exnum;
};

static const struct SysExInfo exception_table[] = {
	{NULL, 0},
	{"CORBA_UNKNOWN", 1},
	{"CORBA_BAD_PARAM", 2},
	{"CORBA_NO_MEMORY", 3},
	{"CORBA_IMP_LIMIT", 4},
	{"CORBA_COMM_FAILURE", 5},
	{"CORBA_INV_OBJREF", 6},
	{"CORBA_NO_PERMISSION", 7},
	{"CORBA_INTERNAL", 8},
	{"CORBA_MARSHAL", 9},
	{"CORBA_INITIALIZE", 10},
	{"CORBA_NO_IMPLEMENT", 11},
	{"CORBA_BAD_TYPECODE", 12},
	{"CORBA_BAD_OPERATION", 13},
	{"CORBA_NO_RESOURCES", 14},
	{"CORBA_NO_RESPONSE", 15},
	{"CORBA_PERSIST_STORE", 16},
	{"CORBA_BAD_INV_ORDER", 17},
	{"CORBA_TRANSIENT", 18},
	{"CORBA_FREE_MEM", 19},
	{"CORBA_INV_IDENT", 20},
	{"CORBA_INV_FLAG", 21},
	{"CORBA_INTF_REPOS", 22},
	{"CORBA_BAD_CONTEXT", 23},
	{"CORBA_OBJ_ADAPTER", 24},
	{"CORBA_DATA_CONVERSION", 25},
	{"CORBA_OBJECT_NOT_EXIST", 26},
	{"CORBA_TRANSACTION_REQUIRED", 27},
	{"CORBA_TRANSACTION_ROLLEDBACK", 28},
	{"CORBA_INVALID_TRANSACTION", 29},
	{NULL,0}
};

void CORBA_exception_free(CORBA_Environment *ev)
{
	g_assert(ev!=NULL);

	ev->_major=CORBA_NO_EXCEPTION;

	if(ev->_repo_id!=NULL) {
		CORBA_free(ev->_repo_id);
		ev->_repo_id=NULL;
	}

	if(ev->_params!=NULL) {
		CORBA_free(ev->_params);
		ev->_params=NULL;
	}

	if(ev->_any!=NULL) {
		CORBA_free(ev->_any);
		ev->_any=NULL;
	}
}

void CORBA_exception_set(CORBA_Environment *ev, CORBA_exception_type major, 
			 CORBA_char *except_repos_id, void *param)
{
	g_assert(ev!=NULL);

	if(ev->_major!=CORBA_NO_EXCEPTION) {
		CORBA_exception_free(ev);
	}

	ev->_major=major;

	if(except_repos_id==NULL) {
		ev->_repo_id=NULL;
	} else {
		ev->_repo_id=CORBA_string_dup(except_repos_id);
	}
	ev->_params=param;
}

void CORBA_exception_set_system(CORBA_Environment *ev, CORBA_unsigned_long minor, CORBA_completion_status completed)
{
	CORBA_SystemException *new;

	new=ORBit_alloc(sizeof(CORBA_SystemException), NULL, NULL);
	if(new!=NULL) {
		new->minor=minor;
		new->completed=completed;

		/* XXX what should the repo ID be? */
		CORBA_exception_set(ev, CORBA_SYSTEM_EXCEPTION,
				    (gchar *)exception_table[ev->_major].repo_id,
				    new);
	}
}

void CORBA_exception_init(CORBA_Environment *ev)
{
	g_assert(ev!=NULL);

	ev->_major=CORBA_NO_EXCEPTION;
	ev->_repo_id=NULL;
	ev->_params=NULL;
	ev->_any=NULL;
}

CORBA_char *CORBA_exception_id(CORBA_Environment *ev)
{
	g_assert(ev!=NULL);

	if(ev->_major==CORBA_NO_EXCEPTION) {
		return(NULL);
	} else {
		return(ev->_repo_id);
	}
}

void *CORBA_exception_value(CORBA_Environment *ev)
{
	g_assert(ev!=NULL);

	if(ev->_major==CORBA_NO_EXCEPTION) {
		return(NULL);
	} else {
		return(ev->_params);
	}
}

CORBA_any *CORBA_exception_as_any(CORBA_Environment *ev)
{
	g_assert(ev!=NULL);

	if(ev->_major==CORBA_NO_EXCEPTION) {
		return(NULL);
	}

	if(ev->_any!=NULL) {
		return(ev->_any);
	}

	ev->_any=g_new(CORBA_any, 1);
	if(ev->_any!=NULL) {
		memcpy(&(ev->_any->_type), &TC_CORBA_ExceptionDescription,
			(size_t)sizeof(CORBA_TypeCode) );
		ev->_any->_value = ev->_params;
		ev->_any->_flags = 0;
	}

	return(ev->_any);
}

/**** ORBit_handle_exception
      Inputs: 'rb' - a receive buffer for which an exception condition has
                     been determined
	      'ev' - memory in which to store the exception information

	      'user_exception_data' - for future use. Will pass in a
	                              list of user exceptions raisable
				      for this particular operation.
      Side-effects: reinitializes '*ev'

      Description:
           During demarshalling a reply, if reply_status != CORBA_NO_EXCEPTION,
	   we must find out what exception was raised and place that information
	   in '*ev'.  */

void ORBit_handle_exception(GIOPRecvBuffer *rb, CORBA_Environment *ev,
			    gpointer user_exception_data)
{
	CORBA_SystemException *new;
	CORBA_unsigned_long len, minor, completion_status;

	g_return_if_fail(GIOP_MESSAGE_BUFFER(rb)->message_header.message_type == GIOP_REPLY);

	CORBA_exception_free(ev);

	rb->cur = ALIGN_ADDRESS(rb->cur, sizeof(len));
	rb->decoder(&len, rb->cur, sizeof(len));
	rb->cur += sizeof(len);

	CORBA_free(ev->_repo_id);
	ev->_repo_id = rb->cur;
	rb->cur += len;

	if(rb->message.u.reply.reply_status == CORBA_SYSTEM_EXCEPTION) {
		rb->cur = ALIGN_ADDRESS(rb->cur, sizeof(minor));
		rb->decoder(&minor, rb->cur, sizeof(len));
		rb->cur += sizeof(len);

		for(minor = 1; exception_table[minor].repo_id; minor++) {
			if(!strcmp(exception_table[minor].repo_id, ev->_repo_id))
				break;
		}

		rb->cur = ALIGN_ADDRESS(rb->cur, sizeof(completion_status));
		rb->decoder(&completion_status, rb->cur, sizeof(completion_status));
		rb->cur += sizeof(completion_status);

		new=ORBit_alloc(sizeof(CORBA_SystemException), NULL, NULL);

		if(new!=NULL) {
			new->minor=minor;
			new->completed=completion_status;
			
			/* XXX what should the repo ID be? */
			CORBA_exception_set(ev, CORBA_SYSTEM_EXCEPTION,
					    (gchar *)exception_table[new->minor].repo_id,
					    new);
		}
	} else
		g_warning("User exceptions NYI\n");
}
