/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  ORBit: A CORBA v2.2 ORB
 *
 *  Copyright (C) 1998 Richard H. Porter and Red Hat Software
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Dick Porter <dick@cymru.net>
 *
 */

/*
 * This file is a repository for random functions that don't fit anywhere
 * else, and for ORBit-specific stuff.
 */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <assert.h>
#include <math.h>

#include "orbit.h"

typedef struct ORBitClassInfo ORBitClassInfo;

typedef void (*ORBitObjectInitFunc)(CORBA_Object _handle_to_be, gpointer class_data);

struct ORBitClassInfo {
  char *name;
  gulong id;
  gpointer method_stubs, method_skels;
  ORBitObjectInitFunc class_vtable_init_func;
  ORBitClassInfo **parent_classes;
};

GHashTable *orbit_class_list = NULL, *orbit_class_byid;
glong class_id_counter = -1;

void CORBA_any_set_release(CORBA_any *any, CORBA_boolean flag)
{
	g_assert(any!=NULL);

	if(flag==CORBA_TRUE) {
		any->_flags |= CORBA_ANYFLAGS_RELEASE;
	} else {
		any->_flags &= ~CORBA_ANYFLAGS_RELEASE;
	}

}

CORBA_boolean CORBA_any_get_release(CORBA_any *any)
{
	g_assert(any!=NULL);

	if(any->_flags & CORBA_ANYFLAGS_RELEASE)
		return(CORBA_TRUE);
	else
		return(CORBA_FALSE);
}

void CORBA_sequence_set_release(void *seq, CORBA_boolean flag)
{
	struct CORBA_Sequence_type *sequence;

	g_assert(seq!=NULL);

	sequence=(struct CORBA_Sequence_type *)seq;

	if(flag==CORBA_TRUE) {
		sequence->_flags |= CORBA_ANYFLAGS_RELEASE;
	} else {
		sequence->_flags &= ~CORBA_ANYFLAGS_RELEASE;
	}
}

CORBA_boolean CORBA_sequence_get_release(void *seq)
{
	struct CORBA_Sequence_type *sequence;

	g_assert(seq!=NULL);

	sequence=(struct CORBA_Sequence_type *)seq;

	if(sequence->_flags & CORBA_ANYFLAGS_RELEASE)
		return(CORBA_TRUE);
	else
		return(CORBA_FALSE);
}

/*
 * As far as I understand, values returned by CORBA_*_alloc() are supposed to be
 * freeable by CORBA_free(), so we can't use memory chunks here in any reasonable
 * fashion.
 */

CORBA_any *CORBA_any_alloc(void)
{
	return ORBit_alloc(sizeof(CORBA_any), NULL, NULL);
}

/*
 * Compares the typecodes of each any
 */
CORBA_boolean ORBit_any_equivalent(CORBA_any obj, CORBA_any any, CORBA_Environment *ev)
{
	return(CORBA_FALSE);
}

CORBA_char *CORBA_string_dup(CORBA_char *string)
{
	CORBA_char *retval;

	if(!string)
		return NULL;

	retval = ORBit_alloc(strlen(string)+1, NULL, NULL);

	strcpy(retval, string);

	return retval;
}

CORBA_char *CORBA_string_alloc(CORBA_unsigned_long len)
{
	if(len)
		return ORBit_alloc(len + 1, NULL, NULL);
	else
		return NULL;
}

CORBA_wchar *CORBA_wstring_alloc(CORBA_unsigned_long len)
{
	if(len)
		return ORBit_alloc(len + 1, NULL, NULL);
	else
		return NULL;
}

/* 19.14 */

/* The big picture for fixeds.
   We have to represent a number in memory.

   1 2 3 . 4 5 6 7

   There are three pieces of information in a fixed:

   - Number of significant digits. (_digits)

   - The scale. The number of places the decimal point is to the right
   of the first significant digit. (_scale)

   - The digits themselves (_value)

 */
CORBA_long CORBA_fixed_integer_part(const void *fp)
{
	CORBA_long retval = 0;
	int i, power_of_ten, digit;
	const CORBA_fixed_d_s *val = fp;

	g_return_val_if_fail(fp != NULL, INT_MIN);

	for(i = 0; i < (val->_digits - val->_scale); i++) {
		power_of_ten = val->_digits - i - val->_scale - 1;
		digit = val->_value[i];
		retval += digit * ((int)pow(10, power_of_ten));
	}

	return retval;
}

CORBA_long CORBA_fixed_fraction_part(const void *fp)
{
	CORBA_long retval = 0;
	int i, power_of_ten, digit;
	const CORBA_fixed_d_s *val = fp;

	g_return_val_if_fail(fp != NULL, INT_MIN);

	for(i = val->_digits - val->_scale; i < val->_digits; i++){
		power_of_ten = val->_digits - i - 1;
		digit = val->_value[i];
		retval += digit * ((int)pow(10, power_of_ten));
	}

	return retval;
}

#define do_div(n) ({ \
int __res; \
__res = ((CORBA_long) n) % (unsigned) 10; \
n = ((CORBA_long) n) / (unsigned) 10; \
__res; })

void CORBA_fixed_set(void *rp, CORBA_long i, CORBA_long f)
{
	CORBA_fixed_d_s *val = rp;
	CORBA_long left_to_eat, cur;
	signed char sign = 1;

	g_return_if_fail(rp != NULL);

	memset(val->_value, 0, val->_digits);

	if(i) sign = i/abs(i);
	val->_sign = sign;
	i = abs(i);
	f = abs(f);

	for(cur = 0, left_to_eat = i;
	    left_to_eat != 0 && cur < val->_digits; cur++) {
		val->_value[cur] = do_div(left_to_eat) * sign;
		sign = 1;
	}

	val->_scale = cur - 1;

	for(left_to_eat = f;
	    left_to_eat != 0 && cur < val->_digits; cur++) {
		val->_value[cur] = do_div(left_to_eat);
	}
}

void CORBA_fixed_add(void *rp, const void *f1p, const void *f2p)
{
	g_assert(!"Not yet implemented");
}

void CORBA_fixed_sub(void *rp, const void *f1p, const void *f2p)
{
	g_assert(!"Not yet implemented");
}

void CORBA_fixed_mul(void *rp, const void *f1p, const void *f2p)
{
	g_assert(!"Not yet implemented");
}

void CORBA_fixed_div(void *rp, const void *f1p, const void *f2p)
{
	g_assert(!"Not yet implemented");
}

CORBA_fixed_d_s *CORBA_fixed_alloc(CORBA_unsigned_short d)
{
	return (CORBA_fixed_d_s *)
		g_malloc(sizeof(CORBA_fixed_d_s) + d + 1);
}

void CORBA_free(void *storage)
{
	ORBit_free(storage);
}

int ORBit_parse_inet(CORBA_Object obj, char *hostname, unsigned short port)
{
	obj->connection = GIOP_CONNECTION(iiop_connection_get(hostname, port));

	if(obj->connection) {
		giop_connection_ref(obj->connection);
		return 0;
	} else
		return -1;
}

/**************************** UNUSED ******************************/
#if 0
ORBitClassInfo **
gen_parent_class_list(glong *parent_class_ids)
{
	int i, n;
	ORBitClassInfo **retval;

	for(i = 0; parent_class_ids[i] >= 0; i++)
		/* */ ;
	n = i;

	retval = g_new(ORBitClassInfo *, n + 1);

	for(i = 0; i < n; i++)
		retval[i] = g_hash_table_lookup(orbit_class_byid,
						(gpointer)parent_class_ids[i]);
	retval[i] = NULL;

	return retval;
}

glong
ORBit_register_class(char *name,
		     gpointer method_stubs,
		     gpointer method_skels,
		     gpointer class_vtable_init_func,
		     glong *parent_class_ids)
{
	ORBitClassInfo *newclass, **pclass;

	if(!orbit_class_list) {
		orbit_class_list = g_hash_table_new(g_str_hash, g_str_equal);
		orbit_class_byid = g_hash_table_new(g_direct_hash, NULL);
	}

	newclass = g_hash_table_lookup(orbit_class_list, name);

	if(!newclass) {
		if(parent_class_ids) {
			pclass = gen_parent_class_list(parent_class_ids);
			g_return_val_if_fail(pclass != NULL, -1);
		} else
			pclass = NULL;

		newclass = g_new(ORBitClassInfo, 1);
      
		newclass->name = g_strdup(name);
		newclass->id = class_id_counter++;
		newclass->method_stubs = method_stubs;
		newclass->method_skels = method_skels;
		newclass->class_vtable_init_func = class_vtable_init_func;
		newclass->parent_classes = pclass;
      
		g_hash_table_insert(orbit_class_list,
				    newclass->name, newclass);
		g_hash_table_insert(orbit_class_byid,
				    (gpointer)newclass->id, newclass);
	}

	return newclass->id;
}

static void
init_object_and_recurse(CORBA_Object o,
			ORBitClassInfo *class,
			gboolean is_impl)
{
	int i;
	gpointer *ptmp;

	if(is_impl)
		ptmp = class->method_skels;
	else
		ptmp = class->method_stubs;

	for(i = 0; ptmp[i]; i++) /**/ ;

	memcpy(o->vtable[class->id], ptmp, sizeof(gpointer) * i);

	if(class->parent_classes) {
		for(i = 0; class->parent_classes[i]; i++)
			init_object_and_recurse(o, class->parent_classes[i],
						is_impl);
	}
}

void
ORBit_initialize_object_vtable(CORBA_Object _handle_to_be,
			       glong class_id,
			       gboolean is_impl)
{
	ORBitClassInfo *c;

	g_return_if_fail(class_id >= 0);
	g_return_if_fail(_handle_to_be != NULL);

	c = g_hash_table_lookup(orbit_class_byid, (gpointer) class_id);

	g_return_if_fail(c != NULL);

	_handle_to_be->vtable = g_new0(gpointer, class_id_counter);

	init_object_and_recurse(_handle_to_be, c, is_impl);
}
#endif
/**************************** END UNUSED ******************************/
