/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtkhtml;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.EventType;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.gtkhtml.dom.DomMouseEvent;
import org.gnu.gtkhtml.dom.DomMouseListener;
import org.gnu.gtkhtml.event.HTMLDocumentEvent;
import org.gnu.gtkhtml.event.HTMLDocumentListener;

public class HTMLDocument extends GObject {

    /**
     * Listeners for handling document events.
     */
    private Vector documentListeners = null;
    /**
     * Listeners for handling dom events.
     */
    private Vector domListeners = null;

    /**
     * Map to collect the events.
     */
    private static EventMap evtMap = new EventMap();

    static {
        addEvents(evtMap);
    }

    public HTMLDocument() {
        super(html_document_new());
    }
	
    public boolean openStream(String mimeType) {
        return html_document_open_stream(getHandle(), mimeType);
    }
	
    public void writeStream(String buffer) {
        html_document_write_stream(getHandle(), buffer, buffer.length());
    }
	
    public void closeStream() {
        html_document_close_stream(getHandle());
    }
	
    public void clear() {
        html_document_clear(getHandle());
    }

    /**
     * Load the HTML document in the given InputStream.
     * @param stream The InputStream containing the HTML document.
     * @param charset The character encoding of the stream.
     * @throws UnsupportedEncodingException if the given charset is 
     * not valid.
     * @throws IOException if an error occured reading data from the 
     * stream.
     */
    public void loadStream( InputStream stream, String charset ) 
        throws UnsupportedEncodingException, IOException {

        openStream( "text/html" );
        try {
            BufferedInputStream in = 
                new BufferedInputStream( stream );
            byte[] buffer = new byte[4096];
            int bytesRead;
            while ( (bytesRead = in.read( buffer )) != -1 ) {
                String data = null;
                if ( charset == null ) {
                    // Use the platform's default encoding.
                    data = new String( buffer );
                } else {
                    data = new String( buffer, charset );
                }
                writeStream( data.trim() );
            }
        } finally {
            closeStream();
        }
    }

    /**
     * Load a file that is local to the process
     * @param filename The name of the file to load
     * @throws FileNotFoundException if the file doesn't exist
     * @throws IOException if the file cannot be read
     */
    public void loadFile(String filename) throws FileNotFoundException, IOException {
        File aFile = new File(filename);
        clear();
        openStream("text/html");
        try {
            BufferedReader in = new BufferedReader(new FileReader(aFile));
            String line;
            while ((line = in.readLine()) != null)
                if (line.length() > 0)
                    writeStream(line);
        } finally {
            closeStream();
        }
    }
	
    /**
     * Load a url that is on a remote web server.
     * @param page The url of the remote page to display
     * @throws MalformedURLException
     * @throws IOException
     */
    public void loadURL(String page) throws MalformedURLException, IOException {
        URL url = new URL(page);
        clear();
        try {
            loadStream( url.openStream(), null );
        } catch( UnsupportedEncodingException ex ) {}
    }

    /**
     * Register an object to handle document events.
     * @see org.gnu.gtkhtml.event.HTMLDocumentListener
     */
    public void addListener(HTMLDocumentListener listener) {
        // Don't add the listener a second time if it is in the Vector.
        int i = findListener(documentListeners, listener);
        if (i == -1) {
            if (null == documentListeners) {
                evtMap.initialize(this, HTMLDocumentEvent.Type.REQUEST_URL);
                evtMap.initialize(this, HTMLDocumentEvent.Type.LINK_CLICKED);
                evtMap.initialize(this, HTMLDocumentEvent.Type.SET_BASE);
                evtMap.initialize(this, HTMLDocumentEvent.Type.TITLE_CHANGED);
                evtMap.initialize(this, HTMLDocumentEvent.Type.SUBMIT);

                documentListeners = new Vector();
            }
            documentListeners.addElement(listener);
        }
    }

    /**
     * Register an object to handle dom events.
     * @see org.gnu.gtkhtml.event.DomMouseListener
     */
    public void addListener( DomMouseListener listener ) {
        int i = findListener(domListeners, listener);
        if (i == -1) {
            if (null == domListeners) {
                evtMap.initialize(this, DomMouseEvent.Type.DOM_MOUSE_DOWN);
                evtMap.initialize(this, DomMouseEvent.Type.DOM_MOUSE_UP);
                evtMap.initialize(this, DomMouseEvent.Type.DOM_MOUSE_CLICK);
                evtMap.initialize(this, DomMouseEvent.Type.DOM_MOUSE_OVER);
                evtMap.initialize(this, DomMouseEvent.Type.DOM_MOUSE_OUT);

                domListeners = new Vector();
            }
            domListeners.addElement( listener );
        }
    }

    /**
     * Removes a listener
     * @see #addListener(HTMLDocumentListener)
     */
    public void removeListener(HTMLDocumentListener listener) {
        int i = findListener(documentListeners, listener);
        if (i > -1) {
            documentListeners.remove(i);
        }
        if (0 == documentListeners.size()) {
            evtMap.uninitialize(this, HTMLDocumentEvent.Type.REQUEST_URL);
            evtMap.uninitialize(this, HTMLDocumentEvent.Type.LINK_CLICKED);
            evtMap.uninitialize(this, HTMLDocumentEvent.Type.SET_BASE);
            evtMap.uninitialize(this, HTMLDocumentEvent.Type.TITLE_CHANGED);
            evtMap.uninitialize(this, HTMLDocumentEvent.Type.SUBMIT);
            documentListeners = null;
        }
    }

    /** 
     * Removes a DomMouseListener.
     * @see #addListener( DomMouseListener )
     */
    public void removeListener( DomMouseListener listener ) {
        int i = findListener(domListeners, listener);
        if (i > -1) {
            domListeners.remove(i);
        }
        if (0 == domListeners.size()) {
            evtMap.uninitialize(this, DomMouseEvent.Type.DOM_MOUSE_DOWN);
            evtMap.uninitialize(this, DomMouseEvent.Type.DOM_MOUSE_UP);
            evtMap.uninitialize(this, DomMouseEvent.Type.DOM_MOUSE_CLICK);
            evtMap.uninitialize(this, DomMouseEvent.Type.DOM_MOUSE_OVER);
            evtMap.uninitialize(this, DomMouseEvent.Type.DOM_MOUSE_OUT);
            domListeners = null;
        }
    }


    /**
     * Give us a way to locate a specific listener in a Vector.
     * @param list The Vector of listeners to search.
     * @param listener The object that is to be located in the Vector.
     * @return Returns the index of the listener in the Vector, or -1 if
     *                 the listener is not contained in the Vector.
     */
    protected static int findListener(Vector list, Object listener) {
        if (null == list || null == listener)
            return -1;
        return list.indexOf(listener);
    }

    protected void fireDocumentEvent(HTMLDocumentEvent event) {
        if (null == documentListeners) {
            return;
        }
        int size = documentListeners.size();
        int i = 0;
        while (i < size) {
            HTMLDocumentListener dl = (HTMLDocumentListener)documentListeners.elementAt(i);
            dl.documentEvent(event);
            i++;
        }
    }

    protected void fireDomMouseEvent( DomMouseEvent event ) {
        if ( domListeners == null ) {
            return;
        }
        int size = domListeners.size();
        int i = 0;
        while (i < size) {
            DomMouseListener dl = (DomMouseListener)domListeners.elementAt(i);
            dl.domMouseEvent(event);
            i++;
        }
    }

    public Class getEventListenerClass(String signal) {
        Class cls = evtMap.getEventListenerClass(signal);
        if (cls == null) cls = super.getEventListenerClass(signal);
        return cls;
    }

    public EventType getEventType(String signal) {
        EventType et = evtMap.getEventType(signal);
        if (et == null) et = super.getEventType(signal);
        return et;
    }

    /**
     * Implementation method to build an EventMap for this widget class.
     * Not useful (or supported) for application use.
     */
    private static void addEvents(EventMap anEvtMap) {
        // HTMLDocument events.
        anEvtMap.addEvent("request_url", "handleRequestURL", 
                          HTMLDocumentEvent.Type.REQUEST_URL, 
                          HTMLDocumentListener.class);
        anEvtMap.addEvent("link_clicked", "handleLinkClicked", 
                          HTMLDocumentEvent.Type.LINK_CLICKED, 
                          HTMLDocumentListener.class);
        anEvtMap.addEvent("set_base", "handleSetBase", 
                          HTMLDocumentEvent.Type.SET_BASE, 
                          HTMLDocumentListener.class);
        anEvtMap.addEvent("title_changed", "handleTitleChanged", 
                          HTMLDocumentEvent.Type.TITLE_CHANGED, 
                          HTMLDocumentListener.class);
        anEvtMap.addEvent("submit", "handleSubmit", 
                          HTMLDocumentEvent.Type.SUBMIT, 
                          HTMLDocumentListener.class);
        // DomEvents
        anEvtMap.addEvent("dom_mouse_down", "handleDomMouseDown", 
                          DomMouseEvent.Type.DOM_MOUSE_OUT, 
                          DomMouseListener.class);
        anEvtMap.addEvent("dom_mouse_up", "handleDomMouseUp", 
                          DomMouseEvent.Type.DOM_MOUSE_OVER, 
                          DomMouseListener.class);
        anEvtMap.addEvent("dom_mouse_click", "handleDomMouseClick", 
                          DomMouseEvent.Type.DOM_MOUSE_OUT, 
                          DomMouseListener.class);
        anEvtMap.addEvent("dom_mouse_over", "handleDomMouseOver", 
                          DomMouseEvent.Type.DOM_MOUSE_OVER, 
                          DomMouseListener.class);
        anEvtMap.addEvent("dom_mouse_out", "handleDomMouseOut", 
                          DomMouseEvent.Type.DOM_MOUSE_OUT, 
                          DomMouseListener.class);
    }
	
    protected void handleRequestURL(String url, Handle stream) {
        HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.REQUEST_URL);
        anEvent.setURL(url);
        anEvent.setHtmlStream(new HTMLStream(stream));
        fireDocumentEvent(anEvent);
    }

    protected void handleLinkClicked(String url) {
        HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.LINK_CLICKED);
        anEvent.setURL(url);
        fireDocumentEvent(anEvent);
    }
	
    protected void handleSetBase(String base) {
        HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.SET_BASE);
        anEvent.setURL(base);
        fireDocumentEvent(anEvent);
    }

    protected void handleTitleChanged(String title) {
        HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.TITLE_CHANGED);
        anEvent.setURL(title);
        fireDocumentEvent(anEvent);
    }

    protected void handleSubmit(String url, String method, String encoding) {
        HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.SUBMIT);
        anEvent.setMethod(method);
        anEvent.setURL(url);
        anEvent.setEncoding(encoding);
        fireDocumentEvent(anEvent);
    }

    protected boolean handleDomMouseDown(DomMouseEvent event) {
        event.setType( DomMouseEvent.Type.DOM_MOUSE_DOWN );
        fireDomMouseEvent(event);
        return true;
    }
    protected boolean handleDomMouseUp(DomMouseEvent event) {
        event.setType( DomMouseEvent.Type.DOM_MOUSE_UP );
        fireDomMouseEvent(event);
        return true;
    }
    protected boolean handleDomMouseClick(DomMouseEvent event) {
        event.setType( DomMouseEvent.Type.DOM_MOUSE_CLICK );
        fireDomMouseEvent(event);
        return true;
    }
    protected boolean handleDomMouseOver(DomMouseEvent event) {
        event.setType( DomMouseEvent.Type.DOM_MOUSE_OVER );
        fireDomMouseEvent(event);
        return true;
    }
    protected boolean handleDomMouseOut(DomMouseEvent event) {
        event.setType( DomMouseEvent.Type.DOM_MOUSE_OUT );
        fireDomMouseEvent(event);
        return true;
    }

    native static final protected int html_document_get_type();
    native static final protected Handle html_document_new();
    native static final protected boolean html_document_open_stream(Handle document, String mimeType);
    native static final protected void html_document_write_stream(Handle document, String buffer, int len);
    native static final protected void html_document_close_stream(Handle document);
    native static final protected void html_document_clear(Handle document);
    native static final protected void html_document_update_hover_node(Handle document, Handle node);
    native static final protected void html_document_update_active_node(Handle document, Handle node);
    native static final protected void html_document_update_focus_element(Handle document, Handle element);
    native static final protected void html_document_find_anchor(Handle document, String anchor);

    static {
        System.loadLibrary("gtkhtmljni-2.6");
    }
}
